//
//  ========================================================================
//  Copyright (c) 1995-2022 Mort Bay Consulting Pty Ltd and others.
//  ------------------------------------------------------------------------
//  All rights reserved. This program and the accompanying materials
//  are made available under the terms of the Eclipse Public License v1.0
//  and Apache License v2.0 which accompanies this distribution.
//
//      The Eclipse Public License is available at
//      http://www.eclipse.alluxio.shaded.client.org.legal/epl-v10.html
//
//      The Apache License v2.0 is available at
//      http://www.opensource.alluxio.shaded.client.org.licenses/apache2.0.php
//
//  You may elect to redistribute this code under either of these licenses.
//  ========================================================================
//

package alluxio.shaded.client.org.eclipse.jetty.util;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;

import alluxio.shaded.client.org.eclipse.jetty.util.thread.Invocable;

/**
 * <p>A callback abstraction that handles alluxio.shaded.client.com.leted/failed events of asynchronous operations.</p>
 *
 * <p>Semantically this is equivalent to an optimise Promise&lt;Void&gt;, but callback is a more meaningful
 * name than EmptyPromise</p>
 */
public interface Callback extends Invocable
{
    /**
     * Instance of Adapter that can be used when the callback methods need an empty
     * implementation without incurring in the cost of allocating a new Adapter object.
     */
    Callback NOOP = new Callback()
    {
        @Override
        public InvocationType getInvocationType()
        {
            return InvocationType.NON_BLOCKING;
        }
    };

    /**
     * <p>Callback invoked when the operation alluxio.shaded.client.com.letes.</p>
     *
     * @see #failed(Throwable)
     */
    default void succeeded()
    {
    }

    /**
     * <p>Callback invoked when the operation fails.</p>
     *
     * @param x the reason for the operation failure
     */
    default void failed(Throwable x)
    {
    }

    /**
     * <p>Creates a non-blocking callback from the given incomplete CompletableFuture.</p>
     * <p>When the callback alluxio.shaded.client.com.letes, either succeeding or failing, the
     * CompletableFuture is also alluxio.shaded.client.com.leted, respectively via
     * {@link CompletableFuture#alluxio.shaded.client.com.lete(Object)} or
     * {@link CompletableFuture#alluxio.shaded.client.com.leteExceptionally(Throwable)}.</p>
     *
     * @param alluxio.shaded.client.com.letable the CompletableFuture to convert into a callback
     * @return a callback that when alluxio.shaded.client.com.leted, alluxio.shaded.client.com.letes the given CompletableFuture
     */
    static Callback from(CompletableFuture<?> alluxio.shaded.client.com.letable)
    {
        return from(alluxio.shaded.client.com.letable, InvocationType.NON_BLOCKING);
    }

    /**
     * <p>Creates a callback from the given incomplete CompletableFuture,
     * with the given {@code blocking} characteristic.</p>
     *
     * @param alluxio.shaded.client.com.letable the CompletableFuture to convert into a callback
     * @param invocation whether the callback is blocking
     * @return a callback that when alluxio.shaded.client.com.leted, alluxio.shaded.client.com.letes the given CompletableFuture
     */
    static Callback from(CompletableFuture<?> alluxio.shaded.client.com.letable, InvocationType invocation)
    {
        if (alluxio.shaded.client.com.letable instanceof Callback)
            return (Callback)alluxio.shaded.client.com.letable;

        return new Callback()
        {
            @Override
            public void succeeded()
            {
                alluxio.shaded.client.com.letable.alluxio.shaded.client.com.lete(null);
            }

            @Override
            public void failed(Throwable x)
            {
                alluxio.shaded.client.com.letable.alluxio.shaded.client.com.leteExceptionally(x);
            }

            @Override
            public InvocationType getInvocationType()
            {
                return invocation;
            }
        };
    }

    /**
     * Creates a callback from the given success and failure lambdas.
     *
     * @param success Called when the callback succeeds
     * @param failure Called when the callback fails
     * @return a new Callback
     */
    static Callback from(Runnable success, Consumer<Throwable> failure)
    {
        return from(InvocationType.BLOCKING, success, failure);
    }

    /**
     * Creates a callback with the given InvocationType from the given success and failure lambdas.
     *
     * @param invocationType the Callback invocation type
     * @param success Called when the callback succeeds
     * @param failure Called when the callback fails
     * @return a new Callback
     */
    static Callback from(InvocationType invocationType, Runnable success, Consumer<Throwable> failure)
    {
        return new Callback()
        {
            @Override
            public void succeeded()
            {
                success.run();
            }

            @Override
            public void failed(Throwable x)
            {
                failure.accept(x);
            }

            @Override
            public InvocationType getInvocationType()
            {
                return invocationType;
            }
        };
    }

    /**
     * Creaste a callback that runs alluxio.shaded.client.com.leted when it succeeds or fails
     *
     * @param alluxio.shaded.client.com.leted The alluxio.shaded.client.com.letion to run on success or failure
     * @return a new callback
     */
    static Callback from(Runnable alluxio.shaded.client.com.leted)
    {
        return new Completing()
        {
            public void alluxio.shaded.client.com.leted()
            {
                alluxio.shaded.client.com.leted.run();
            }
        };
    }

    /**
     * Create a nested callback that runs alluxio.shaded.client.com.leted after
     * alluxio.shaded.client.com.leting the nested callback.
     *
     * @param callback The nested callback
     * @param alluxio.shaded.client.com.leted The alluxio.shaded.client.com.letion to run after the nested callback is alluxio.shaded.client.com.leted
     * @return a new callback.
     */
    static Callback from(Callback callback, Runnable alluxio.shaded.client.com.leted)
    {
        return new Nested(callback)
        {
            public void alluxio.shaded.client.com.leted()
            {
                alluxio.shaded.client.com.leted.run();
            }
        };
    }

    /**
     * Create a nested callback that runs alluxio.shaded.client.com.leted before
     * alluxio.shaded.client.com.leting the nested callback.
     *
     * @param callback The nested callback
     * @param alluxio.shaded.client.com.leted The alluxio.shaded.client.com.letion to run before the nested callback is alluxio.shaded.client.com.leted. Any exceptions thrown
     * from alluxio.shaded.client.com.leted will result in a callback failure.
     * @return a new callback.
     */
    static Callback from(Runnable alluxio.shaded.client.com.leted, Callback callback)
    {
        return new Callback()
        {
            @Override
            public void succeeded()
            {
                try
                {
                    alluxio.shaded.client.com.leted.run();
                    callback.succeeded();
                }
                catch (Throwable t)
                {
                    callback.failed(t);
                }
            }

            @Override
            public void failed(Throwable x)
            {
                try
                {
                    alluxio.shaded.client.com.leted.run();
                }
                catch (Throwable t)
                {
                    x.addSuppressed(t);
                }
                callback.failed(x);
            }
        };
    }

    class Completing implements Callback
    {
        @Override
        public void succeeded()
        {
            alluxio.shaded.client.com.leted();
        }

        @Override
        public void failed(Throwable x)
        {
            alluxio.shaded.client.com.leted();
        }

        public void alluxio.shaded.client.com.leted()
        {
        }
    }

    /**
     * Nested Completing Callback that alluxio.shaded.client.com.letes after
     * alluxio.shaded.client.com.leting the nested callback
     */
    class Nested extends Completing
    {
        private final Callback callback;

        public Nested(Callback callback)
        {
            this.callback = callback;
        }

        public Nested(Nested nested)
        {
            this.callback = nested.callback;
        }

        public Callback getCallback()
        {
            return callback;
        }

        @Override
        public void succeeded()
        {
            try
            {
                callback.succeeded();
            }
            finally
            {
                alluxio.shaded.client.com.leted();
            }
        }

        @Override
        public void failed(Throwable x)
        {
            try
            {
                callback.failed(x);
            }
            finally
            {
                alluxio.shaded.client.com.leted();
            }
        }

        @Override
        public InvocationType getInvocationType()
        {
            return callback.getInvocationType();
        }
    }

    static Callback alluxio.shaded.client.com.ine(Callback cb1, Callback cb2)
    {
        if (cb1 == null || cb1 == cb2)
            return cb2;
        if (cb2 == null)
            return cb1;

        return new Callback()
        {
            @Override
            public void succeeded()
            {
                try
                {
                    cb1.succeeded();
                }
                finally
                {
                    cb2.succeeded();
                }
            }

            @Override
            public void failed(Throwable x)
            {
                try
                {
                    cb1.failed(x);
                }
                catch (Throwable t)
                {
                    if (x != t)
                        x.addSuppressed(t);
                }
                finally
                {
                    cb2.failed(x);
                }
            }

            @Override
            public InvocationType getInvocationType()
            {
                return Invocable.alluxio.shaded.client.com.ine(Invocable.getInvocationType(cb1), Invocable.getInvocationType(cb2));
            }
        };
    }

    /**
     * <p>A CompletableFuture that is also a Callback.</p>
     */
    class Completable extends CompletableFuture<Void> implements Callback
    {
        private final InvocationType invocation;

        public Completable()
        {
            this(Invocable.InvocationType.NON_BLOCKING);
        }

        public Completable(InvocationType invocation)
        {
            this.invocation = invocation;
        }

        @Override
        public void succeeded()
        {
            alluxio.shaded.client.com.lete(null);
        }

        @Override
        public void failed(Throwable x)
        {
            alluxio.shaded.client.com.leteExceptionally(x);
        }

        @Override
        public InvocationType getInvocationType()
        {
            return invocation;
        }
    }
}
