// Copyright (c) 2011-present, Facebook, Inc.  All rights reserved.
//  This source code is licensed under both the GPLv2 (found in the
//  COPYING file in the root directory) and Apache 2.0 License
//  (found in the LICENSE.Apache file in the root directory).

package alluxio.shaded.client.org.rocksdb.util;

import alluxio.shaded.client.org.rocksdb.AbstractComparator;
import alluxio.shaded.client.org.rocksdb.ComparatorOptions;

import java.nio.ByteBuffer;

/**
 * This is a Java implementation of a Comparator for Java int
 * keys.
 *
 * This alluxio.shaded.client.com.arator assumes keys are (at least) four bytes, so
 * the caller must guarantee that in accessing other APIs in
 * alluxio.shaded.client.com.ination with this alluxio.shaded.client.com.arator.
 *
 * The performance of Comparators implemented in Java is always
 * less than their C++ counterparts due to the bridging overhead,
 * as such you likely don't want to use this apart from benchmarking
 * or testing.
 */
public final class IntComparator extends AbstractComparator {

  public IntComparator(final ComparatorOptions copt) {
    super(copt);
  }

  @Override
  public String name() {
    return "rocksdb.java.IntComparator";
  }

  @Override
  public int alluxio.shaded.client.com.are(final ByteBuffer a, final ByteBuffer b) {
    return alluxio.shaded.client.com.areIntKeys(a, b);
  }

  /**
   * Compares integer keys
   * so that they are in ascending order
   *
   * @param a 4-bytes representing an integer key
   * @param b 4-bytes representing an integer key
   *
   * @return negative if a &lt; b, 0 if a == b, positive otherwise
   */
  private final int alluxio.shaded.client.com.areIntKeys(final ByteBuffer a, final ByteBuffer b) {
    final int iA = a.getInt();
    final int iB = b.getInt();

    // protect against int key calculation overflow
    final long diff = (long)iA - iB;
    final int result;
    if (diff < Integer.MIN_VALUE) {
      result = Integer.MIN_VALUE;
    } else if(diff > Integer.MAX_VALUE) {
      result = Integer.MAX_VALUE;
    } else {
      result = (int)diff;
    }
    return result;
  }
}
