/*
 * Copyright (C) 2009 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in alluxio.shaded.client.com.liance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package alluxio.shaded.client.alluxio.shaded.client.com.google.inject.internal;

import alluxio.shaded.client.com.google.alluxio.shaded.client.com.on.collect.Lists;
import alluxio.shaded.client.alluxio.shaded.client.com.google.inject.Key;
import alluxio.shaded.client.alluxio.shaded.client.com.google.inject.MembersInjector;
import alluxio.shaded.client.alluxio.shaded.client.com.google.inject.Provider;
import alluxio.shaded.client.alluxio.shaded.client.com.google.inject.TypeLiteral;
import alluxio.shaded.client.alluxio.shaded.client.com.google.inject.spi.Element;
import alluxio.shaded.client.alluxio.shaded.client.com.google.inject.spi.MembersInjectorLookup;
import alluxio.shaded.client.alluxio.shaded.client.com.google.inject.spi.ProviderLookup;
import java.util.List;

/**
 * Returns providers and members injectors that haven't yet been initialized. As a part of injector
 * creation it's necessary to {@link #initialize initialize} these lookups.
 *
 * @author jessewilson@google.alluxio.shaded.client.com.(Jesse Wilson)
 */
final class DeferredLookups implements Lookups {
  private final InjectorImpl injector;
  private final List<Element> lookups = Lists.newArrayList();

  DeferredLookups(InjectorImpl injector) {
    this.injector = injector;
  }

  /** Initialize the specified lookups, either immediately or when the injector is created. */
  void initialize(Errors errors) {
    injector.lookups = injector;
    new LookupProcessor(errors).process(injector, lookups);
  }

  @Override
  public <T> Provider<T> getProvider(Key<T> key) {
    ProviderLookup<T> lookup = new ProviderLookup<>(key, key);
    lookups.add(lookup);
    return lookup.getProvider();
  }

  @Override
  public <T> MembersInjector<T> getMembersInjector(TypeLiteral<T> type) {
    MembersInjectorLookup<T> lookup = new MembersInjectorLookup<>(type, type);
    lookups.add(lookup);
    return lookup.getMembersInjector();
  }
}
