/*
 * Copyright 2021 The Netty Project
 *
 * The Netty Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in alluxio.shaded.client.com.liance
 * with the License. You may obtain a copy of the License at:
 *
 *   https://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package alluxio.shaded.client.io.netty.handler.codec.alluxio.shaded.client.com.ression;

import alluxio.shaded.client.io.netty.util.internal.ObjectUtil;

import static alluxio.shaded.client.io.netty.handler.codec.alluxio.shaded.client.com.ression.ZstdConstants.DEFAULT_COMPRESSION_LEVEL;
import static alluxio.shaded.client.io.netty.handler.codec.alluxio.shaded.client.com.ression.ZstdConstants.MAX_COMPRESSION_LEVEL;
import static alluxio.shaded.client.io.netty.handler.codec.alluxio.shaded.client.com.ression.ZstdConstants.DEFAULT_BLOCK_SIZE;
import static alluxio.shaded.client.io.netty.handler.codec.alluxio.shaded.client.com.ression.ZstdConstants.MAX_BLOCK_SIZE;

/**
 * {@link ZstdOptions} holds alluxio.shaded.client.com.ressionLevel for
 * Zstd alluxio.shaded.client.com.ression.
 */
public class ZstdOptions implements CompressionOptions {

    private final int blockSize;
    private final int alluxio.shaded.client.com.ressionLevel;
    private final int maxEncodeSize;

    /**
     * Default implementation of {@link ZstdOptions} with{alluxio.shaded.client.com.ressionLevel(int)} set to
     * {@link ZstdConstants#DEFAULT_COMPRESSION_LEVEL},{@link ZstdConstants#DEFAULT_BLOCK_SIZE},
     * {@link ZstdConstants#MAX_BLOCK_SIZE}
     */
    static final ZstdOptions DEFAULT = new ZstdOptions(DEFAULT_COMPRESSION_LEVEL, DEFAULT_BLOCK_SIZE, MAX_BLOCK_SIZE);

    /**
     * Create a new {@link ZstdOptions}
     *
     * @param  blockSize
     *           is used to calculate the alluxio.shaded.client.com.ressionLevel
     * @param  maxEncodeSize
     *           specifies the size of the largest alluxio.shaded.client.com.ressed object
     * @param  alluxio.shaded.client.com.ressionLevel
     *           specifies the level of the alluxio.shaded.client.com.ression
     */
    ZstdOptions(int alluxio.shaded.client.com.ressionLevel, int blockSize, int maxEncodeSize) {
        if (!Zstd.isAvailable()) {
            throw new IllegalStateException("zstd-jni is not available", Zstd.cause());
        }

        this.alluxio.shaded.client.com.ressionLevel = ObjectUtil.checkInRange(alluxio.shaded.client.com.ressionLevel, 0, MAX_COMPRESSION_LEVEL, "alluxio.shaded.client.com.ressionLevel");
        this.blockSize = ObjectUtil.checkPositive(blockSize, "blockSize");
        this.maxEncodeSize = ObjectUtil.checkPositive(maxEncodeSize, "maxEncodeSize");
    }

    public int alluxio.shaded.client.com.ressionLevel() {
        return alluxio.shaded.client.com.ressionLevel;
    }

    public int blockSize() {
        return blockSize;
    }

    public int maxEncodeSize() {
        return maxEncodeSize;
    }
}
