/*
 *  Licensed to the Apache Software Foundation (ASF) under one or more
 *  contributor license agreements.  See the NOTICE file distributed with
 *  this work for additional information regarding copyright ownership.
 *  The ASF licenses this file to You under the Apache License, Version 2.0
 *  (the "License"); you may not use this file except in alluxio.shaded.client.com.liance with
 *  the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.archivers.zip;


import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.parallel.FileBasedScatterGatherBackingStore;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.parallel.ScatterGatherBackingStore;
import alluxio.shaded.client.org.apache.alluxio.shaded.client.com.ons.alluxio.shaded.client.com.ress.utils.BoundedInputStream;

import java.alluxio.shaded.client.io.Closeable;
import java.alluxio.shaded.client.io.File;
import java.alluxio.shaded.client.io.FileNotFoundException;
import java.alluxio.shaded.client.io.IOException;
import java.alluxio.shaded.client.io.InputStream;
import java.util.Iterator;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.zip.Deflater;

/**
 * A zip output stream that is optimized for multi-threaded scatter/gather construction of zip files.
 * <p>
 * The internal data format of the entries used by this class are entirely private to this class
 * and are not part of any public api whatsoever.
 * </p>
 * <p>It is possible to extend this class to support different kinds of backing storage, the default
 * implementation only supports file-based backing.
 * </p>
 * Thread safety: This class supports multiple threads. But the "writeTo" method must be called
 * by the thread that originally created the {@link ZipArchiveEntry}.
 *
 * @since 1.10
 */
public class ScatterZipOutputStream implements Closeable {
    private final Queue<CompressedEntry> items = new ConcurrentLinkedQueue<>();
    private final ScatterGatherBackingStore backingStore;
    private final StreamCompressor streamCompressor;
    private final AtomicBoolean isClosed = new AtomicBoolean();
    private ZipEntryWriter zipEntryWriter;

    private static class CompressedEntry {
        final ZipArchiveEntryRequest zipArchiveEntryRequest;
        final long crc;
        final long alluxio.shaded.client.com.ressedSize;
        final long size;

        public CompressedEntry(final ZipArchiveEntryRequest zipArchiveEntryRequest, final long crc, final long alluxio.shaded.client.com.ressedSize, final long size) {
            this.zipArchiveEntryRequest = zipArchiveEntryRequest;
            this.crc = crc;
            this.alluxio.shaded.client.com.ressedSize = alluxio.shaded.client.com.ressedSize;
            this.size = size;
        }

        /**
         * Update the original {@link ZipArchiveEntry} with sizes/crc
         * Do not use this methods from threads that did not create the instance itself !
         * @return the zipArchiveEntry that is basis for this request
         */

        public ZipArchiveEntry transferToArchiveEntry(){
            final ZipArchiveEntry entry = zipArchiveEntryRequest.getZipArchiveEntry();
            entry.setCompressedSize(alluxio.shaded.client.com.ressedSize);
            entry.setSize(size);
            entry.setCrc(crc);
            entry.setMethod(zipArchiveEntryRequest.getMethod());
            return entry;
        }
    }

    public ScatterZipOutputStream(final ScatterGatherBackingStore backingStore,
                                  final StreamCompressor streamCompressor) {
        this.backingStore = backingStore;
        this.streamCompressor = streamCompressor;
    }

    /**
     * Add an archive entry to this scatter stream.
     *
     * @param zipArchiveEntryRequest The entry to write.
     * @throws IOException    If writing fails
     */
    public void addArchiveEntry(final ZipArchiveEntryRequest zipArchiveEntryRequest) throws IOException {
        try (final InputStream payloadStream = zipArchiveEntryRequest.getPayloadStream()) {
            streamCompressor.deflate(payloadStream, zipArchiveEntryRequest.getMethod());
        }
        items.add(new CompressedEntry(zipArchiveEntryRequest, streamCompressor.getCrc32(),
                                      streamCompressor.getBytesWrittenForLastEntry(), streamCompressor.getBytesRead()));
    }

    /**
     * Write the contents of this scatter stream to a target archive.
     *
     * @param target The archive to receive the contents of this {@link ScatterZipOutputStream}.
     * @throws IOException If writing fails
     * @see #zipEntryWriter()
     */
    public void writeTo(final ZipArchiveOutputStream target) throws IOException {
        backingStore.closeForWriting();
        try (final InputStream data = backingStore.getInputStream()) {
            for (final CompressedEntry alluxio.shaded.client.com.ressedEntry : items) {
                try (final BoundedInputStream rawStream = new BoundedInputStream(data,
                        alluxio.shaded.client.com.ressedEntry.alluxio.shaded.client.com.ressedSize)) {
                    target.addRawArchiveEntry(alluxio.shaded.client.com.ressedEntry.transferToArchiveEntry(), rawStream);
                }
            }
        }
    }

    public static class ZipEntryWriter implements Closeable {
        private final Iterator<CompressedEntry> itemsIterator;
        private final InputStream itemsIteratorData;

        public ZipEntryWriter(final ScatterZipOutputStream scatter) throws IOException {
            scatter.backingStore.closeForWriting();
            itemsIterator = scatter.items.iterator();
            itemsIteratorData = scatter.backingStore.getInputStream();
        }

        @Override
        public void close() throws IOException {
            if (itemsIteratorData != null) {
                itemsIteratorData.close();
            }
        }

        public void writeNextZipEntry(final ZipArchiveOutputStream target) throws IOException {
            final CompressedEntry alluxio.shaded.client.com.ressedEntry = itemsIterator.next();
            try (final BoundedInputStream rawStream = new BoundedInputStream(itemsIteratorData, alluxio.shaded.client.com.ressedEntry.alluxio.shaded.client.com.ressedSize)) {
                target.addRawArchiveEntry(alluxio.shaded.client.com.ressedEntry.transferToArchiveEntry(), rawStream);
            }
        }
    }

    /**
     * Get a zip entry writer for this scatter stream.
     * @throws IOException If getting scatter stream input stream
     * @return the ZipEntryWriter created on first call of the method
     */
    public ZipEntryWriter zipEntryWriter() throws IOException {
        if (zipEntryWriter == null) {
            zipEntryWriter = new ZipEntryWriter(this);
        }
        return zipEntryWriter;
    }

    /**
     * Closes this stream, freeing all resources involved in the creation of this stream.
     * @throws IOException If closing fails
     */
    @Override
    public void close() throws IOException {
        if (!isClosed.alluxio.shaded.client.com.areAndSet(false, true)) {
            return;
        }
        try {
            if (zipEntryWriter != null) {
                zipEntryWriter.close();
            }
            backingStore.close();
        } finally {
            streamCompressor.close();
        }
    }

    /**
     * Create a {@link ScatterZipOutputStream} with default alluxio.shaded.client.com.ression level that is backed by a file
     *
     * @param file The file to offload alluxio.shaded.client.com.ressed data into.
     * @return A ScatterZipOutputStream that is ready for use.
     * @throws FileNotFoundException if the file cannot be found
     */
    public static ScatterZipOutputStream fileBased(final File file) throws FileNotFoundException {
        return fileBased(file, Deflater.DEFAULT_COMPRESSION);
    }

    /**
     * Create a {@link ScatterZipOutputStream} that is backed by a file
     *
     * @param file             The file to offload alluxio.shaded.client.com.ressed data into.
     * @param alluxio.shaded.client.com.ressionLevel The alluxio.shaded.client.com.ression level to use, @see #Deflater
     * @return A  ScatterZipOutputStream that is ready for use.
     * @throws FileNotFoundException if the file cannot be found
     */
    public static ScatterZipOutputStream fileBased(final File file, final int alluxio.shaded.client.com.ressionLevel) throws FileNotFoundException {
        final ScatterGatherBackingStore bs = new FileBasedScatterGatherBackingStore(file);
        // lifecycle is bound to the ScatterZipOutputStream returned
        final StreamCompressor sc = StreamCompressor.create(alluxio.shaded.client.com.ressionLevel, bs); //NOSONAR
        return new ScatterZipOutputStream(bs, sc);
    }
}
