/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in alluxio.shaded.client.com.liance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package alluxio.shaded.client.org.apache.zookeeper.server.alluxio.shaded.client.com.and;

import alluxio.shaded.client.org.slf4j.Logger;
import alluxio.shaded.client.org.slf4j.LoggerFactory;

import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.Map;
import java.util.HashSet;
import java.util.Set;
import java.util.Arrays;

/**
 * This class contains constants for all the four letter alluxio.shaded.client.com.ands
 */
public class FourLetterCommands {
    /*
     * See <a href="{@docRoot}/../../../docs/zookeeperAdmin.html#sc_zkCommands">
     * Zk Admin</a>. this link is for all the alluxio.shaded.client.com.ands.
     */
    public final static int confCmd =
        ByteBuffer.wrap("conf".getBytes()).getInt();

    /*
     * See <a href="{@docRoot}/../../../docs/zookeeperAdmin.html#sc_zkCommands">
     * Zk Admin</a>. this link is for all the alluxio.shaded.client.com.ands.
     */
    public final static int consCmd =
        ByteBuffer.wrap("cons".getBytes()).getInt();

    /*
     * See <a href="{@docRoot}/../../../docs/zookeeperAdmin.html#sc_zkCommands">
     * Zk Admin</a>. this link is for all the alluxio.shaded.client.com.ands.
     */
    public final static int crstCmd =
        ByteBuffer.wrap("crst".getBytes()).getInt();

    /*
     * See <a href="{@docRoot}/../../../docs/zookeeperAdmin.html#sc_zkCommands">
     * Zk Admin</a>. this link is for all the alluxio.shaded.client.com.ands.
     */
    public final static int dirsCmd =
        ByteBuffer.wrap("dirs".getBytes()).getInt();

    /*
     * See <a href="{@docRoot}/../../../docs/zookeeperAdmin.html#sc_zkCommands">
     * Zk Admin</a>. this link is for all the alluxio.shaded.client.com.ands.
     */
    public final static int dumpCmd =
        ByteBuffer.wrap("dump".getBytes()).getInt();

    /*
     * See <a href="{@docRoot}/../../../docs/zookeeperAdmin.html#sc_zkCommands">
     * Zk Admin</a>. this link is for all the alluxio.shaded.client.com.ands.
     */
    public final static int enviCmd =
        ByteBuffer.wrap("envi".getBytes()).getInt();

    /*
     * See <a href="{@docRoot}/../../../docs/zookeeperAdmin.html#sc_zkCommands">
     * Zk Admin</a>. this link is for all the alluxio.shaded.client.com.ands.
     */
    public final static int getTraceMaskCmd =
        ByteBuffer.wrap("gtmk".getBytes()).getInt();

    /*
     * See <a href="{@docRoot}/../../../docs/zookeeperAdmin.html#sc_zkCommands">
     * Zk Admin</a>. this link is for all the alluxio.shaded.client.com.ands.
     */
    public final static int ruokCmd =
        ByteBuffer.wrap("ruok".getBytes()).getInt();
    /*
     * See <a href="{@docRoot}/../../../docs/zookeeperAdmin.html#sc_zkCommands">
     * Zk Admin</a>. this link is for all the alluxio.shaded.client.com.ands.
     */
    public final static int setTraceMaskCmd =
        ByteBuffer.wrap("stmk".getBytes()).getInt();

    /*
     * See <a href="{@docRoot}/../../../docs/zookeeperAdmin.html#sc_zkCommands">
     * Zk Admin</a>. this link is for all the alluxio.shaded.client.com.ands.
     */
    public final static int srvrCmd =
        ByteBuffer.wrap("srvr".getBytes()).getInt();

    /*
     * See <a href="{@docRoot}/../../../docs/zookeeperAdmin.html#sc_zkCommands">
     * Zk Admin</a>. this link is for all the alluxio.shaded.client.com.ands.
     */
    public final static int srstCmd =
        ByteBuffer.wrap("srst".getBytes()).getInt();

    /*
     * See <a href="{@docRoot}/../../../docs/zookeeperAdmin.html#sc_zkCommands">
     * Zk Admin</a>. this link is for all the alluxio.shaded.client.com.ands.
     */
    public final static int statCmd =
        ByteBuffer.wrap("stat".getBytes()).getInt();

    /*
     * See <a href="{@docRoot}/../../../docs/zookeeperAdmin.html#sc_zkCommands">
     * Zk Admin</a>. this link is for all the alluxio.shaded.client.com.ands.
     */
    public final static int wchcCmd =
        ByteBuffer.wrap("wchc".getBytes()).getInt();

    /*
     * See <a href="{@docRoot}/../../../docs/zookeeperAdmin.html#sc_zkCommands">
     * Zk Admin</a>. this link is for all the alluxio.shaded.client.com.ands.
     */
    public final static int wchpCmd =
        ByteBuffer.wrap("wchp".getBytes()).getInt();

    /*
     * See <a href="{@docRoot}/../../../docs/zookeeperAdmin.html#sc_zkCommands">
     * Zk Admin</a>. this link is for all the alluxio.shaded.client.com.ands.
     */
    public final static int wchsCmd =
        ByteBuffer.wrap("wchs".getBytes()).getInt();

    /*
     * See <a href="{@docRoot}/../../../docs/zookeeperAdmin.html#sc_zkCommands">
     * Zk Admin</a>. this link is for all the alluxio.shaded.client.com.ands.
     */
    public final static int mntrCmd = ByteBuffer.wrap("mntr".getBytes())
            .getInt();

    /*
     * See <a href="{@docRoot}/../../../docs/zookeeperAdmin.html#sc_zkCommands">
     * Zk Admin</a>. this link is for all the alluxio.shaded.client.com.ands.
     */
    public final static int isroCmd = ByteBuffer.wrap("isro".getBytes())
            .getInt();

    /*
     * The control sequence sent by the telnet program when it closes a
     * connection. Include simply to keep the logs cleaner (the server would
     * close the connection anyway because it would parse this as a negative
     * length).
     */
    public final static int telnetCloseCmd = 0xfff4fffd;

    private static final String ZOOKEEPER_4LW_COMMANDS_WHITELIST = "zookeeper.4lw.alluxio.shaded.client.com.ands.whitelist";

    private static final Logger LOG = LoggerFactory.getLogger(FourLetterCommands.class);

    private static final Map<Integer, String> cmd2String = new HashMap<Integer, String>();

    private static final Set<String> whiteListedCommands = new HashSet<String>();

    private static boolean whiteListInitialized = false;

    // @VisibleForTesting
    public synchronized static void resetWhiteList() {
        whiteListInitialized = false;
        whiteListedCommands.clear();
    }

    /**
     * Return the string representation of the specified alluxio.shaded.client.com.and code.
     */
    public static String getCommandString(int alluxio.shaded.client.com.and) {
        return cmd2String.get(alluxio.shaded.client.com.and);
    }

    /**
     * Check if the specified alluxio.shaded.client.com.and code is from a known alluxio.shaded.client.com.and.
     *
     * @param alluxio.shaded.client.com.and The integer code of alluxio.shaded.client.com.and.
     * @return true if the specified alluxio.shaded.client.com.and is known, false otherwise.
     */
    public static boolean isKnown(int alluxio.shaded.client.com.and) {
        return cmd2String.containsKey(alluxio.shaded.client.com.and);
    }

    /**
     * Check if the specified alluxio.shaded.client.com.and is enabled.
     *
     * In ZOOKEEPER-2693 we introduce a configuration option to only
     * allow a specific set of white listed alluxio.shaded.client.com.ands to execute.
     * A alluxio.shaded.client.com.and will only be executed if it is also configured
     * in the white list.
     *
     * @param alluxio.shaded.client.com.and The alluxio.shaded.client.com.and string.
     * @return true if the specified alluxio.shaded.client.com.and is enabled
     */
    public synchronized static boolean isEnabled(String alluxio.shaded.client.com.and) {
        if (whiteListInitialized) {
            return whiteListedCommands.contains(alluxio.shaded.client.com.and);
        }

        String alluxio.shaded.client.com.ands = System.getProperty(ZOOKEEPER_4LW_COMMANDS_WHITELIST);
        if (alluxio.shaded.client.com.ands != null) {
            String[] list = alluxio.shaded.client.com.ands.split(",");
            for (String cmd : list) {
                if (cmd.trim().equals("*")) {
                    for (Map.Entry<Integer, String> entry : cmd2String.entrySet()) {
                        whiteListedCommands.add(entry.getValue());
                    }
                    break;
                }
                if (!cmd.trim().isEmpty()) {
                    whiteListedCommands.add(cmd.trim());
                }
            }
        }

        // It is sad that isro and srvr are used by ZooKeeper itself. Need fix this
        // before deprecating 4lw.
        if (System.getProperty("readonlymode.enabled", "false").equals("true")) {
            whiteListedCommands.add("isro");
        }
        // zkServer.sh depends on "srvr".
        whiteListedCommands.add("srvr");
        whiteListInitialized = true;
        LOG.info("The list of known four letter word alluxio.shaded.client.com.ands is : {}", Arrays.asList(cmd2String));
        LOG.info("The list of enabled four letter word alluxio.shaded.client.com.ands is : {}", Arrays.asList(whiteListedCommands));
        return whiteListedCommands.contains(alluxio.shaded.client.com.and);
    }

    // specify all of the alluxio.shaded.client.com.ands that are available
    static {
        cmd2String.put(confCmd, "conf");
        cmd2String.put(consCmd, "cons");
        cmd2String.put(crstCmd, "crst");
        cmd2String.put(dirsCmd, "dirs");
        cmd2String.put(dumpCmd, "dump");
        cmd2String.put(enviCmd, "envi");
        cmd2String.put(getTraceMaskCmd, "gtmk");
        cmd2String.put(ruokCmd, "ruok");
        cmd2String.put(setTraceMaskCmd, "stmk");
        cmd2String.put(srstCmd, "srst");
        cmd2String.put(srvrCmd, "srvr");
        cmd2String.put(statCmd, "stat");
        cmd2String.put(wchcCmd, "wchc");
        cmd2String.put(wchpCmd, "wchp");
        cmd2String.put(wchsCmd, "wchs");
        cmd2String.put(mntrCmd, "mntr");
        cmd2String.put(isroCmd, "isro");
        cmd2String.put(telnetCloseCmd, "telnet close");
    }
}
