package alluxio.shaded.client.com.codahale.metrics.jvm;

import alluxio.shaded.client.com.codahale.metrics.Gauge;

import java.alluxio.shaded.client.io.IOException;
import alluxio.shaded.client.javax.management.JMException;
import alluxio.shaded.client.javax.management.MBeanServerConnection;
import alluxio.shaded.client.javax.management.ObjectName;
import java.lang.management.ManagementFactory;
import java.util.Set;

/**
 * A {@link Gauge} implementation which queries an {@link MBeanServerConnection} for an attribute of an object.
 */
public class JmxAttributeGauge implements Gauge<Object> {
    private final MBeanServerConnection mBeanServerConn;
    private final ObjectName objectName;
    private final String attributeName;

    /**
     * Creates a new JmxAttributeGauge.
     *
     * @param objectName    the name of the object
     * @param attributeName the name of the object's attribute
     */
    public JmxAttributeGauge(ObjectName objectName, String attributeName) {
        this(ManagementFactory.getPlatformMBeanServer(), objectName, attributeName);
    }

    /**
     * Creates a new JmxAttributeGauge.
     *
     * @param mBeanServerConn the {@link MBeanServerConnection}
     * @param objectName      the name of the object
     * @param attributeName   the name of the object's attribute
     */
    public JmxAttributeGauge(MBeanServerConnection mBeanServerConn, ObjectName objectName, String attributeName) {
        this.mBeanServerConn = mBeanServerConn;
        this.objectName = objectName;
        this.attributeName = attributeName;
    }

    @Override
    public Object getValue() {
        try {
            return mBeanServerConn.getAttribute(getObjectName(), attributeName);
        } catch (IOException | JMException e) {
            return null;
        }
    }

    private ObjectName getObjectName() throws IOException {
        if (objectName.isPattern()) {
            Set<ObjectName> foundNames = mBeanServerConn.queryNames(objectName, null);
            if (foundNames.size() == 1) {
                return foundNames.iterator().next();
            }
        }
        return objectName;
    }
}
