/*
 * Copyright 2018 The gRPC Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in alluxio.shaded.client.com.liance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package alluxio.shaded.client.io.grpc.grpclb;

import alluxio.shaded.client.io.grpc.Attributes;
import alluxio.shaded.client.io.grpc.ConnectivityStateInfo;
import alluxio.shaded.client.io.grpc.EquivalentAddressGroup;
import alluxio.shaded.client.io.grpc.LoadBalancer.Subchannel;
import alluxio.shaded.client.javax.annotation.concurrent.NotThreadSafe;

/**
 * Manages life-cycle of Subchannels for {@link GrpclbState}.
 *
 * <p>All methods are run from the ChannelExecutor that the helper uses.
 */
@NotThreadSafe
interface SubchannelPool {

  /**
   * Registers a listener to received Subchannel status updates.
   */
  void registerListener(PooledSubchannelStateListener listener);

  /**
   * Takes a {@link Subchannel} from the pool for the given {@code eag} if there is one available.
   * Otherwise, creates and returns a new {@code Subchannel} with the given {@code eag} and {@code
   * defaultAttributes}.
   */
  Subchannel takeOrCreateSubchannel(EquivalentAddressGroup eag, Attributes defaultAttributes);

  /**
   * Puts a {@link Subchannel} back to the pool.  From this point the Subchannel is owned by the
   * pool, and the caller should stop referencing to this Subchannel.
   */
  void returnSubchannel(Subchannel subchannel, ConnectivityStateInfo lastKnownState);

  /**
   * Shuts down all subchannels in the pool immediately.
   */
  void clear();

  /**
   * Receives state changes for a pooled {@link Subchannel}.
   */
  interface PooledSubchannelStateListener {

    /**
     * Handles a state change on a Subchannel. The behavior is similar to {@link
     * alluxio.shaded.client.io.grpc.LoadBalancer.SubchannelStateListener}.
     *
     * <p>When a subchannel is reused, subchannel state change event will be triggered even if the
     * underlying status remains same.
     */
    void onSubchannelState(Subchannel subchannel, ConnectivityStateInfo newState);

  }
}
