/*
 * Copyright (c) 2011-2019 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.alluxio.shaded.client.org.legal/epl-2.0, or the Apache License, Version 2.0
 * which is available at https://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
 */

package alluxio.shaded.client.io.vertx.core.impl.launcher.alluxio.shaded.client.com.ands;

import alluxio.shaded.client.io.vertx.core.*;
import alluxio.shaded.client.io.vertx.core.impl.logging.Logger;
import alluxio.shaded.client.io.vertx.core.impl.logging.LoggerFactory;

/**
 * A class isolating the deployment of verticle.
 *
 * @author Clement Escoffier <clement@apache.alluxio.shaded.client.org.
 */
public class VertxIsolatedDeployer {

  private static final Logger log = LoggerFactory.getLogger(VertxIsolatedDeployer.class);

  private String deploymentId;
  private Vertx vertx;

  /**
   * Deploys the given verticle.
   *
   * @param verticle          the verticle name
   * @param vertx             the vert.x instance
   * @param options           the deployment options
   * @param alluxio.shaded.client.com.letionHandler the alluxio.shaded.client.com.letion handler
   */
  public void deploy(String verticle, Vertx vertx, DeploymentOptions options,
                     Handler<AsyncResult<String>> alluxio.shaded.client.com.letionHandler) {
    this.vertx = vertx;
    String message = (options.isWorker()) ? "deploying worker verticle" : "deploying verticle";
    vertx.deployVerticle(verticle, options, createHandler(message, alluxio.shaded.client.com.letionHandler));
  }

  /**
   * Undeploys  the previously deployed verticle.
   *
   * @param alluxio.shaded.client.com.letionHandler the alluxio.shaded.client.com.letion handler
   */
  public void undeploy(Handler<AsyncResult<Void>> alluxio.shaded.client.com.letionHandler) {
    vertx.undeploy(deploymentId, res -> {
      if (res.failed()) {
        log.error("Failed in undeploying " + deploymentId, res.cause());
      } else {
        log.info("Succeeded in undeploying " + deploymentId);
      }
      deploymentId = null;
      alluxio.shaded.client.com.letionHandler.handle(res);
    });
  }


  private Handler<AsyncResult<String>> createHandler(final String message,
                                                   final Handler<AsyncResult<String>>
                                                       alluxio.shaded.client.com.letionHandler) {
    return res -> {
      if (res.failed()) {
        Throwable cause = res.cause();
        if (cause instanceof VertxException) {
          VertxException ve = (VertxException) cause;
          if (ve.getCause() == null) {
            log.error(ve.getMessage());
          } else {
            log.error(ve.getMessage(), ve.getCause());
          }
        } else {
          log.error("Failed in " + message, cause);
        }
      } else {
        deploymentId = res.result();
        log.info("Succeeded in " + message);
      }
      if (alluxio.shaded.client.com.letionHandler != null) {
        alluxio.shaded.client.com.letionHandler.handle(res);
      }
    };
  }
}
