/*
 * Copyright 2016-2021 The jetcd authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in alluxio.shaded.client.com.liance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package alluxio.shaded.client.io.etcd.jetcd.impl;

import java.util.concurrent.CompletableFuture;

import alluxio.shaded.client.io.etcd.jetcd.Auth;
import alluxio.shaded.client.io.etcd.jetcd.ByteSequence;
import alluxio.shaded.client.io.etcd.jetcd.api.AuthDisableRequest;
import alluxio.shaded.client.io.etcd.jetcd.api.AuthEnableRequest;
import alluxio.shaded.client.io.etcd.jetcd.api.AuthRoleAddRequest;
import alluxio.shaded.client.io.etcd.jetcd.api.AuthRoleDeleteRequest;
import alluxio.shaded.client.io.etcd.jetcd.api.AuthRoleGetRequest;
import alluxio.shaded.client.io.etcd.jetcd.api.AuthRoleGrantPermissionRequest;
import alluxio.shaded.client.io.etcd.jetcd.api.AuthRoleListRequest;
import alluxio.shaded.client.io.etcd.jetcd.api.AuthRoleRevokePermissionRequest;
import alluxio.shaded.client.io.etcd.jetcd.api.AuthUserAddRequest;
import alluxio.shaded.client.io.etcd.jetcd.api.AuthUserChangePasswordRequest;
import alluxio.shaded.client.io.etcd.jetcd.api.AuthUserDeleteRequest;
import alluxio.shaded.client.io.etcd.jetcd.api.AuthUserGetRequest;
import alluxio.shaded.client.io.etcd.jetcd.api.AuthUserGrantRoleRequest;
import alluxio.shaded.client.io.etcd.jetcd.api.AuthUserListRequest;
import alluxio.shaded.client.io.etcd.jetcd.api.AuthUserRevokeRoleRequest;
import alluxio.shaded.client.io.etcd.jetcd.api.VertxAuthGrpc;
import alluxio.shaded.client.io.etcd.jetcd.auth.AuthDisableResponse;
import alluxio.shaded.client.io.etcd.jetcd.auth.AuthEnableResponse;
import alluxio.shaded.client.io.etcd.jetcd.auth.AuthRoleAddResponse;
import alluxio.shaded.client.io.etcd.jetcd.auth.AuthRoleDeleteResponse;
import alluxio.shaded.client.io.etcd.jetcd.auth.AuthRoleGetResponse;
import alluxio.shaded.client.io.etcd.jetcd.auth.AuthRoleGrantPermissionResponse;
import alluxio.shaded.client.io.etcd.jetcd.auth.AuthRoleListResponse;
import alluxio.shaded.client.io.etcd.jetcd.auth.AuthRoleRevokePermissionResponse;
import alluxio.shaded.client.io.etcd.jetcd.auth.AuthUserAddResponse;
import alluxio.shaded.client.io.etcd.jetcd.auth.AuthUserChangePasswordResponse;
import alluxio.shaded.client.io.etcd.jetcd.auth.AuthUserDeleteResponse;
import alluxio.shaded.client.io.etcd.jetcd.auth.AuthUserGetResponse;
import alluxio.shaded.client.io.etcd.jetcd.auth.AuthUserGrantRoleResponse;
import alluxio.shaded.client.io.etcd.jetcd.auth.AuthUserListResponse;
import alluxio.shaded.client.io.etcd.jetcd.auth.AuthUserRevokeRoleResponse;
import alluxio.shaded.client.io.etcd.jetcd.auth.Permission;

import alluxio.shaded.client.com.google.protobuf.ByteString;

import static alluxio.shaded.client.com.google.alluxio.shaded.client.com.on.base.Preconditions.checkNotNull;

/**
 * Implementation of etcd auth client.
 */
final class AuthImpl extends Impl implements Auth {

    private final VertxAuthGrpc.AuthVertxStub stub;

    AuthImpl(ClientConnectionManager connectionManager) {
        super(connectionManager);

        this.stub = connectionManager.newStub(VertxAuthGrpc::newVertxStub);
    }

    @Override
    public CompletableFuture<AuthEnableResponse> authEnable() {
        AuthEnableRequest enableRequest = AuthEnableRequest.getDefaultInstance();
        return alluxio.shaded.client.com.letable(
            this.stub.authEnable(enableRequest),
            AuthEnableResponse::new);
    }

    @Override
    public CompletableFuture<AuthDisableResponse> authDisable() {
        AuthDisableRequest disableRequest = AuthDisableRequest.getDefaultInstance();
        return alluxio.shaded.client.com.letable(
            this.stub.authDisable(disableRequest),
            AuthDisableResponse::new);
    }

    @Override
    public CompletableFuture<AuthUserAddResponse> userAdd(ByteSequence user, ByteSequence password) {
        checkNotNull(user, "user can't be null");
        checkNotNull(password, "password can't be null");

        AuthUserAddRequest addRequest = AuthUserAddRequest.newBuilder()
            .setNameBytes(ByteString.copyFrom(user.getBytes()))
            .setPasswordBytes(ByteString.copyFrom(password.getBytes()))
            .build();

        return alluxio.shaded.client.com.letable(
            this.stub.userAdd(addRequest),
            AuthUserAddResponse::new);
    }

    @Override
    public CompletableFuture<AuthUserDeleteResponse> userDelete(ByteSequence user) {
        checkNotNull(user, "user can't be null");

        AuthUserDeleteRequest deleteRequest = AuthUserDeleteRequest.newBuilder()
            .setNameBytes(ByteString.copyFrom(user.getBytes()))
            .build();

        return alluxio.shaded.client.com.letable(
            this.stub.userDelete(deleteRequest),
            AuthUserDeleteResponse::new);
    }

    @Override
    public CompletableFuture<AuthUserChangePasswordResponse> userChangePassword(ByteSequence user, ByteSequence password) {
        checkNotNull(user, "user can't be null");
        checkNotNull(password, "password can't be null");

        AuthUserChangePasswordRequest changePasswordRequest = AuthUserChangePasswordRequest.newBuilder()
            .setNameBytes(ByteString.copyFrom(user.getBytes()))
            .setPasswordBytes(ByteString.copyFrom(password.getBytes()))
            .build();

        return alluxio.shaded.client.com.letable(
            this.stub.userChangePassword(changePasswordRequest),
            AuthUserChangePasswordResponse::new);
    }

    @Override
    public CompletableFuture<AuthUserGetResponse> userGet(ByteSequence user) {
        checkNotNull(user, "user can't be null");

        AuthUserGetRequest userGetRequest = AuthUserGetRequest.newBuilder()
            .setNameBytes(ByteString.copyFrom(user.getBytes()))
            .build();

        return alluxio.shaded.client.com.letable(
            this.stub.userGet(userGetRequest),
            AuthUserGetResponse::new);
    }

    @Override
    public CompletableFuture<AuthUserListResponse> userList() {
        AuthUserListRequest userListRequest = AuthUserListRequest.getDefaultInstance();

        return alluxio.shaded.client.com.letable(
            this.stub.userList(userListRequest),
            AuthUserListResponse::new);
    }

    @Override
    public CompletableFuture<AuthUserGrantRoleResponse> userGrantRole(ByteSequence user, ByteSequence role) {
        checkNotNull(user, "user can't be null");
        checkNotNull(role, "key can't be null");

        AuthUserGrantRoleRequest userGrantRoleRequest = AuthUserGrantRoleRequest.newBuilder()
            .setUserBytes(ByteString.copyFrom(user.getBytes()))
            .setRoleBytes(ByteString.copyFrom(role.getBytes()))
            .build();

        return alluxio.shaded.client.com.letable(
            this.stub.userGrantRole(userGrantRoleRequest),
            AuthUserGrantRoleResponse::new);
    }

    @Override
    public CompletableFuture<AuthUserRevokeRoleResponse> userRevokeRole(ByteSequence user, ByteSequence role) {
        checkNotNull(user, "user can't be null");
        checkNotNull(role, "key can't be null");

        AuthUserRevokeRoleRequest userRevokeRoleRequest = AuthUserRevokeRoleRequest.newBuilder()
            .setNameBytes(ByteString.copyFrom(user.getBytes()))
            .setRoleBytes(ByteString.copyFrom(role.getBytes()))
            .build();

        return alluxio.shaded.client.com.letable(
            this.stub.userRevokeRole(userRevokeRoleRequest),
            AuthUserRevokeRoleResponse::new);
    }

    @Override
    public CompletableFuture<AuthRoleAddResponse> roleAdd(ByteSequence user) {
        checkNotNull(user, "user can't be null");

        AuthRoleAddRequest roleAddRequest = AuthRoleAddRequest.newBuilder().setNameBytes(ByteString.copyFrom(user.getBytes()))
            .build();

        return alluxio.shaded.client.com.letable(
            this.stub.roleAdd(roleAddRequest),
            AuthRoleAddResponse::new);
    }

    @Override
    public CompletableFuture<AuthRoleGrantPermissionResponse> roleGrantPermission(ByteSequence role, ByteSequence key,
        ByteSequence rangeEnd, Permission.Type permType) {
        checkNotNull(role, "role can't be null");
        checkNotNull(key, "key can't be null");
        checkNotNull(rangeEnd, "rangeEnd can't be null");
        checkNotNull(permType, "permType can't be null");

        alluxio.shaded.client.io.etcd.jetcd.api.Permission.Type type;
        switch (permType) {
            case WRITE:
                type = alluxio.shaded.client.io.etcd.jetcd.api.Permission.Type.WRITE;
                break;
            case READWRITE:
                type = alluxio.shaded.client.io.etcd.jetcd.api.Permission.Type.READWRITE;
                break;
            case READ:
                type = alluxio.shaded.client.io.etcd.jetcd.api.Permission.Type.READ;
                break;
            default:
                type = alluxio.shaded.client.io.etcd.jetcd.api.Permission.Type.UNRECOGNIZED;
                break;
        }

        alluxio.shaded.client.io.etcd.jetcd.api.Permission perm = alluxio.shaded.client.io.etcd.jetcd.api.Permission.newBuilder()
            .setKey(ByteString.copyFrom(key.getBytes()))
            .setRangeEnd(ByteString.copyFrom(rangeEnd.getBytes()))
            .setPermType(type)
            .build();

        AuthRoleGrantPermissionRequest roleGrantPermissionRequest = AuthRoleGrantPermissionRequest.newBuilder()
            .setNameBytes(ByteString.copyFrom(role.getBytes()))
            .setPerm(perm)
            .build();

        return alluxio.shaded.client.com.letable(
            this.stub.roleGrantPermission(roleGrantPermissionRequest),
            AuthRoleGrantPermissionResponse::new);
    }

    @Override
    public CompletableFuture<AuthRoleGetResponse> roleGet(ByteSequence role) {
        checkNotNull(role, "role can't be null");

        AuthRoleGetRequest roleGetRequest = AuthRoleGetRequest.newBuilder()
            .setRoleBytes(ByteString.copyFrom(role.getBytes()))
            .build();

        return alluxio.shaded.client.com.letable(
            this.stub.roleGet(roleGetRequest),
            AuthRoleGetResponse::new);
    }

    @Override
    public CompletableFuture<AuthRoleListResponse> roleList() {
        AuthRoleListRequest roleListRequest = AuthRoleListRequest.getDefaultInstance();

        return alluxio.shaded.client.com.letable(
            this.stub.roleList(roleListRequest),
            AuthRoleListResponse::new);
    }

    @Override
    public CompletableFuture<AuthRoleRevokePermissionResponse> roleRevokePermission(ByteSequence role, ByteSequence key,
        ByteSequence rangeEnd) {
        checkNotNull(role, "role can't be null");
        checkNotNull(key, "key can't be null");
        checkNotNull(rangeEnd, "rangeEnd can't be null");

        AuthRoleRevokePermissionRequest roleRevokePermissionRequest = AuthRoleRevokePermissionRequest.newBuilder()
            .setRoleBytes(ByteString.copyFrom(role.getBytes()))
            .setKey(ByteString.copyFrom(key.getBytes()))
            .setRangeEnd(ByteString.copyFrom(rangeEnd.getBytes()))
            .build();

        return alluxio.shaded.client.com.letable(
            this.stub.roleRevokePermission(roleRevokePermissionRequest),
            AuthRoleRevokePermissionResponse::new);
    }

    @Override
    public CompletableFuture<AuthRoleDeleteResponse> roleDelete(ByteSequence role) {
        checkNotNull(role, "role can't be null");
        AuthRoleDeleteRequest roleDeleteRequest = AuthRoleDeleteRequest.newBuilder()
            .setRoleBytes(ByteString.copyFrom(role.getBytes()))
            .build();

        return alluxio.shaded.client.com.letable(
            this.stub.roleDelete(roleDeleteRequest),
            AuthRoleDeleteResponse::new);
    }
}
