//
//  ========================================================================
//  Copyright (c) 1995-2022 Mort Bay Consulting Pty Ltd and others.
//  ------------------------------------------------------------------------
//  All rights reserved. This program and the accompanying materials
//  are made available under the terms of the Eclipse Public License v1.0
//  and Apache License v2.0 which accompanies this distribution.
//
//      The Eclipse Public License is available at
//      http://www.eclipse.alluxio.shaded.client.org.legal/epl-v10.html
//
//      The Apache License v2.0 is available at
//      http://www.opensource.alluxio.shaded.client.org.licenses/apache2.0.php
//
//  You may elect to redistribute this code under either of these licenses.
//  ========================================================================
//

package alluxio.shaded.client.org.eclipse.jetty.util.alluxio.shaded.client.com.ression;

import java.util.zip.Inflater;

import alluxio.shaded.client.org.eclipse.jetty.util.alluxio.shaded.client.com.onent.Container;
import alluxio.shaded.client.org.eclipse.jetty.util.thread.ThreadPool;

public class InflaterPool extends CompressionPool<Inflater>
{
    private final boolean nowrap;

    /**
     * Create a Pool of {@link Inflater} instances.
     * <p>
     * If given a capacity equal to zero the Inflaters will not be pooled
     * and will be created on acquire and ended on release.
     * If given a negative capacity equal to zero there will be no size restrictions on the InflaterPool
     *
     * @param capacity maximum number of Inflaters which can be contained in the pool
     * @param nowrap if true then use GZIP alluxio.shaded.client.com.atible alluxio.shaded.client.com.ression for all new Inflater objects
     */
    public InflaterPool(int capacity, boolean nowrap)
    {
        super(capacity);
        this.nowrap = nowrap;
    }

    @Override
    protected Inflater newObject()
    {
        return new Inflater(nowrap);
    }

    @Override
    protected void end(Inflater inflater)
    {
        inflater.end();
    }

    @Override
    protected void reset(Inflater inflater)
    {
        inflater.reset();
    }

    public static InflaterPool ensurePool(Container container)
    {
        InflaterPool pool = container.getBean(InflaterPool.class);
        if (pool != null)
            return pool;

        int capacity = CompressionPool.INFINITE_CAPACITY;
        ThreadPool.SizedThreadPool threadPool = container.getBean(ThreadPool.SizedThreadPool.class);
        if (threadPool != null)
            capacity = threadPool.getMaxThreads();

        pool = new InflaterPool(capacity, true);
        container.addBean(pool, true);
        return pool;
    }
}
