/*
 * Copyright (c) 2011-2019 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.alluxio.shaded.client.org.legal/epl-2.0, or the Apache License, Version 2.0
 * which is available at https://www.apache.alluxio.shaded.client.org.licenses/LICENSE-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
 */

package alluxio.shaded.client.io.vertx.core.datagram.impl;

import alluxio.shaded.client.io.vertx.core.AsyncResult;
import alluxio.shaded.client.io.vertx.core.Future;
import alluxio.shaded.client.io.vertx.core.Handler;
import alluxio.shaded.client.io.vertx.core.Promise;
import alluxio.shaded.client.io.vertx.core.buffer.Buffer;
import alluxio.shaded.client.io.vertx.core.streams.WriteStream;

/**
 * A write stream for packets.
 *
* @author <a href="mailto:julien@julienviet.alluxio.shaded.client.com.>Julien Viet</a>
*/
class PacketWriteStreamImpl implements WriteStream<Buffer>, Handler<AsyncResult<Void>> {

  private DatagramSocketImpl datagramSocket;
  private Handler<Throwable> exceptionHandler;
  private final int port;
  private final String host;

  PacketWriteStreamImpl(DatagramSocketImpl datagramSocket, int port, String host) {
    this.datagramSocket = datagramSocket;
    this.port = port;
    this.host = host;
  }

  @Override
  public void handle(AsyncResult<Void> event) {
    if (event.failed() && exceptionHandler != null) {
      exceptionHandler.handle(event.cause());
    }
  }

  @Override
  public PacketWriteStreamImpl exceptionHandler(Handler<Throwable> handler) {
    exceptionHandler = handler;
    return this;
  }

  @Override
  public Future<Void> write(Buffer data) {
    Promise<Void> promise = Promise.promise();
    write(data, promise);
    return promise.future();
  }

  @Override
  public void write(Buffer data, Handler<AsyncResult<Void>> handler) {
    datagramSocket.send(data, port, host, ar -> {
      PacketWriteStreamImpl.this.handle(ar);
      if (handler != null) {
        handler.handle(ar.mapEmpty());
      }
    });
  }

  @Override
  public PacketWriteStreamImpl setWriteQueueMaxSize(int maxSize) {
    return this;
  }

  @Override
  public boolean writeQueueFull() {
    return false;
  }

  @Override
  public PacketWriteStreamImpl drainHandler(Handler<Void> handler) {
    return this;
  }

  @Override
  public Future<Void> end() {
    Promise<Void> promide = Promise.promise();
    end(promide);
    return promide.future();
  }

  @Override
  public void end(Handler<AsyncResult<Void>> handler) {
    datagramSocket.close(handler);
  }
}
