/*
 * Copyright 2020-2021 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License")
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * @author : anas
 * Date :   27-Feb-2021
 */

package org.anasoid.jmc.core.wrapper.jmeter.protocol.http.control;

import com.thoughtworks.xstream.annotations.XStreamOmitField;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import lombok.Builder.Default;
import lombok.Getter;
import lombok.experimental.SuperBuilder;
import org.anasoid.jmc.core.application.validator.annotations.JmcChildrenTypes;
import org.anasoid.jmc.core.wrapper.jmeter.config.ConfigTestElementWrapper;
import org.anasoid.jmc.core.xstream.annotations.JmcCollection;
import org.anasoid.jmc.core.xstream.annotations.JmcDefaultName;
import org.anasoid.jmc.core.xstream.annotations.JmcEmptyAllowed;
import org.apache.jmeter.protocol.http.control.HeaderManager;
import org.apache.jmeter.protocol.http.gui.HeaderPanel;

/**
 * Wrapper for HeaderManager.
 *
 * @see HeaderManager
 */
@SuperBuilder(setterPrefix = "with", toBuilder = true)
@JmcChildrenTypes(type = {HeaderWrapper.class})
@JmcDefaultName("HTTP Header Manager")
public class HeaderManagerWrapper extends ConfigTestElementWrapper<HeaderManager, HeaderPanel> {

  @XStreamOmitField private static final long serialVersionUID = 8482403319301067878L;

  @JmcCollection(value = "HeaderManager.headers")
  @Getter
  @JmcEmptyAllowed
  @Default
  private final List<HeaderWrapper> headers = new ArrayList<>();

  @Override
  public Class<?> getGuiClass() {
    return HeaderPanel.class;
  }

  @Override
  public Class<?> getTestClass() {
    return HeaderManager.class;
  }

  /** Builder. */
  @SuppressWarnings({"PMD.TooManyMethods"})
  public abstract static class HeaderManagerWrapperBuilder<
          C extends HeaderManagerWrapper, B extends HeaderManagerWrapperBuilder<C, B>>
      extends ConfigTestElementWrapper.ConfigTestElementWrapperBuilder<
          HeaderManager, HeaderPanel, C, B> {

    /** hide method , generated by Lombok. */
    @SuppressWarnings("PMD.UnusedFormalParameter")
    private B withHeaders(List<HeaderWrapper> headers) {
      this.headers$value = headers;
      this.headers$set = true;
      return self();
    }

    /** Add header List. */
    public B addHeaders(List<HeaderWrapper> headers) {
      if (!this.headers$set) {
        withHeaders(new ArrayList<>());
      }
      this.headers$value.addAll(headers);
      return self();
    }

    /** Add header. */
    public B addHeader(HeaderWrapper headers) {
      return this.addHeaders(Arrays.asList(headers));
    }

    /**
     * Add header .
     *
     * @param name name.
     * @param value value.
     */
    public B addHeader(String name, String value) {
      return this.addHeader(HeaderWrapper.builder().withName(name).withValue(value).build());
    }

    /** set Accept header. */
    public B acceptHeader(String value) {
      return addHeader("Accept", value);
    }

    /** set Accept-Charset header. */
    public B acceptCharsetHeader(String value) {
      return addHeader("Accept-Charset", value);
    }

    /** set Accept-Encoding header. */
    public B acceptEncodingHeader(String value) {
      return addHeader("Accept-Encoding", value);
    }

    /** set Accept-Language header. */
    public B acceptLanguageHeader(String value) {
      return addHeader("Accept-Language", value);
    }

    /** set Authorization header. */
    public B authorizationHeader(String value) {
      return addHeader("Authorization", value);
    }

    /** set Content-Type header. */
    public B contentTypeHeader(String value) {
      return addHeader("Content-Type", value);
    }

    /** set User-Agent header. */
    public B userAgentHeader(String value) {
      return addHeader("User-Agent", value);
    }
  }
}
