/*-
 * #%L
 * anchor-feature-session
 * %%
 * Copyright (C) 2010 - 2020 Owen Feehan, ETH Zurich, University of Zurich, Hoffmann-La Roche
 * %%
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * #L%
 */

package org.anchoranalysis.feature.session.cache.finder;

import java.util.Optional;
import java.util.Set;
import lombok.AllArgsConstructor;
import org.anchoranalysis.core.cache.LRUCache;
import org.anchoranalysis.core.exception.OperationFailedException;
import org.anchoranalysis.feature.calculate.FeatureCalculationInput;
import org.anchoranalysis.feature.calculate.cache.ChildCacheName;
import org.anchoranalysis.feature.input.FeatureInput;

/**
 * An existing cache that can be used as a source for child-caches elsewhere.
 *
 * @author Owen Feehan
 * @param <T> input-type associated with cache
 */
@AllArgsConstructor
public class CacheTransferSource<T extends FeatureInput> {

    /** The cache that is searched for child caches. */
    private final CacheSupplier<T, OperationFailedException> cacheToSearch;

    /** The names of the child-caches that can be used as sources. */
    private final Set<ChildCacheName> specificChildren;

    /**
     * Whether a particular child-cache exists in the source?
     *
     * @param name the name of the child-cache.
     * @return true iff the child-cache exists.
     */
    public boolean containsChild(ChildCacheName name) {
        return specificChildren.contains(name);
    }

    /**
     * Gets corresponding {@link FeatureCalculationInput} for an {@code input}, if present, but
     * doesn't create any new entry if it's absent.
     *
     * @param input the input.
     * @return an existing element if present or {@link Optional#empty()} otherwise.
     * @throws OperationFailedException if unable to get the cache that is searched.
     */
    public Optional<FeatureCalculationInput<T>> getInputIfPresent(T input)
            throws OperationFailedException {
        Optional<LRUCache<T, FeatureCalculationInput<T>>> cache = cacheToSearch.get();
        return cache.flatMap(cacheInternal -> cacheInternal.getIfPresent(input));
    }

    /**
     * The names of the child-caches that can be used as sources.
     *
     * @return the names, as a set, which should not be modified (i.e. treated as read-only).
     */
    public Set<ChildCacheName> getCacheNames() {
        return specificChildren;
    }
}
