/*-
 * #%L
 * anchor-io-output
 * %%
 * Copyright (C) 2010 - 2020 Owen Feehan, ETH Zurich, University of Zurich, Hoffmann-La Roche
 * %%
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * #L%
 */

package org.anchoranalysis.io.output.bean.enabled;

import lombok.NoArgsConstructor;

/**
 * All outputs are allowed unless the name's first character is an underscore.
 *
 * <p>Additionally, when a underscore occurs in any location, preceded by a period i.e. {@code ._}
 * this output is also allowed. This is intended for cases of namespace-like hierarchy of outputs.
 *
 * <p>This is intended as a filter to allow declaration of certain <i>private</i> entities that are
 * not typically outputted, unless explicitly specified.
 *
 * @author Owen Feehan
 */
@NoArgsConstructor
public class IgnoreUnderscorePrefix extends OutputEnabled {

    /**
     * A singleton instance of {@link IgnoreUnderscorePrefix}.
     *
     * <p>The class retains a public constructor so it can also be instantiated as a bean.
     */
    public static final OutputEnabled INSTANCE = new IgnoreUnderscorePrefix();

    @Override
    public boolean isOutputEnabled(String outputName) {
        return outputName.isEmpty()
                || (outputName.charAt(0) != '_' && !containsPeriodUnderscore(outputName));
    }

    private static boolean containsPeriodUnderscore(String outputName) {
        return outputName.contains("._");
    }
}
