/*
 * Decompiled with CFR 0.152.
 */
package org.anchoranalysis.math.histogram;

import java.util.Arrays;
import java.util.function.DoublePredicate;
import java.util.function.IntFunction;
import java.util.function.IntPredicate;
import java.util.function.LongUnaryOperator;
import lombok.Generated;
import org.anchoranalysis.core.exception.OperationFailedException;
import org.anchoranalysis.core.exception.friendly.AnchorImpossibleSituationException;
import org.anchoranalysis.math.statistics.VarianceCalculatorLong;
import org.apache.commons.lang.ArrayUtils;

public final class Histogram {
    private int minValue;
    private int maxValue;
    private int[] counts;
    private long sumCount = 0L;

    public Histogram(int maxValue) {
        this(0, maxValue);
    }

    public Histogram(int minValue, int maxValue) {
        this.counts = new int[maxValue - minValue + 1];
        this.sumCount = 0L;
        this.maxValue = maxValue;
        this.minValue = minValue;
    }

    public Histogram duplicate() {
        Histogram out = new Histogram(this.minValue, this.maxValue);
        out.counts = ArrayUtils.clone((int[])this.counts);
        out.sumCount = this.sumCount;
        return out;
    }

    public void reset() {
        this.sumCount = 0L;
        for (int i = this.minValue; i <= this.maxValue; ++i) {
            this.set(i, 0);
        }
    }

    public void zeroValue(int value) {
        int index = this.index(value);
        this.sumCount -= (long)this.counts[index];
        this.counts[index] = 0;
    }

    public void transferCount(int valueFrom, int valueTo) {
        int indexFrom = this.index(valueFrom);
        this.incrementCount(valueTo, this.counts[indexFrom]);
        this.counts[indexFrom] = 0;
    }

    public void incrementValue(int value) {
        this.incrementCount(value, 1);
        ++this.sumCount;
    }

    public void incrementValueBy(int value, int increase) {
        this.incrementCount(value, increase);
        this.sumCount += (long)increase;
    }

    public void incrementValueBy(int value, long increase) {
        this.incrementValueBy(value, Math.toIntExact(increase));
    }

    public void removeBelowThreshold(int threshold) {
        for (int bin = this.minValue; bin < threshold; ++bin) {
            this.zeroValue(bin);
        }
        this.chopBefore(this.index(threshold));
        this.minValue = threshold;
    }

    public boolean isEmpty() {
        return this.sumCount == 0L;
    }

    public int getCount(int value) {
        return this.counts[this.index(value)];
    }

    public int size() {
        return this.counts.length;
    }

    public void addHistogram(Histogram other) throws OperationFailedException {
        if (this.getMaxValue() != other.getMaxValue()) {
            throw new OperationFailedException("Cannot add histograms with different max-bin-values");
        }
        if (this.minValue != other.minValue) {
            throw new OperationFailedException("Cannot add histograms with different min-bin-values");
        }
        for (int bin = this.minValue; bin <= this.getMaxValue(); ++bin) {
            int otherCount = other.getCount(bin);
            this.incrementCount(bin, otherCount);
            this.sumCount += (long)otherCount;
        }
    }

    public double mean() throws OperationFailedException {
        this.checkAtLeastOneItemExists();
        long sum = 0L;
        for (int bin = this.minValue; bin <= this.maxValue; ++bin) {
            sum += this.getCountAsLong(bin) * (long)bin;
        }
        return (double)sum / (double)this.sumCount;
    }

    public int quantile(double quantile) throws OperationFailedException {
        this.checkAtLeastOneItemExists();
        if (quantile < 0.0 || quantile > 1.0) {
            throw new OperationFailedException(String.format("The quantile must be >= 0 and <= 1 but is %f", quantile));
        }
        double threshold = quantile * (double)this.sumCount;
        long sum = 0L;
        for (int bin = this.minValue; bin <= this.maxValue; ++bin) {
            if (!((double)(sum += (long)this.getCount(bin)) > threshold)) continue;
            return bin;
        }
        return this.calculateMaximum();
    }

    public boolean hasNonZeroCount(int threshold) {
        for (int bin = threshold; bin <= this.maxValue; ++bin) {
            if (this.getCount(bin) <= 0) continue;
            return true;
        }
        return false;
    }

    public int calculateMode() throws OperationFailedException {
        this.checkAtLeastOneItemExists();
        int maxIndex = -1;
        int maxCount = -1;
        for (int bin = this.minValue; bin <= this.maxValue; ++bin) {
            int count = this.getCount(bin);
            if (count <= maxCount) continue;
            maxCount = count;
            maxIndex = bin;
        }
        return maxIndex;
    }

    public int calculateMaximum() throws OperationFailedException {
        this.checkAtLeastOneItemExists();
        for (int bin = this.maxValue; bin >= this.minValue; --bin) {
            if (this.getCount(bin) <= 0) continue;
            return bin;
        }
        throw new AnchorImpossibleSituationException();
    }

    public int calculateMinimum() throws OperationFailedException {
        this.checkAtLeastOneItemExists();
        for (int bin = this.minValue; bin <= this.maxValue; ++bin) {
            if (this.getCount(bin) <= 0) continue;
            return bin;
        }
        throw new AnchorImpossibleSituationException();
    }

    public long calculateSum() {
        return this.calculateSumHelper(value -> value);
    }

    public long calculateSumSquares() {
        return this.calculateSumHelper(value -> value * value);
    }

    public long calculateSumCubes() {
        return this.calculateSumHelper(value -> value * value * value);
    }

    public double standardDeviation() throws OperationFailedException {
        this.checkAtLeastOneItemExists();
        return Math.sqrt(this.variance());
    }

    public double variance() throws OperationFailedException {
        this.checkAtLeastOneItemExists();
        return new VarianceCalculatorLong(this.calculateSum(), this.calculateSumSquares(), this.getTotalCount()).variance();
    }

    public long countMatching(IntPredicate predicate) {
        long sum = 0L;
        for (int bin = this.minValue; bin <= this.maxValue; ++bin) {
            if (!predicate.test(bin)) continue;
            sum += this.getCountAsLong(bin);
        }
        return sum;
    }

    public Histogram threshold(DoublePredicate predicate) {
        Histogram out = new Histogram(this.maxValue);
        out.sumCount = 0L;
        for (int bin = this.minValue; bin <= this.maxValue; ++bin) {
            if (!predicate.test(bin)) continue;
            int count = this.getCount(bin);
            out.set(bin, count);
            out.sumCount += (long)count;
        }
        return out;
    }

    public String toString() {
        return this.concatenateForEachBin(value -> String.format("%d: %d%n", value, this.getCount(value)));
    }

    public long getTotalCount() {
        return this.sumCount;
    }

    public Histogram cropRemoveSmallerValues(long maxCount) {
        int count;
        Histogram out = new Histogram(this.maxValue);
        long remaining = maxCount;
        for (int bin = this.getMaxValue(); bin >= this.minValue && ((count = this.getCount(bin)) == 0 || (remaining = Histogram.extractBin(out, bin, count, remaining)) != 0L); --bin) {
        }
        return out;
    }

    public Histogram cropRemoveLargerValues(long maxCount) {
        int count;
        Histogram out = new Histogram(this.maxValue);
        long remaining = maxCount;
        for (int bin = this.minValue; bin <= this.getMaxValue() && ((count = this.getCount(bin)) == 0 || (remaining = Histogram.extractBin(out, bin, count, remaining)) != 0L); ++bin) {
        }
        return out;
    }

    public double mean(double power) throws OperationFailedException {
        this.checkAtLeastOneItemExists();
        return this.mean(power, 0.0);
    }

    public double mean(double power, double subtractValue) throws OperationFailedException {
        this.checkAtLeastOneItemExists();
        double sum = 0.0;
        for (int bin = this.minValue; bin <= this.maxValue; ++bin) {
            double binSubtracted = (double)bin - subtractValue;
            sum += (double)this.getCountAsLong(bin) * Math.pow(binSubtracted, power);
        }
        return sum / (double)this.sumCount;
    }

    public void iterateValues(BinConsumer consumer) {
        for (int bin = this.minValue; bin <= this.maxValue; ++bin) {
            consumer.accept(bin, this.getCount(bin));
        }
    }

    public void iterateValuesUntil(int limit, BinConsumer consumer) {
        for (int bin = this.minValue; bin <= limit; ++bin) {
            consumer.accept(bin, this.getCount(bin));
        }
    }

    private long calculateSumHelper(LongUnaryOperator function) {
        long sum = 0L;
        for (int bin = this.minValue; bin <= this.maxValue; ++bin) {
            long add = this.getCountAsLong(bin) * function.applyAsLong(bin);
            sum += add;
        }
        return sum;
    }

    private int index(int value) {
        return value - this.minValue;
    }

    private void set(int value, int countToAssign) {
        this.counts[this.index((int)value)] = countToAssign;
    }

    private void incrementCount(int value, int incrementBy) {
        int n = this.index(value);
        this.counts[n] = this.counts[n] + incrementBy;
    }

    private long getCountAsLong(int value) {
        return this.getCount(value);
    }

    private void chopBefore(int index) {
        this.counts = Arrays.copyOfRange(this.counts, index, this.counts.length);
    }

    private void checkAtLeastOneItemExists() throws OperationFailedException {
        if (this.isEmpty()) {
            throw new OperationFailedException("There are no items in the histogram so this operation cannot occur");
        }
    }

    private static long extractBin(Histogram destination, int bin, int countForBin, long remaining) {
        if (remaining >= (long)countForBin) {
            destination.incrementValueBy(bin, countForBin);
            return remaining - (long)countForBin;
        }
        destination.incrementValueBy(bin, remaining);
        return 0L;
    }

    private String concatenateForEachBin(IntFunction<String> stringForBin) {
        StringBuilder builder = new StringBuilder();
        for (int bin = this.minValue; bin <= this.maxValue; ++bin) {
            builder.append(stringForBin.apply(bin));
        }
        return builder.toString();
    }

    @Generated
    public int getMaxValue() {
        return this.maxValue;
    }

    @FunctionalInterface
    public static interface BinConsumer {
        public void accept(int var1, int var2);
    }
}

