/*-
 * #%L
 * anchor-plugin-image-feature
 * %%
 * Copyright (C) 2010 - 2020 Owen Feehan, ETH Zurich, University of Zurich, Hoffmann-La Roche
 * %%
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * #L%
 */

package org.anchoranalysis.plugin.image.feature.bean.shared.object;

import lombok.Getter;
import lombok.Setter;
import org.anchoranalysis.bean.annotation.BeanField;
import org.anchoranalysis.core.exception.InitializeException;
import org.anchoranalysis.feature.bean.Feature;
import org.anchoranalysis.feature.calculate.FeatureCalculationException;
import org.anchoranalysis.feature.calculate.FeatureCalculationInput;
import org.anchoranalysis.feature.calculate.cache.CalculateForChild;
import org.anchoranalysis.feature.initialization.FeatureInitialization;
import org.anchoranalysis.feature.input.FeatureInput;
import org.anchoranalysis.feature.input.FeatureInputEnergy;
import org.anchoranalysis.image.bean.nonbean.init.ImageInitialization;
import org.anchoranalysis.image.feature.bean.FeatureEnergy;
import org.anchoranalysis.image.feature.input.FeatureInputSingleObject;

/**
 * Calculates object-masks from entities in shared, using the feature-input only for an
 * energy-stack.
 *
 * <p>This abstract class extends {@link FeatureEnergy} to provide a base for features that operate
 * on single objects from shared data.
 *
 * @author Owen Feehan
 * @param <T> feature-input type that extends {@link FeatureInputEnergy}
 */
public abstract class FeatureSingleObjectFromShared<T extends FeatureInputEnergy>
        extends FeatureEnergy<T> {

    // START BEAN PROPERTIES
    /** The feature to be calculated on a single object. */
    @BeanField @Getter @Setter private Feature<FeatureInputSingleObject> item;
    // END BEAN PROPERTIES

    @Override
    protected void beforeCalc(FeatureInitialization initialization) throws InitializeException {
        super.beforeCalc(initialization);
        beforeCalcWithInitialization(
                new ImageInitialization(initialization.sharedObjectsRequired()));
    }

    /**
     * Performs initialization before calculation with image-specific initialization.
     *
     * @param initialization the image-specific initialization
     * @throws InitializeException if initialization fails
     */
    protected abstract void beforeCalcWithInitialization(ImageInitialization initialization)
            throws InitializeException;

    @Override
    public double calculate(FeatureCalculationInput<T> input) throws FeatureCalculationException {
        return calc(input.forChild(), item);
    }

    /**
     * Calculates the feature value for a single object.
     *
     * @param calculateForChild the calculation context for child features
     * @param featureForSingleObject the feature to calculate on a single object
     * @return the calculated feature value
     * @throws FeatureCalculationException if the calculation fails
     */
    protected abstract double calc(
            CalculateForChild<T> calculateForChild,
            Feature<FeatureInputSingleObject> featureForSingleObject)
            throws FeatureCalculationException;

    @Override
    public Class<? extends FeatureInput> inputType() {
        return FeatureInputEnergy.class;
    }
}
