/*-
 * #%L
 * anchor-plugin-image
 * %%
 * Copyright (C) 2010 - 2020 Owen Feehan, ETH Zurich, University of Zurich, Hoffmann-La Roche
 * %%
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * #L%
 */

package org.anchoranalysis.plugin.image.bean.object.filter.independent;

import java.util.Optional;
import lombok.Getter;
import lombok.Setter;
import org.anchoranalysis.bean.annotation.BeanField;
import org.anchoranalysis.core.exception.OperationFailedException;
import org.anchoranalysis.image.core.dimensions.Dimensions;
import org.anchoranalysis.image.voxel.object.ObjectCollection;
import org.anchoranalysis.image.voxel.object.ObjectMask;
import org.anchoranalysis.plugin.image.bean.object.filter.ObjectFilterPredicate;
import org.anchoranalysis.spatial.point.ReadableTuple3i;

/**
 * Keeps only objects that are not adjacent to the scene-border (i.e. have a bounding-box on the
 * very edge of the image)
 *
 * @author Owen Feehan
 */
public class NotTouchingSceneBorder extends ObjectFilterPredicate {

    // START BEAN PROPERTIES
    @BeanField @Getter @Setter private boolean includeZ = false;

    // END BEAN PROPERTIES

    @Override
    protected boolean precondition(ObjectCollection objectsToFilter) {
        return true;
    }

    @Override
    protected boolean match(ObjectMask object, Optional<Dimensions> dimensions)
            throws OperationFailedException {

        if (!dimensions.isPresent()) {
            throw new OperationFailedException("Image-dimensions are required for this operation");
        }

        if (object.boundingBox().atBorderXY(dimensions.get().extent())) {
            return false;
        }

        if (includeZ) {
            ReadableTuple3i cornerMin = object.boundingBox().cornerMin();
            if (cornerMin.z() == 0) {
                return false;
            }

            ReadableTuple3i cornerMax = object.boundingBox().calculateCornerMaxInclusive();
            if (cornerMax.z() == (dimensions.get().z() - 1)) {
                return false;
            }
        }
        return true;
    }

    @Override
    protected void end() throws OperationFailedException {
        // NOTHING TO DO
    }
}
