/*-
 * #%L
 * anchor-test-image
 * %%
 * Copyright (C) 2010 - 2020 Owen Feehan, ETH Zurich, University of Zurich, Hoffmann-La Roche
 * %%
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * #L%
 */
package org.anchoranalysis.test.image;

import io.vavr.control.Either;
import java.nio.file.Path;
import java.util.List;
import java.util.Optional;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import org.anchoranalysis.core.exception.CreateException;
import org.anchoranalysis.core.exception.friendly.AnchorFriendlyRuntimeException;
import org.anchoranalysis.core.index.SetOperationFailedException;
import org.anchoranalysis.image.bean.displayer.IntensityQuantiles;
import org.anchoranalysis.image.bean.displayer.StackDisplayer;
import org.anchoranalysis.image.core.channel.Channel;
import org.anchoranalysis.image.core.channel.factory.ChannelFactory;
import org.anchoranalysis.image.core.dimensions.Dimensions;
import org.anchoranalysis.image.core.object.properties.ObjectCollectionWithProperties;
import org.anchoranalysis.image.core.stack.DisplayStack;
import org.anchoranalysis.image.core.stack.Stack;
import org.anchoranalysis.image.io.object.output.mask.ObjectAsMaskGenerator;
import org.anchoranalysis.image.io.object.output.rgb.DrawObjectsGenerator;
import org.anchoranalysis.image.io.stack.output.generator.DisplayStackGenerator;
import org.anchoranalysis.image.voxel.Voxels;
import org.anchoranalysis.image.voxel.buffer.primitive.UnsignedByteBuffer;
import org.anchoranalysis.image.voxel.object.ObjectCollection;
import org.anchoranalysis.image.voxel.object.ObjectMask;
import org.anchoranalysis.io.generator.collection.CollectionGenerator;
import org.anchoranalysis.io.output.error.OutputWriteFailedException;
import org.anchoranalysis.io.output.outputter.BindFailedException;
import org.anchoranalysis.io.output.outputter.Outputter;
import org.anchoranalysis.spatial.box.BoundedList;
import org.anchoranalysis.spatial.box.BoundingBox;
import org.anchoranalysis.test.image.io.OutputterFixture;

/**
 * Writes one or more stacks/objects/channels into a directory during testing.
 *
 * <p>Any checked-exceptions thrown during writing stacks are converted into run-time exceptions, to
 * make it easy to temporarily use this class in a test for debugging with minimal alteration of
 * functions.
 *
 * @author Owen Feehan
 */
@RequiredArgsConstructor
public class WriteIntoDirectory {

    private static final StackDisplayer DISPLAYER = new IntensityQuantiles();

    /**
     * If there are no objects or specified dimensions, this size is used for an output image as a
     * fallback
     */
    private static final Dimensions FALLBACK_SIZE = new Dimensions(100, 100, 1);

    // START REQUIRED ARGUMENTS
    /**
     * The directory in which stacks and other outputs are written.
     *
     * <p>This directory is used as the root location for all file outputs generated by this class.
     * It should be a valid, writable directory path.
     *
     * <p>If {@link #printDirectoryToConsole} is set to true, this directory path will be printed to
     * the console when the first output is written.
     *
     * @see #printDirectoryToConsole
     */
    @Getter private final Path directory;

    /** If true, the path of {@code folder} is printed to the console */
    private final boolean printDirectoryToConsole;
    // END REQUIRED ARGUMENTS

    /**
     * Creates a new WriteIntoDirectory instance with directory printing enabled.
     *
     * <p>This constructor sets {@link #printDirectoryToConsole} to true, meaning the directory path
     * will be printed to the console when the first output is written.
     *
     * @param directory the directory in which stacks and other outputs will be written. It should
     *     be a valid, writable directory path.
     */
    public WriteIntoDirectory(Path directory) {
        this.printDirectoryToConsole = true;
        this.directory = directory;
    }

    private Outputter outputter;

    private DisplayStackGenerator generatorStack = new DisplayStackGenerator(false);

    private ObjectAsMaskGenerator generatorSingleObject = new ObjectAsMaskGenerator();

    /**
     * Writes a DisplayStack to the output directory.
     *
     * @param outputName The name to use for the output file.
     * @param stack The DisplayStack to write.
     */
    public void writeStack(String outputName, DisplayStack stack) {
        setupOutputterIfNecessary();
        outputter.writerPermissive().write(outputName, () -> generatorStack, () -> stack);
    }

    /**
     * Writes an ObjectMask to the output directory.
     *
     * @param outputName The name to use for the output file.
     * @param object The ObjectMask to write.
     * @throws SetOperationFailedException If the operation fails.
     */
    public void writeObject(String outputName, ObjectMask object)
            throws SetOperationFailedException {
        setupOutputterIfNecessary();
        outputter.writerPermissive().write(outputName, () -> generatorSingleObject, () -> object);
    }

    /**
     * Writes the outline of objects on a blank RGB image, inferring dimensions of the image to
     * center the object.
     *
     * @param outputName output-name
     * @param objects the objects to draw an outline for
     */
    public void writeObjects(String outputName, ObjectCollection objects) {

        Dimensions dimensionsResolved = dimensionsToCenterObjects(objects);

        writeObjectsEither(outputName, objects, Either.left(dimensionsResolved));
    }

    /**
     * Writes the outline of objects on a background.
     *
     * @param outputName output-name
     * @param objects the objects to draw an outline for
     * @param background the background
     */
    public void writeObjects(String outputName, ObjectCollection objects, Stack background) {
        writeObjectsEither(outputName, objects, Either.right(displayStackFor(background)));
    }

    /**
     * Writes Voxels to the output directory.
     *
     * @param outputName The name to use for the output file.
     * @param voxels The Voxels to write.
     */
    public void writeVoxels(String outputName, Voxels<UnsignedByteBuffer> voxels) {

        Channel channel = ChannelFactory.instance().create(voxels);

        writeChannel(outputName, channel);
    }

    /**
     * Writes a Channel to the output directory.
     *
     * @param outputName The name to use for the output file.
     * @param channel The Channel to write.
     */
    public void writeChannel(String outputName, Channel channel) {

        setupOutputterIfNecessary();

        writeStack(outputName, displayStackFor(channel));
    }

    /**
     * Writes a list of display-stacks.
     *
     * @param outputName the output-name
     * @param stacks the list of display-stacks
     * @param always2D if true, the stacks are guaranteed to always to have only one z-slice (which
     *     can influence the output format).
     * @throws OutputWriteFailedException if the stacks cannot be successfully written to the
     *     file-system.
     */
    public void writeList(String outputName, List<DisplayStack> stacks, boolean always2D)
            throws OutputWriteFailedException {

        setupOutputterIfNecessary();
        outputter
                .getChecked()
                .getWriters()
                .permissive()
                .write(
                        outputName,
                        () -> new CollectionGenerator<>(generatorStack, outputName),
                        () -> stacks);
    }

    private static DisplayStack displayStackFor(Channel channel) {
        try {
            return DISPLAYER.deriveFrom(channel);
        } catch (CreateException e) {
            throw new AnchorFriendlyRuntimeException(e);
        }
    }

    private static DisplayStack displayStackFor(Stack stack) {
        try {
            return DISPLAYER.deriveFrom(stack);
        } catch (CreateException e) {
            throw new AnchorFriendlyRuntimeException(e);
        }
    }

    private void setupOutputterIfNecessary() {
        try {
            if (outputter == null) {

                outputter = OutputterFixture.outputter(Optional.of(directory));

                if (printDirectoryToConsole) {
                    System.out.println("Outputs written in test to: " + directory); // NOSONAR
                }
            }
        } catch (BindFailedException e) {
            throw new AnchorFriendlyRuntimeException(e);
        }
    }

    /** Writes objects with either dimensions (for a blank background) or a particular background */
    private void writeObjectsEither(
            String outputName,
            ObjectCollection objects,
            Either<Dimensions, DisplayStack> background) {

        setupOutputterIfNecessary();

        DrawObjectsGenerator generatorObjects =
                DrawObjectsGenerator.outlineVariedColors(objects.size(), 1, background);
        outputter
                .writerPermissive()
                .write(
                        outputName,
                        () -> generatorObjects,
                        () -> new ObjectCollectionWithProperties(objects));
    }

    /** Finds dimensions that place the objects in the center */
    private static Dimensions dimensionsToCenterObjects(ObjectCollection objects) {

        if (objects.size() == 0) {
            return FALLBACK_SIZE;
        }

        BoundingBox boxSpans = boundingBoxThatSpans(objects);

        BoundingBox boxCentered =
                boxSpans.changeExtent(boxSpans.extent().growBy(boxSpans.cornerMin()));

        return new Dimensions(boxCentered.calculateCornerMaxExclusive());
    }

    private static BoundingBox boundingBoxThatSpans(ObjectCollection objects) {
        return BoundedList.createFromList(objects.asList(), ObjectMask::boundingBox).boundingBox();
    }
}
