package org.antipathy.scoozie

import org.antipathy.scoozie.properties.JobProperties
import org.antipathy.scoozie.coordinator.Coordinator
import org.antipathy.scoozie.io.ArtefactWriter

/**
  * Trait that may be extended by clients of the library
  */
trait GeneratedCoordinator extends JobProperties {
  this: GeneratedWorkflow =>

  /**
    * Get the coordinator generated by this class
    */
  def coordinator: Coordinator

  /**
    * Get the job properties for this coordinator and its workflow
    */
  override def jobProperties: String = coordinator.jobProperties + workflow.jobProperties

  /**
    * Writes this coordinator, workflow and properties to the specified folder.
    * The folder will be created if it does not exist
    *
    * file names on output will be `coordinator.xml`, `workflow.xml` and `job.properties`
    *
    * @param folderLocation the location to write to
    */
  final def saveCoordinator(folderLocation: String): Unit = {
    Scoozie.Test.validate(this.workflow)
    val formattedCoordinator = Scoozie.Formatting.format(this.coordinator)
    this.saveWorkflow(folderLocation)
    writeFile(folderLocation, ArtefactWriter.coordinatorFileName, formattedCoordinator)
    saveJobProperties(folderLocation)
  }

  /**
    * Writes this workflow's properties to the specified folder.
    * The folder will be created if it does not exist
    *
    * file name on output will `job.properties`
    *
    * @param folderLocation the location to write to
    */
  private def saveJobProperties(folderLocation: String): Unit =
    this.writeFile(folderLocation, ArtefactWriter.propertiesFileName, this.jobProperties)
}
