package org.antipathy.scoozie.action

import org.antipathy.scoozie.action.prepare.Prepare
import org.antipathy.scoozie.configuration._
import scala.xml.Elem
import scala.collection.immutable._
import com.typesafe.config.Config
import org.antipathy.scoozie.builder.{ConfigurationBuilder, PrepareBuilder}
import scala.collection.JavaConverters._
import com.typesafe.config.ConfigException
import org.antipathy.scoozie.exception.ConfigurationMissingException

/**
  * Oozie Spark action definition
  * @param name the name of the action
  * @param sparkMasterURL the url of the spark master
  * @param sparkMode the mode the spark job should run in
  * @param sparkJobName the name of the spark job
  * @param mainClass the main class of the spark job
  * @param sparkJar the location of the spark jar
  * @param sparkOptions options for the spark job
  * @param commandLineArgs command line arguments for the spark job
  * @param jobXmlOption optional job.xml path
  * @param prepareOption an optional prepare phase for the action
  * @param configuration additional config for this action
  * @param yarnConfig Yarn configuration for this action
  */
final class SparkAction(override val name: String,
                        sparkMasterURL: String,
                        sparkMode: String,
                        sparkJobName: String,
                        mainClass: String,
                        sparkJar: String,
                        sparkOptions: String,
                        commandLineArgs: Seq[String],
                        files: Seq[String],
                        jobXmlOption: Option[String],
                        prepareOption: Option[Prepare],
                        configuration: Configuration,
                        yarnConfig: YarnConfig)
    extends Action {

  private val sparkMasterURLProperty = formatProperty(s"${name}_sparkMasterURL")
  private val sparkModeProperty = formatProperty(s"${name}_sparkMode")
  private val sparkJobNameProperty = formatProperty(s"${name}_sparkJobName")
  private val mainClassProperty = formatProperty(s"${name}_mainClass")
  private val sparkJarProperty = formatProperty(s"${name}_sparkJar")
  private val sparkOptionsProperty = formatProperty(s"${name}_sparkOptions")
  private val commandLineArgsProperties =
    buildSequenceProperties(name, "commandLineArg", commandLineArgs)
  private val filesProperties = buildSequenceProperties(name, "files", files)
  private val jobXmlProperty =
    buildStringOptionProperty(name, "jobXml", jobXmlOption)

  private val prepareOptionAndProps =
    prepareOption.map(_.withActionProperties(name))
  private val prepareProperties =
    prepareOptionAndProps.map(_._2).getOrElse(Map[String, String]())
  private val prepareOptionMapped = prepareOptionAndProps.map(_._1)
  private val mappedConfigAndProperties = configuration.withActionProperties(name)
  private val mappedConfig = mappedConfigAndProperties._1

  /**
    * The XML namespace for an action element
    */
  override val xmlns: Option[String] = Some("uri:oozie:spark-action:0.1")

  /**
    * Get the Oozie properties for this object
    */
  override def properties: Map[String, String] =
    Map(sparkMasterURLProperty -> sparkMasterURL,
        sparkModeProperty -> sparkMode,
        sparkJobNameProperty -> sparkJobName,
        mainClassProperty -> mainClass,
        sparkJarProperty -> sparkJar,
        sparkOptionsProperty -> sparkOptions) ++
    commandLineArgsProperties ++
    prepareProperties ++
    filesProperties ++
    mappedConfigAndProperties._2 ++
    jobXmlProperty

  /**
    * The XML for this node
    */
  override def toXML: Elem =
    <spark xmlns={xmlns.orNull}>
      {yarnConfig.jobTrackerXML}
      {yarnConfig.nameNodeXML}
      {if (prepareOptionMapped.isDefined) {
          prepareOptionMapped.get.toXML
        }
      }
      {if (jobXmlOption.isDefined) {
          <job-xml>{jobXmlProperty.keys}</job-xml>
        }
      }
      {if (mappedConfig.configProperties.nonEmpty) {
         mappedConfig.toXML
        }
      }
      <master>{sparkMasterURLProperty}</master>
      <mode>{sparkModeProperty}</mode>
      <name>{sparkJobNameProperty}</name>
      <class>{mainClassProperty}</class>
      <jar>{sparkJarProperty}</jar>
      <spark-opts>{sparkOptionsProperty}</spark-opts>
      {commandLineArgsProperties.keys.map(Arg(_).toXML)}
      {filesProperties.keys.map(File(_).toXML)}
    </spark>
}

/**
  * Companion object
  */
object SparkAction {

  /**
    * Create a new instance of this action
    */
  def apply(name: String,
            sparkMasterURL: String,
            sparkMode: String,
            sparkJobName: String,
            mainClass: String,
            sparkJar: String,
            sparkOptions: String,
            commandLineArgs: Seq[String],
            files: Seq[String],
            jobXmlOption: Option[String],
            prepareOption: Option[Prepare],
            configuration: Configuration,
            yarnConfig: YarnConfig)(implicit credentialsOption: Option[Credentials]): Node =
    Node(
      new SparkAction(name,
                      sparkMasterURL,
                      sparkMode,
                      sparkJobName,
                      mainClass,
                      sparkJar,
                      sparkOptions,
                      commandLineArgs,
                      files,
                      jobXmlOption,
                      prepareOption,
                      configuration,
                      yarnConfig)
    )

  /**
    * Create a new instance of this action from a configuration
    */
  def apply(config: Config, yarnConfig: YarnConfig)(implicit credentials: Option[Credentials]): Node =
    try {
      SparkAction(name = config.getString("name"),
                  sparkMasterURL = config.getString("spark-master-url"),
                  sparkMode = config.getString("spark-mode"),
                  sparkJobName = config.getString("spark-job-name"),
                  mainClass = config.getString("main-class"),
                  sparkJar = config.getString("spark-jar"),
                  sparkOptions = config.getString("spark-options"),
                  commandLineArgs = Seq(config.getStringList("command-line-arguments").asScala: _*),
                  files = Seq(config.getStringList("files").asScala: _*),
                  jobXmlOption = if (config.hasPath("job-xml")) {
                    Some(config.getString("job-xml"))
                  } else None,
                  configuration = ConfigurationBuilder.buildConfiguration(config),
                  yarnConfig = yarnConfig,
                  prepareOption = PrepareBuilder.build(config))
    } catch {
      case c: ConfigException =>
        throw new ConfigurationMissingException(s"${c.getMessage} in ${config.getString("name")}")
    }
}
