package org.antipathy.scoozie

import org.antipathy.scoozie.io.ArtefactWriter
import org.antipathy.scoozie.properties.JobProperties
import org.antipathy.scoozie.workflow.Workflow

/**
  * Trait that may be extended by clients of the library
  */
trait ScoozieWorkflow extends JobProperties with ArtefactWriter {

  /**
    * Get the workflow generated by this class
    */
  def workflow: Workflow

  /**
    * Get the job properties for this workflow
    */
  override def jobProperties: String = this.workflow.jobProperties

  /**
    * Writes this workflow and its properties to the specified folder.
    * The folder will be created if it does not exist
    *
    * file names on output will be `workflow.xml` and `job.properties`
    *
    * @param folderLocation the location to write to
    */
  final def saveWorkflow(folderLocation: String): Unit = {
    Scoozie.Test.validate(this.workflow)
    val formattedWorkflow = Scoozie.Formatting.format(this.workflow)
    writeFile(folderLocation, ArtefactWriter.workflowFileName, formattedWorkflow)
    saveJobProperties(folderLocation)
  }

  /**
    * Writes this workflow's properties to the specified folder.
    * The folder will be created if it does not exist
    *
    * file name on output will `job.properties`
    *
    * @param folderLocation the location to write to
    */
  private def saveJobProperties(folderLocation: String): Unit =
    this.writeFile(folderLocation, ArtefactWriter.propertiesFileName, this.jobProperties)

}
