/*
 * Copyright 2022-2023 Douglas Hoard
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.antublue.map.accessor;

import java.io.IOException;
import java.io.Reader;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.TreeMap;
import java.util.regex.Pattern;

/**
 * Class to wrap a SnakeYML Map with to allow access via a path
 */
@SuppressWarnings({"unchecked", "PMD.GodClass"})
public final class MapAccessor {

    private final Map<String, Object> map;

    /**
     * Constructor
     *
     * @param map
     */
    private MapAccessor(Map<String, Object> map) {
        this.map = map;
    }

    /**
     * Method to get whether the map is empty
     *
     * @return
     */
    public boolean isEmpty() {
        return map.isEmpty();
    }

    /**
     * Method to get the internal Map
     *
     * @return
     */
    public Map<String, Object> getMap() {
        return map;
    }

    /**
     * Method to set a path value
     *
     * @param path
     * @param value
     * @return
     */
    public MapAccessor put(String path, boolean value) {
        putObject(path, value);
        return this;
    }

    /**
     * Method to set a path value
     *
     * @param path
     * @param value
     * @return
     */
    public MapAccessor put(String path, short value) {
        putObject(path, (int) value);
        return this;
    }
    
    /**
     * Method to set a path value
     *
     * @param path
     * @param value
     * @return
     */
    public MapAccessor put(String path, int value) {
        putObject(path, value);
        return this;
    }

    /**
     * Method to set a path value
     *
     * @param path
     * @param value
     * @return
     */
    public MapAccessor put(String path, float value) {
        putObject(path, value);
        return this;
    }

    /**
     * Method to set a path value
     *
     * @param path
     * @param value
     * @return
     */
    public MapAccessor put(String path, String value) {
        putObject(path, value);
        return this;
    }

    /**
     * Method to set a path value
     *
     * @param path
     * @param value
     * @return
     */
    public MapAccessor put(String path, Map<String, Object> value) {
        putObject(path, value);
        return this;
    }

    /**
     * Method to set a path value
     *
     * @param path
     * @param value
     * @return
     */
    public MapAccessor put(String path, List<Object> value) {
        putObject(path, value);
        return this;
    }

    /**
     * Method to get a path value
     *
     * @param path
     * @return
     */
    public Optional<Object> get(String path) {
        if (path == null) {
            throw new IllegalArgumentException("Path is null");
        }

        String pathTrimmed = path.trim();

        if (pathTrimmed.isEmpty()) {
            throw new IllegalArgumentException("Path is empty");
        }

        if ("/".equals(pathTrimmed)) {
            return Optional.of((Object) map);
        }

        if (!pathTrimmed.startsWith("/") || pathTrimmed.endsWith("/")) {
            throw new IllegalArgumentException(String.format("Path [%s] is invalid", pathTrimmed));
        }

        Object object = null;
        String[] keys = pathTrimmed.split(Pattern.quote("/"));
        Map<String, Object> map = this.map;
        for (int i = 1; i < keys.length; i++) {
            String key = keys[i];
            object = map.get(key);
            if (object == null) {
                return Optional.ofNullable(null);
            } else if (object instanceof Map) {
                map = (Map<String, Object>) object;
            } else {
                if (i < (keys.length - 1)) {
                    throw new UnsupportedOperationException(
                            String.format(
                                    "Path [%s] type [%s] can't be converted to Map", pathTrimmed, object.getClass()));
                }
            }
        }

        return Optional.of(object);
    }

    /**
     * Method to delete a path value
     *
     * @param path
     */
    public Optional<Object> delete(String path) {
        if (path == null) {
            throw new IllegalArgumentException("Path is null");
        }

        String pathTrimmed = path.trim();

        if ("/".equals(pathTrimmed)) {
            throw new IllegalArgumentException(String.format("Path [%s] is invalid", pathTrimmed));
        }

        if (!pathTrimmed.startsWith("/") || pathTrimmed.endsWith("/")) {
            throw new IllegalArgumentException(String.format("Path [%s] is invalid", pathTrimmed));
        }

        String key = pathTrimmed.substring(pathTrimmed.lastIndexOf("/") + 1);
        pathTrimmed = pathTrimmed.substring(0, pathTrimmed.indexOf(key));
        if (!"/".equals(pathTrimmed) && pathTrimmed.endsWith("/")) {
            pathTrimmed = pathTrimmed.substring(0, pathTrimmed.length() - 1);
        }

        Object object = get(pathTrimmed).orElse(new LinkedHashMap());

        if (!(object instanceof Map)) {
            throw new UnsupportedOperationException(
                    String.format(
                            "Path [%s] type [%s] can't be converted to a Map", pathTrimmed, object.getClass()));
        }

        return Optional.of(object);
    }

    /**
     * Method to determine if a path contains a value, equivalent to get(path).isPresent()
     *
     * @param path
     * @return
     */
    public boolean containsPath(String path) {
        return get(path).isPresent();
    }

    /**
     * Method to get an instance of a MapAccessor
     *
     * @param map
     * @return
     */
    public static MapAccessor of(Map<String, Object> map) {
        if (map == null) {
            throw new IllegalArgumentException("map is null");
        }

        Map<String, Object> treeMap = map;

        if (!(map instanceof TreeMap)) {
            treeMap = new TreeMap<String, Object>(map);
        }

        return new MapAccessor(treeMap);
    }

    /**
     * Method to get an instance of MapAccessor from YAML
     *
     * @param yaml
     * @return
     * @throws IOException
     */
    /*
    public static MapAccessor of(String yaml) throws IOException {
        try {
            return new MapAccessor((Map<String, Object>) new Yaml(new FilteringSafeConstructor()).load(yaml));
        } catch (IllegalStateException e) {
            throw new IOException(e);
        } catch (Throwable t) {
            throw new IOException(t);
        }
    }

    /**
     * Method to get an instance of a MapAccessor from a Reader
     *
     * @param reader
     * @return
     * @throws IOException
     */
    /*
    public static MapAccessor of(Reader reader) throws IOException {
        String yaml = toString(reader);
        return of(yaml);
    }
    */

    /**
     * Method to set a path value
     *
     * @param path
     * @param value
     */
    private void putObject(String path, Object value) {
        if (!path.startsWith("/") || path.endsWith("/")) {
            throw new IllegalArgumentException(String.format("Path [%s] is invalid", path));
        }

        Object object;
        String[] keys = path.split(Pattern.quote("/"));
        Map<String, Object> map = this.map;

        for (int i = 1; i < keys.length - 1; i++) {
            String key = keys[i];
            object = map.get(key);
            if (object == null) {
                Map<String, Object> childMap = new LinkedHashMap<String, Object>();
                map.put(key, childMap);
                map = childMap;
            } else if (object instanceof Map) {
                map = (Map<String, Object>) object;
            } else {
                if (i < (keys.length - 1)) {
                    throw new UnsupportedOperationException(
                            String.format(
                                    "Path [%s] type [%s] can't be converted to a Map", path, object.getClass()));
                }
            }
        }

        map.put(keys[keys.length - 1], value);
    }

    private static String toString(Reader reader) throws IOException {
        char[] buffer = new char[1024];
        StringBuilder stringBuilder = new StringBuilder();

        while (true) {
            int count = reader.read(buffer, 0, buffer.length);
            if (count == -1) {
                break;
            }

            stringBuilder.append(buffer, 0, count);
        }

        return new String(buffer.toString().getBytes("UTF-8"));
    }

}