/*
 * Copyright 2023 Douglas Hoard
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.antublue.map.accessor.function;

import java.lang.reflect.Constructor;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.function.Function;
import java.util.regex.Pattern;

@SuppressWarnings("PMD.PreserveStackTrace")
public final class ToBoolean implements Function<Object, Boolean> {

    public static final ToBoolean INSTANCE = new ToBoolean();

    private final Set<String> TRUE_SET;
    private final Set<String> FALSE_SET;

    private Class<? extends RuntimeException> clazz;
    private String message;

    private ToBoolean() {
        TRUE_SET = new HashSet<String>();
        FALSE_SET = new HashSet<String>();

        String trueString = "y|Y|yes|Yes|YES|true|True|TRUE|on|On|ON";
        String falseString = "n|N|no|No|NO|false|False|FALSE|off|Off|OFF";

        String[] tokens = trueString.split(Pattern.quote("|"));
        Collections.addAll(TRUE_SET, tokens);

        tokens = falseString.split(Pattern.quote("|"));
        Collections.addAll(FALSE_SET, tokens);
    }

    public ToBoolean(Class<? extends RuntimeException> clazz, String message) {
        this();

        this.clazz = clazz;
        this.message = message;
    }

    @Override
    public Boolean apply(Object value) {
        if (value == null) {
            return null;
        }

        if (value instanceof Boolean) {
            return (Boolean) value;
        }

        String string = value.toString();

        if (TRUE_SET.contains(string)) {
            return true;
        } else if (FALSE_SET.contains(string)) {
            return false;
        }

        if (clazz != null) {
            RuntimeException runtimeException;

            try {
                Constructor<?> constructor = clazz.getConstructor(String.class);
                runtimeException = (RuntimeException) constructor.newInstance(message);
            } catch (Throwable t) {
                throw new RuntimeException(t);
            }

            throw runtimeException;
        }

        throw new UnsupportedOperationException(
                String.format("Unsupported conversion [%s] to Boolean", string));
    }
}
