/*
 * Copyright 2023 Douglas Hoard
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.antublue.map.accessor.function;

import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.function.Function;

@SuppressWarnings("PMD.PreserveStackTrace")
public final class ToDate implements Function<Object, Date> {

    public static final ToDate INSTANCE = new ToDate();

    private static final List<DateTimeFormatter> DATE_TIME_FORMATTER_LIST;

    static {
        DATE_TIME_FORMATTER_LIST = new ArrayList<DateTimeFormatter>();
        DATE_TIME_FORMATTER_LIST.add(DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ssSSSZ"));
	    DATE_TIME_FORMATTER_LIST.add(DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ssZ"));
        DATE_TIME_FORMATTER_LIST.add(DateTimeFormat.forPattern("yyyyMMdd'T'HHmmssSSSZ"));
        DATE_TIME_FORMATTER_LIST.add(DateTimeFormat.forPattern("yyyyMMdd'T'HHmmssZ"));
    }	

    private Class<? extends RuntimeException> clazz;
    private String message;

    private ToDate() {
        // DO NOTHING
    }

    public ToDate(Class<? extends RuntimeException> clazz, String message) {
        this();

        this.clazz = clazz;
        this.message = message;
    }

    @Override
    public Date apply(Object value) {
        if (value == null) {
            return null;
        }

        if (value instanceof Date) {
            return (Date) value;
        }

        if (value instanceof Long) {
            Long longValue = (Long) value;
            if (longValue < 0) {
                throw new UnsupportedOperationException(
                        String.format("Unsupported conversion [%s] to Date", value));
            }

            return new Date(longValue);
        }

        String stringValue = value.toString();

        // Convert to String, then Long, then Date
        try {
            Long longValue = Long.parseLong(stringValue);
            if (longValue < 0) {
                throw new UnsupportedOperationException(
                        String.format("Unsupported conversion [%s] to Date", value));
            }

            return new Date(longValue);
        } catch (NumberFormatException e) {
            // DO NOTHING
        }


        // Try the various DateTimeFormatter
        for (DateTimeFormatter dateTimeFormatter : DATE_TIME_FORMATTER_LIST) {
            try {
                return dateTimeFormatter.parseDateTime(stringValue).toDate();
            } catch (Throwable t) {
                // DO NOTHING
            }
        }

        if (clazz != null) {
            RuntimeException runtimeException;

            try {
                Constructor<?> constructor = clazz.getConstructor(String.class);
                runtimeException = (RuntimeException) constructor.newInstance(message);
            } catch (Throwable t) {
                throw new RuntimeException(t);
            }

            throw runtimeException;
        }

        throw new UnsupportedOperationException(
                String.format("Unsupported conversion [%s] to Date", value));
    }
}
