/*
 * Decompiled with CFR 0.152.
 */
package org.aoju.bus.health.linux.software;

import com.sun.jna.Native;
import com.sun.jna.platform.linux.LibC;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import org.aoju.bus.core.annotation.ThreadSafe;
import org.aoju.bus.core.lang.RegEx;
import org.aoju.bus.core.lang.tuple.Pair;
import org.aoju.bus.core.lang.tuple.Triple;
import org.aoju.bus.core.toolkit.FileKit;
import org.aoju.bus.health.Builder;
import org.aoju.bus.health.Executor;
import org.aoju.bus.health.builtin.software.AbstractOperatingSystem;
import org.aoju.bus.health.builtin.software.FileSystem;
import org.aoju.bus.health.builtin.software.InternetProtocolStats;
import org.aoju.bus.health.builtin.software.NetworkParams;
import org.aoju.bus.health.builtin.software.OSProcess;
import org.aoju.bus.health.builtin.software.OSService;
import org.aoju.bus.health.builtin.software.OperatingSystem;
import org.aoju.bus.health.linux.LinuxLibc;
import org.aoju.bus.health.linux.ProcPath;
import org.aoju.bus.health.linux.drivers.CpuStat;
import org.aoju.bus.health.linux.drivers.ProcessStat;
import org.aoju.bus.health.linux.drivers.UpTime;
import org.aoju.bus.health.linux.software.LinuxFileSystem;
import org.aoju.bus.health.linux.software.LinuxInternetProtocolStats;
import org.aoju.bus.health.linux.software.LinuxNetworkParams;
import org.aoju.bus.health.linux.software.LinuxUserGroupInfo;
import org.aoju.bus.logger.Logger;

@ThreadSafe
public class LinuxOperatingSystem
extends AbstractOperatingSystem {
    private static final String OS_RELEASE_LOG = "os-release: {}";
    private static final String LSB_RELEASE_A_LOG = "lsb_release -a: {}";
    private static final String LSB_RELEASE_LOG = "lsb-release: {}";
    private static final String RELEASE_DELIM = " release ";
    private static final String DOUBLE_QUOTES = "^\"|\"$";
    private static final String LS_F_PROC_PID_FD = "ls -f " + ProcPath.PID_FD;
    private static final long BOOTTIME;
    private static final int[] PROC_PID_STAT_ORDERS;
    private static final int PROC_PID_STAT_LENGTH;
    private static final long USER_HZ;
    private static final long BOOT_TIME;
    private final int memoryPageSize = Builder.parseIntOrDefault(Executor.getFirstAnswer("getconf PAGESIZE"), 4096);

    private static int getParentPidFromProcFile(int pid) {
        String stat = Builder.getStringFromFile(String.format("/proc/%d/stat", pid));
        if (stat.isEmpty()) {
            return 0;
        }
        long[] statArray = Builder.parseStringToLongArray(stat, PROC_PID_STAT_ORDERS, PROC_PID_STAT_LENGTH, ' ');
        return (int)statArray[ProcPidStat.PPID.ordinal()];
    }

    private static Triple<String, String, String> queryFamilyVersionCodenameFromReleaseFiles() {
        Triple<String, String, String> familyVersionCodename = LinuxOperatingSystem.readDistribRelease("/etc/system-release");
        if (familyVersionCodename != null) {
            return familyVersionCodename;
        }
        familyVersionCodename = LinuxOperatingSystem.readOsRelease();
        if (familyVersionCodename != null) {
            return familyVersionCodename;
        }
        familyVersionCodename = LinuxOperatingSystem.execLsbRelease();
        if (familyVersionCodename != null) {
            return familyVersionCodename;
        }
        familyVersionCodename = LinuxOperatingSystem.readLsbRelease();
        if (familyVersionCodename != null) {
            return familyVersionCodename;
        }
        String etcDistribRelease = LinuxOperatingSystem.getReleaseFilename();
        familyVersionCodename = LinuxOperatingSystem.readDistribRelease(etcDistribRelease);
        if (familyVersionCodename != null) {
            return familyVersionCodename;
        }
        String family = LinuxOperatingSystem.filenameToFamily(etcDistribRelease.replace("/etc/", "").replace("release", "").replace("version", "").replace("-", "").replace("_", ""));
        return Triple.of(family, "unknown", "unknown");
    }

    private static Triple<String, String, String> readOsRelease() {
        String family = null;
        String versionId = "unknown";
        String codeName = "unknown";
        List<String> osRelease = FileKit.readLines("/etc/os-release");
        for (String line : osRelease) {
            if (line.startsWith("VERSION=")) {
                Logger.debug(OS_RELEASE_LOG, line);
                line = line.replace("VERSION=", "").replaceAll(DOUBLE_QUOTES, "").trim();
                String[] split = line.split("[()]");
                if (split.length <= 1) {
                    split = line.split(", ");
                }
                if (split.length > 0) {
                    versionId = split[0].trim();
                }
                if (split.length <= 1) continue;
                codeName = split[1].trim();
                continue;
            }
            if (line.startsWith("NAME=") && family == null) {
                Logger.debug(OS_RELEASE_LOG, line);
                family = line.replace("NAME=", "").replaceAll(DOUBLE_QUOTES, "").trim();
                continue;
            }
            if (!line.startsWith("VERSION_ID=") || !versionId.equals("unknown")) continue;
            Logger.debug(OS_RELEASE_LOG, line);
            versionId = line.replace("VERSION_ID=", "").replaceAll(DOUBLE_QUOTES, "").trim();
        }
        return family == null ? null : Triple.of(family, versionId, codeName);
    }

    private static Triple<String, String, String> execLsbRelease() {
        String family = null;
        String versionId = "unknown";
        String codeName = "unknown";
        for (String line : Executor.runNative("lsb_release -a")) {
            if (line.startsWith("Description:")) {
                Logger.debug(LSB_RELEASE_A_LOG, line);
                if (!(line = line.replace("Description:", "").trim()).contains(RELEASE_DELIM)) continue;
                Triple<String, String, String> Triple2 = LinuxOperatingSystem.parseRelease(line, RELEASE_DELIM);
                family = Triple2.getLeft();
                if (versionId.equals("unknown")) {
                    versionId = Triple2.getMiddle();
                }
                if (!codeName.equals("unknown")) continue;
                codeName = Triple2.getRight();
                continue;
            }
            if (line.startsWith("Distributor ID:") && family == null) {
                Logger.debug(LSB_RELEASE_A_LOG, line);
                family = line.replace("Distributor ID:", "").trim();
                continue;
            }
            if (line.startsWith("Release:") && versionId.equals("unknown")) {
                Logger.debug(LSB_RELEASE_A_LOG, line);
                versionId = line.replace("Release:", "").trim();
                continue;
            }
            if (!line.startsWith("Codename:") || !codeName.equals("unknown")) continue;
            Logger.debug(LSB_RELEASE_A_LOG, line);
            codeName = line.replace("Codename:", "").trim();
        }
        return family == null ? null : Triple.of(family, versionId, codeName);
    }

    private static Triple<String, String, String> readLsbRelease() {
        String family = null;
        String versionId = "unknown";
        String codeName = "unknown";
        List<String> osRelease = FileKit.readLines("/etc/lsb-release");
        for (String line : osRelease) {
            if (line.startsWith("DISTRIB_DESCRIPTION=")) {
                Logger.debug(LSB_RELEASE_LOG, line);
                if (!(line = line.replace("DISTRIB_DESCRIPTION=", "").replaceAll(DOUBLE_QUOTES, "").trim()).contains(RELEASE_DELIM)) continue;
                Triple<String, String, String> Triple2 = LinuxOperatingSystem.parseRelease(line, RELEASE_DELIM);
                family = Triple2.getLeft();
                if (versionId.equals("unknown")) {
                    versionId = Triple2.getMiddle();
                }
                if (!codeName.equals("unknown")) continue;
                codeName = Triple2.getRight();
                continue;
            }
            if (line.startsWith("DISTRIB_ID=") && family == null) {
                Logger.debug(LSB_RELEASE_LOG, line);
                family = line.replace("DISTRIB_ID=", "").replaceAll(DOUBLE_QUOTES, "").trim();
                continue;
            }
            if (line.startsWith("DISTRIB_RELEASE=") && versionId.equals("unknown")) {
                Logger.debug(LSB_RELEASE_LOG, line);
                versionId = line.replace("DISTRIB_RELEASE=", "").replaceAll(DOUBLE_QUOTES, "").trim();
                continue;
            }
            if (!line.startsWith("DISTRIB_CODENAME=") || !codeName.equals("unknown")) continue;
            Logger.debug(LSB_RELEASE_LOG, line);
            codeName = line.replace("DISTRIB_CODENAME=", "").replaceAll(DOUBLE_QUOTES, "").trim();
        }
        return family == null ? null : Triple.of(family, versionId, codeName);
    }

    private static Triple<String, String, String> readDistribRelease(String filename) {
        if (new File(filename).exists()) {
            List<String> osRelease = FileKit.readLines(filename);
            for (String line : osRelease) {
                Logger.debug("{}: {}", filename, line);
                if (line.contains(RELEASE_DELIM)) {
                    return LinuxOperatingSystem.parseRelease(line, RELEASE_DELIM);
                }
                if (!line.contains(" VERSION ")) continue;
                return LinuxOperatingSystem.parseRelease(line, " VERSION ");
            }
        }
        return null;
    }

    private static Triple<String, String, String> parseRelease(String line, String splitLine) {
        String[] split = line.split(splitLine);
        String family = split[0].trim();
        String versionId = "unknown";
        String codeName = "unknown";
        if (split.length > 1) {
            if ((split = split[1].split("[()]")).length > 0) {
                versionId = split[0].trim();
            }
            if (split.length > 1) {
                codeName = split[1].trim();
            }
        }
        return Triple.of(family, versionId, codeName);
    }

    protected static String getReleaseFilename() {
        File etc = new File("/etc");
        File[] matchingFiles = etc.listFiles(f -> (f.getName().endsWith("-release") || f.getName().endsWith("-version") || f.getName().endsWith("_release") || f.getName().endsWith("_version")) && !f.getName().endsWith("os-release") && !f.getName().endsWith("lsb-release") && !f.getName().endsWith("system-release"));
        if (matchingFiles != null && matchingFiles.length > 0) {
            return matchingFiles[0].getPath();
        }
        if (new File("/etc/release").exists()) {
            return "/etc/release";
        }
        return "/etc/issue";
    }

    private static String filenameToFamily(String name) {
        switch (name.toLowerCase()) {
            case "": {
                return "Solaris";
            }
            case "blackcat": {
                return "Black Cat";
            }
            case "bluewhite64": {
                return "BlueWhite64";
            }
            case "e-smith": {
                return "SME Server";
            }
            case "eos": {
                return "FreeEOS";
            }
            case "hlfs": {
                return "HLFS";
            }
            case "lfs": {
                return "Linux-From-Scratch";
            }
            case "linuxppc": {
                return "Linux-PPC";
            }
            case "meego": {
                return "MeeGo";
            }
            case "mandakelinux": {
                return "Mandrake";
            }
            case "mklinux": {
                return "MkLinux";
            }
            case "nld": {
                return "Novell Linux Desktop";
            }
            case "novell": 
            case "SuSE": {
                return "SUSE Linux";
            }
            case "pld": {
                return "PLD";
            }
            case "redhat": {
                return "Red Hat Linux";
            }
            case "sles": {
                return "SUSE Linux ES9";
            }
            case "sun": {
                return "Sun JDS";
            }
            case "synoinfo": {
                return "Synology";
            }
            case "tinysofa": {
                return "Tiny Sofa";
            }
            case "turbolinux": {
                return "TurboLinux";
            }
            case "ultrapenguin": {
                return "UltraPenguin";
            }
            case "va": {
                return "VA-Linux";
            }
            case "vmware": {
                return "VMWareESX";
            }
            case "yellowdog": {
                return "Yellow Dog";
            }
            case "issue": {
                return "Unknown";
            }
        }
        return name.substring(0, 1).toUpperCase() + name.substring(1);
    }

    public static long getHz() {
        return USER_HZ;
    }

    @Override
    public String queryManufacturer() {
        return "GNU/Linux";
    }

    @Override
    public AbstractOperatingSystem.FamilyVersionInfo queryFamilyVersionInfo() {
        Triple<String, String, String> familyVersionCodename = LinuxOperatingSystem.queryFamilyVersionCodenameFromReleaseFiles();
        String buildNumber = null;
        List<String> procVersion = FileKit.readLines(ProcPath.VERSION);
        if (!procVersion.isEmpty()) {
            String[] split;
            for (String s : split = RegEx.SPACES.split(procVersion.get(0))) {
                if ("Linux".equals(s) || "version".equals(s)) continue;
                buildNumber = s;
                break;
            }
        }
        OperatingSystem.OSVersionInfo versionInfo = new OperatingSystem.OSVersionInfo(familyVersionCodename.getMiddle(), familyVersionCodename.getRight(), buildNumber);
        return new AbstractOperatingSystem.FamilyVersionInfo(familyVersionCodename.getLeft(), versionInfo);
    }

    @Override
    protected int queryBitness(int jvmBitness) {
        if (jvmBitness < 64 && Executor.getFirstAnswer("uname -m").indexOf("64") == -1) {
            return jvmBitness;
        }
        return 64;
    }

    @Override
    protected boolean queryElevated() {
        return System.getenv("SUDO_COMMAND") != null;
    }

    @Override
    public FileSystem getFileSystem() {
        return new LinuxFileSystem();
    }

    @Override
    public InternetProtocolStats getInternetProtocolStats() {
        return new LinuxInternetProtocolStats();
    }

    @Override
    public OSProcess[] getProcesses(int limit, OperatingSystem.ProcessSort sort, boolean slowFields) {
        ArrayList<OSProcess> procs = new ArrayList<OSProcess>();
        File[] pids = ProcessStat.getPidFiles();
        LinuxUserGroupInfo userGroupInfo = new LinuxUserGroupInfo();
        for (File pidFile : pids) {
            int pid = Builder.parseIntOrDefault(pidFile.getName(), 0);
            OSProcess proc = this.getProcess(pid, userGroupInfo, slowFields);
            if (proc == null) continue;
            procs.add(proc);
        }
        List<OSProcess> sorted = this.processSort(procs, limit, sort);
        return sorted.toArray(new OSProcess[0]);
    }

    @Override
    public OSProcess getProcess(int pid, boolean slowFields) {
        return this.getProcess(pid, new LinuxUserGroupInfo(), slowFields);
    }

    private OSProcess getProcess(int pid, LinuxUserGroupInfo userGroupInfo, boolean slowFields) {
        String path = "";
        String procPidExe = String.format(ProcPath.PID_EXE, pid);
        try {
            Path link = Paths.get(procPidExe, new String[0]);
            path = Files.readSymbolicLink(link).toString();
            int index = path.indexOf(" (deleted)");
            if (index != -1) {
                path = path.substring(0, index);
            }
        }
        catch (IOException | SecurityException | UnsupportedOperationException | InvalidPathException e) {
            Logger.debug("Unable to open symbolic link {}", procPidExe);
        }
        Map<String, String> io = Builder.getKeyValueMapFromFile(String.format(ProcPath.PID_IO, pid), ":");
        long now = System.currentTimeMillis();
        String stat = Builder.getStringFromFile(String.format(ProcPath.PID_STAT, pid));
        if (stat.isEmpty()) {
            return null;
        }
        long[] statArray = Builder.parseStringToLongArray(stat, PROC_PID_STAT_ORDERS, PROC_PID_STAT_LENGTH, ' ');
        OSProcess proc = new OSProcess(this);
        proc.setProcessID(pid);
        proc.setCommandLine(Builder.getStringFromFile(String.format(ProcPath.PID_CMDLINE, pid)));
        long startTime = BOOT_TIME + statArray[ProcPidStat.START_TIME.ordinal()] * 1000L / USER_HZ;
        if (startTime >= now) {
            startTime = now - 1L;
        }
        proc.setStartTime(startTime);
        proc.setParentProcessID((int)statArray[ProcPidStat.PPID.ordinal()]);
        proc.setThreadCount((int)statArray[ProcPidStat.THREAD_COUNT.ordinal()]);
        proc.setPriority((int)statArray[ProcPidStat.PRIORITY.ordinal()]);
        proc.setVirtualSize(statArray[ProcPidStat.VSZ.ordinal()]);
        proc.setResidentSetSize(statArray[ProcPidStat.RSS.ordinal()] * (long)this.memoryPageSize);
        proc.setKernelTime(statArray[ProcPidStat.KERNEL_TIME.ordinal()] * 1000L / USER_HZ);
        proc.setUserTime(statArray[ProcPidStat.USER_TIME.ordinal()] * 1000L / USER_HZ);
        proc.setUpTime(now - proc.getStartTime());
        proc.setBytesRead(Builder.parseLongOrDefault(io.getOrDefault("read_bytes", ""), 0L));
        proc.setBytesWritten(Builder.parseLongOrDefault(io.getOrDefault("write_bytes", ""), 0L));
        if (slowFields) {
            List<String> openFilesList = Executor.runNative(String.format(LS_F_PROC_PID_FD, pid));
            proc.setOpenFiles((long)openFilesList.size() - 1L);
            byte[] buffer = new byte[5];
            if (!path.isEmpty()) {
                try (FileInputStream is = new FileInputStream(path);){
                    if (((InputStream)is).read(buffer) == buffer.length) {
                        proc.setBitness(buffer[4] == 1 ? 32 : 64);
                    }
                }
                catch (IOException e) {
                    Logger.warn("Failed to read process file: {}", path);
                }
            }
        }
        Map<String, String> status = Builder.getKeyValueMapFromFile(String.format(ProcPath.PID_STATUS, pid), ":");
        proc.setName(status.getOrDefault("Name", ""));
        proc.setPath(path);
        switch (status.getOrDefault("State", "U").charAt(0)) {
            case 'R': {
                proc.setState(OSProcess.State.RUNNING);
                break;
            }
            case 'S': {
                proc.setState(OSProcess.State.SLEEPING);
                break;
            }
            case 'D': {
                proc.setState(OSProcess.State.WAITING);
                break;
            }
            case 'Z': {
                proc.setState(OSProcess.State.ZOMBIE);
                break;
            }
            case 'T': {
                proc.setState(OSProcess.State.STOPPED);
                break;
            }
            default: {
                proc.setState(OSProcess.State.OTHER);
            }
        }
        proc.setUserID(RegEx.SPACES.split(status.getOrDefault("Uid", ""))[0]);
        proc.setGroupID(RegEx.SPACES.split(status.getOrDefault("Gid", ""))[0]);
        Pair<String, String> user = userGroupInfo.getUser(proc.getUserID());
        proc.setUser(user.getRight());
        proc.setGroup(userGroupInfo.getGroupName(proc.getGroupID()));
        try {
            String cwdLink = String.format(ProcPath.PID_CWD, pid);
            String cwd = new File(cwdLink).getCanonicalPath();
            if (!cwd.equals(cwdLink)) {
                proc.setCurrentWorkingDirectory(cwd);
            }
        }
        catch (IOException e) {
            Logger.trace("Couldn't find cwd for pid {}: {}", pid, e.getMessage());
        }
        return proc;
    }

    @Override
    public OSProcess[] getChildProcesses(int parentPid, int limit, OperatingSystem.ProcessSort sort) {
        ArrayList<OSProcess> procs = new ArrayList<OSProcess>();
        File[] procFiles = ProcessStat.getPidFiles();
        LinuxUserGroupInfo userGroupInfo = new LinuxUserGroupInfo();
        for (File procFile : procFiles) {
            OSProcess proc;
            int pid = Builder.parseIntOrDefault(procFile.getName(), 0);
            if (parentPid != LinuxOperatingSystem.getParentPidFromProcFile(pid) || (proc = this.getProcess(pid, userGroupInfo, true)) == null) continue;
            procs.add(proc);
        }
        List<OSProcess> sorted = this.processSort(procs, limit, sort);
        return sorted.toArray(new OSProcess[0]);
    }

    @Override
    public long getProcessAffinityMask(int processId) {
        String mask = Executor.getFirstAnswer("taskset -p " + processId);
        String[] split = RegEx.SPACES.split(mask);
        try {
            return new BigInteger(split[split.length - 1], 16).longValue();
        }
        catch (NumberFormatException e) {
            return 0L;
        }
    }

    @Override
    public int getProcessId() {
        return LinuxLibc.INSTANCE.getpid();
    }

    @Override
    public int getProcessCount() {
        return ProcessStat.getPidFiles().length;
    }

    @Override
    public int getThreadCount() {
        try {
            LibC.Sysinfo info = new LibC.Sysinfo();
            if (0 != LibC.INSTANCE.sysinfo(info)) {
                Logger.error("Failed to get process thread count. Error code: {}", Native.getLastError());
                return 0;
            }
            return info.procs;
        }
        catch (NoClassDefFoundError | UnsatisfiedLinkError e) {
            Logger.error("Failed to get procs from sysinfo. {}", e.getMessage());
            return 0;
        }
    }

    @Override
    public long getSystemUptime() {
        return (long)UpTime.getSystemUptimeSeconds();
    }

    @Override
    public long getSystemBootTime() {
        return BOOTTIME;
    }

    @Override
    public NetworkParams getNetworkParams() {
        return new LinuxNetworkParams();
    }

    @Override
    public OSService[] getServices() {
        ArrayList<OSService> services = new ArrayList<OSService>();
        HashSet<String> running = new HashSet<String>();
        for (OSProcess p : this.getChildProcesses(1, 0, OperatingSystem.ProcessSort.PID)) {
            OSService s = new OSService(p.getName(), p.getProcessID(), OSService.State.RUNNING);
            services.add(s);
            running.add(p.getName());
        }
        boolean systemctlFound = false;
        List<String> systemctl = Executor.runNative("systemctl list-unit-files");
        for (String str : systemctl) {
            String shortName;
            String[] split = RegEx.SPACES.split(str);
            if (split.length != 2 || !split[0].endsWith(".service") || !"enabled".equals(split[1])) continue;
            String name2 = split[0].substring(0, split[0].length() - 8);
            int index = name2.lastIndexOf(46);
            String string = shortName = index < 0 || index > name2.length() - 2 ? name2 : name2.substring(index + 1);
            if (running.contains(name2) || running.contains(shortName)) continue;
            OSService s = new OSService(name2, 0, OSService.State.STOPPED);
            services.add(s);
            systemctlFound = true;
        }
        if (!systemctlFound) {
            File dir = new File("/etc/init");
            if (dir.exists() && dir.isDirectory()) {
                for (File f2 : dir.listFiles((f, name) -> name.toLowerCase().endsWith(".conf"))) {
                    String shortName;
                    String name3 = f2.getName().substring(0, f2.getName().length() - 5);
                    int index = name3.lastIndexOf(46);
                    String string = shortName = index < 0 || index > name3.length() - 2 ? name3 : name3.substring(index + 1);
                    if (running.contains(name3) || running.contains(shortName)) continue;
                    OSService s = new OSService(name3, 0, OSService.State.STOPPED);
                    services.add(s);
                }
            } else {
                Logger.error("Directory: /etc/init does not exist", new Object[0]);
            }
        }
        return services.toArray(new OSService[0]);
    }

    static {
        PROC_PID_STAT_ORDERS = new int[ProcPidStat.values().length];
        USER_HZ = Builder.parseLongOrDefault(Executor.getFirstAnswer("getconf CLK_TCK"), 100L);
        long tempBT = CpuStat.getBootTime();
        if (tempBT == 0L) {
            tempBT = System.currentTimeMillis() / 1000L - (long)UpTime.getSystemUptimeSeconds();
        }
        BOOTTIME = tempBT;
        for (ProcPidStat stat : ProcPidStat.values()) {
            LinuxOperatingSystem.PROC_PID_STAT_ORDERS[stat.ordinal()] = stat.getOrder() - 1;
        }
        String stat = Builder.getStringFromFile(ProcPath.SELF_STAT);
        PROC_PID_STAT_LENGTH = !stat.isEmpty() && stat.contains(")") ? Builder.countStringToLongArray(stat, ' ') + 3 : 52;
        double uptime = UpTime.getSystemUptimeSeconds();
        long now = System.currentTimeMillis();
        BOOT_TIME = now - (long)(500.0 * (uptime += UpTime.getSystemUptimeSeconds()) + 0.5);
    }

    static enum ProcPidStat {
        PPID(4),
        USER_TIME(14),
        KERNEL_TIME(15),
        PRIORITY(18),
        THREAD_COUNT(20),
        START_TIME(22),
        VSZ(23),
        RSS(24);

        private int order;

        private ProcPidStat(int order) {
            this.order = order;
        }

        public int getOrder() {
            return this.order;
        }
    }
}

