/*********************************************************************************
 *                                                                               *
 * The MIT License (MIT)                                                         *
 *                                                                               *
 * Copyright (c) 2015-2020 aoju.org and other contributors.                      *
 *                                                                               *
 * Permission is hereby granted, free of charge, to any person obtaining a copy  *
 * of this software and associated documentation files (the "Software"), to deal *
 * in the Software without restriction, including without limitation the rights  *
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     *
 * copies of the Software, and to permit persons to whom the Software is         *
 * furnished to do so, subject to the following conditions:                      *
 *                                                                               *
 * The above copyright notice and this permission notice shall be included in    *
 * all copies or substantial portions of the Software.                           *
 *                                                                               *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   *
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, *
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     *
 * THE SOFTWARE.                                                                 *
 *                                                                               *
 ********************************************************************************/
package org.aoju.bus.office.support.excel;

import org.aoju.bus.core.toolkit.StringKit;
import org.apache.poi.ss.usermodel.*;

/**
 * Excel样式工具类
 *
 * @author Kimi Liu
 * @version 6.1.6
 * @since JDK 1.8+
 */
public class StyleKit {

    /**
     * 克隆新的{@link CellStyle}
     *
     * @param cell      单元格
     * @param cellStyle 被复制的样式
     * @return {@link CellStyle}
     */
    public static CellStyle cloneCellStyle(Cell cell, CellStyle cellStyle) {
        return cloneCellStyle(cell.getSheet().getWorkbook(), cellStyle);
    }

    /**
     * 克隆新的{@link CellStyle}
     *
     * @param workbook  工作簿
     * @param cellStyle 被复制的样式
     * @return {@link CellStyle}
     */
    public static CellStyle cloneCellStyle(Workbook workbook, CellStyle cellStyle) {
        final CellStyle newCellStyle = createCellStyle(workbook);
        newCellStyle.cloneStyleFrom(cellStyle);
        return newCellStyle;
    }

    /**
     * 设置cell文本对齐样式
     *
     * @param cellStyle {@link CellStyle}
     * @param halign    横向位置
     * @param valign    纵向位置
     * @return {@link CellStyle}
     */
    public static CellStyle setAlign(CellStyle cellStyle, HorizontalAlignment halign, VerticalAlignment valign) {
        cellStyle.setAlignment(halign);
        cellStyle.setVerticalAlignment(valign);
        return cellStyle;
    }

    /**
     * 设置cell的四个边框粗细和颜色
     *
     * @param cellStyle  {@link CellStyle}
     * @param borderSize 边框粗细{@link BorderStyle}枚举
     * @param colorIndex 颜色的short值
     * @return {@link CellStyle}
     */
    public static CellStyle setBorder(CellStyle cellStyle, BorderStyle borderSize, IndexedColors colorIndex) {
        cellStyle.setBorderBottom(borderSize);
        cellStyle.setBottomBorderColor(colorIndex.index);

        cellStyle.setBorderLeft(borderSize);
        cellStyle.setLeftBorderColor(colorIndex.index);

        cellStyle.setBorderRight(borderSize);
        cellStyle.setRightBorderColor(colorIndex.index);

        cellStyle.setBorderTop(borderSize);
        cellStyle.setTopBorderColor(colorIndex.index);

        return cellStyle;
    }

    /**
     * 给cell设置颜色
     *
     * @param cellStyle   {@link CellStyle}
     * @param color       背景颜色
     * @param fillPattern 填充方式 {@link FillPatternType}枚举
     * @return {@link CellStyle}
     */
    public static CellStyle setColor(CellStyle cellStyle, IndexedColors color, FillPatternType fillPattern) {
        return setColor(cellStyle, color.index, fillPattern);
    }

    /**
     * 给cell设置颜色
     *
     * @param cellStyle   {@link CellStyle}
     * @param color       背景颜色
     * @param fillPattern 填充方式 {@link FillPatternType}枚举
     * @return {@link CellStyle}
     */
    public static CellStyle setColor(CellStyle cellStyle, short color, FillPatternType fillPattern) {
        cellStyle.setFillForegroundColor(color);
        cellStyle.setFillPattern(fillPattern);
        return cellStyle;
    }

    /**
     * 创建字体
     *
     * @param workbook {@link Workbook}
     * @param color    字体颜色
     * @param fontSize 字体大小
     * @param fontName 字体名称,可以为null使用默认字体
     * @return {@link Font}
     */
    public static Font createFont(Workbook workbook, short color, short fontSize, String fontName) {
        final Font font = workbook.createFont();
        return setFontStyle(font, color, fontSize, fontName);
    }

    /**
     * 设置字体样式
     *
     * @param font     字体{@link Font}
     * @param color    字体颜色
     * @param fontSize 字体大小
     * @param fontName 字体名称,可以为null使用默认字体
     * @return {@link Font}
     */
    public static Font setFontStyle(Font font, short color, short fontSize, String fontName) {
        if (color > 0) {
            font.setColor(color);
        }
        if (fontSize > 0) {
            font.setFontHeightInPoints(fontSize);
        }
        if (StringKit.isNotBlank(fontName)) {
            font.setFontName(fontName);
        }
        return font;
    }

    /**
     * 创建默认普通单元格样式
     *
     * <pre>
     * 1. 文字上下左右居中
     * 2. 细边框,黑色
     * </pre>
     *
     * @param workbook {@link Workbook} 工作簿
     * @return {@link CellStyle}
     */
    public static CellStyle createDefaultCellStyle(Workbook workbook) {
        final CellStyle cellStyle = createCellStyle(workbook);
        setAlign(cellStyle, HorizontalAlignment.CENTER, VerticalAlignment.CENTER);
        setBorder(cellStyle, BorderStyle.THIN, IndexedColors.BLACK);
        return cellStyle;
    }

    /**
     * 创建默认头部样式
     *
     * @param workbook {@link Workbook} 工作簿
     * @return {@link CellStyle}
     */
    public static CellStyle createHeadCellStyle(Workbook workbook) {
        final CellStyle cellStyle = createCellStyle(workbook);
        setAlign(cellStyle, HorizontalAlignment.CENTER, VerticalAlignment.CENTER);
        setBorder(cellStyle, BorderStyle.THIN, IndexedColors.BLACK);
        setColor(cellStyle, IndexedColors.GREY_25_PERCENT, FillPatternType.SOLID_FOREGROUND);
        return cellStyle;
    }

    /**
     * 创建单元格样式
     *
     * @param workbook {@link Workbook} 工作簿
     * @return {@link CellStyle}
     * @see Workbook#createCellStyle()
     */
    public static CellStyle createCellStyle(Workbook workbook) {
        if (null == workbook) {
            return null;
        }
        return workbook.createCellStyle();
    }

    /**
     * 创建数据格式并获取格式
     *
     * @param workbook 工作簿
     * @param format   数据格式
     * @return 数据格式
     */
    public Short getFormat(Workbook workbook, String format) {
        final DataFormat dataFormat = workbook.createDataFormat();
        return dataFormat.getFormat(format);
    }

}
