/*********************************************************************************
 *                                                                               *
 * The MIT License (MIT)                                                         *
 *                                                                               *
 * Copyright (c) 2015-2020 aoju.org and other contributors.                      *
 *                                                                               *
 * Permission is hereby granted, free of charge, to any person obtaining a copy  *
 * of this software and associated documentation files (the "Software"), to deal *
 * in the Software without restriction, including without limitation the rights  *
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     *
 * copies of the Software, and to permit persons to whom the Software is         *
 * furnished to do so, subject to the following conditions:                      *
 *                                                                               *
 * The above copyright notice and this permission notice shall be included in    *
 * all copies or substantial portions of the Software.                           *
 *                                                                               *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   *
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, *
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     *
 * THE SOFTWARE.                                                                 *
 *                                                                               *
 ********************************************************************************/
package org.aoju.bus.core.toolkit;

import org.aoju.bus.core.date.*;
import org.aoju.bus.core.lang.Fields;

import java.util.Calendar;
import java.util.Date;
import java.util.List;

/**
 * 时间工具类
 *
 * @author Kimi Liu
 * @version 6.1.8
 * @since JDK 1.8+
 */
public class DateKit extends Sandbox {

    /**
     * 计时,常用于记录某段代码的执行时间,单位：纳秒
     *
     * @param preTime 之前记录的时间
     * @return 时间差, 纳秒
     */
    public static long spendNt(long preTime) {
        return System.nanoTime() - preTime;
    }

    /**
     * 计时,常用于记录某段代码的执行时间,单位：毫秒
     *
     * @param preTime 之前记录的时间
     * @return 时间差, 毫秒
     */
    public static long spendMs(long preTime) {
        return System.currentTimeMillis() - preTime;
    }

    /**
     * 生日转为年龄，计算法定年龄
     *
     * @param birthDay 生日，标准日期字符串
     * @return 年龄
     */
    public static int ageOfNow(String birthDay) {
        return ageOfNow(parse(birthDay));
    }

    /**
     * 生日转为年龄，计算法定年龄
     *
     * @param birthDay 生日
     * @return 年龄
     */
    public static int ageOfNow(Date birthDay) {
        return getAge(birthDay, date());
    }

    /**
     * 返回文字描述的日期
     *
     * @param date 日期
     * @return 日期
     */
    public static String getLastTime(Date date) {
        if (date == null) {
            return null;
        }
        long diff = System.currentTimeMillis() - date.getTime();
        long r;
        if (diff > Fields.Units.WEEK.getUnit()) {
            r = (diff / Fields.Units.WEEK.getUnit());
            return r + "周前";
        }
        if (diff > Fields.Units.DAY.getUnit()) {
            r = (diff / Fields.Units.DAY.getUnit());
            return r + "天前";
        }
        if (diff > Fields.Units.HOUR.getUnit()) {
            r = (diff / Fields.Units.HOUR.getUnit());
            return r + "个小时前";
        }
        if (diff > Fields.Units.MINUTE.getUnit()) {
            r = (diff / Fields.Units.MINUTE.getUnit());
            return r + "分钟前";
        }
        return "刚刚";
    }

    /**
     * 创建日期范围生成器
     *
     * @param start 起始日期时间
     * @param end   结束日期时间
     * @param type  步进单位
     * @return {@link Boundary}
     */
    public static Boundary range(Date start, Date end, final Fields.Type type) {
        return new Boundary(start, end, type);
    }

    /**
     * 创建日期范围生成器
     *
     * @param start 起始日期时间
     * @param end   结束日期时间
     * @param type  步进单位
     * @return {@link Boundary}
     */
    public static List<DateTime> rangeToList(Date start, Date end, final Fields.Type type) {
        return CollKit.newArrayList((Iterable<DateTime>) range(start, end, type));
    }

    /**
     * 通过公历构造
     *
     * @return {@link Lunar}
     */
    public Lunar getLunar() {
        return new Lunar(date());
    }

    /**
     * 通过公历构造
     *
     * @param calendar 　公历日期
     * @return {@link Lunar}
     */
    public Lunar getLunar(Calendar calendar) {
        return new Lunar(calendar.get(Calendar.YEAR), calendar.get(Calendar.MONTH), calendar.get(Calendar.DAY_OF_MONTH),
                calendar.get(Calendar.HOUR_OF_DAY), calendar.get(Calendar.MINUTE), calendar.get(Calendar.SECOND));
    }

    /**
     * 通过年、月、日构造
     *
     * @param year  农历年
     * @param month 农历月份,范围1-12
     * @param day   农历日1-30
     * @return {@link Lunar}
     */
    public Lunar getLunar(int year, int month, int day) {
        return getLunar(year, month, day, 0, 0);
    }

    /**
     * 通过年、月、日构造
     *
     * @param year   农历年
     * @param month  农历月份,范围1-12
     * @param day    农历日1-30
     * @param hour   小时
     * @param minute 分钟
     * @return {@link Lunar}
     */
    public Lunar getLunar(int year, int month, int day, int hour, int minute) {
        return getLunar(year, month, day, hour, minute, 0);
    }

    /**
     * 通过年、月、日构造
     *
     * @param year   农历年
     * @param month  农历月份,范围1-12
     * @param day    农历日1-30
     * @param hour   小时
     * @param minute 分钟
     * @param second 秒
     * @return {@link Lunar}
     */
    public Lunar getLunar(int year, int month, int day, int hour, int minute, int second) {
        return new Lunar(year, month, day, hour, minute, second);
    }

    /**
     * 通过公历构造
     *
     * @return {@link Solar}
     */
    public Solar getSolar() {
        return new Solar(date());
    }

    /**
     * 通过公历构造
     *
     * @param calendar 　公历日期
     * @return {@link Solar}
     */
    public Solar getSolar(Calendar calendar) {
        return new Solar(calendar.get(Calendar.YEAR), calendar.get(Calendar.MONTH), calendar.get(Calendar.DAY_OF_MONTH),
                calendar.get(Calendar.HOUR_OF_DAY), calendar.get(Calendar.MINUTE), calendar.get(Calendar.SECOND));
    }

    /**
     * 通过年、月、日构造
     *
     * @param year  农历年
     * @param month 农历月份,范围1-12
     * @param day   农历日1-30
     * @return {@link Solar}
     */
    public Solar getSolar(int year, int month, int day) {
        return getSolar(year, month, day, 0, 0);
    }

    /**
     * 通过年、月、日构造
     *
     * @param year   农历年
     * @param month  农历月份,范围1-12
     * @param day    农历日1-30
     * @param hour   小时
     * @param minute 分钟
     * @return {@link Solar}
     */
    public Solar getSolar(int year, int month, int day, int hour, int minute) {
        return getSolar(year, month, day, hour, minute, 0);
    }

    /**
     * 通过年、月、日构造
     *
     * @param year   农历年
     * @param month  农历月份,范围1-12
     * @param day    农历日1-30
     * @param hour   小时
     * @param minute 分钟
     * @param second 秒
     * @return {@link Solar}
     */
    public Solar getSolar(int year, int month, int day, int hour, int minute, int second) {
        return new Solar(year, month, day, hour, minute, second);
    }

}
