/*
 * Decompiled with CFR 0.152.
 */
package org.aoju.bus.health.mac.hardware;

import com.sun.jna.Native;
import com.sun.jna.Structure;
import com.sun.jna.platform.mac.IOKit;
import com.sun.jna.platform.mac.IOKitUtil;
import com.sun.jna.platform.mac.SystemB;
import com.sun.jna.ptr.IntByReference;
import com.sun.jna.ptr.PointerByReference;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.function.Supplier;
import org.aoju.bus.core.annotation.ThreadSafe;
import org.aoju.bus.core.lang.tuple.Triple;
import org.aoju.bus.core.toolkit.StringKit;
import org.aoju.bus.health.Builder;
import org.aoju.bus.health.Memoize;
import org.aoju.bus.health.builtin.hardware.AbstractCentralProcessor;
import org.aoju.bus.health.builtin.hardware.CentralProcessor;
import org.aoju.bus.health.mac.SysctlKit;
import org.aoju.bus.logger.Logger;

@ThreadSafe
final class MacCentralProcessor
extends AbstractCentralProcessor {
    private static final int ROSETTA_CPUTYPE = 7;
    private static final int ROSETTA_CPUFAMILY = 1463508716;
    private static final int M1_CPUTYPE = 0x100000C;
    private static final int M1_CPUFAMILY = 458787763;
    private final Supplier<String> vendor = Memoize.memoize(MacCentralProcessor::platformExpert);
    private final Supplier<Triple<Integer, Integer, Long>> typeFamilyFreq = Memoize.memoize(MacCentralProcessor::queryArmCpu);

    MacCentralProcessor() {
    }

    private static String platformExpert() {
        String manufacturer = null;
        IOKit.IOService platformExpert = IOKitUtil.getMatchingService((String)"IOPlatformExpertDevice");
        if (null != platformExpert) {
            byte[] data = platformExpert.getByteArrayProperty("manufacturer");
            if (null != data) {
                manufacturer = Native.toString((byte[])data, (Charset)StandardCharsets.UTF_8);
            }
            platformExpert.release();
        }
        return StringKit.isBlank(manufacturer) ? "Apple Inc." : manufacturer;
    }

    private static Triple<Integer, Integer, Long> queryArmCpu() {
        int type = 7;
        int family = 1463508716;
        long freq = 0L;
        IOKit.IOIterator iter = IOKitUtil.getMatchingServices((String)"IOPlatformDevice");
        if (null != iter) {
            HashSet<String> compatibleStrSet = new HashSet<String>();
            IOKit.IORegistryEntry cpu = iter.next();
            while (null != cpu) {
                if (cpu.getName().startsWith("cpu")) {
                    long cpuFreq;
                    byte[] data = cpu.getByteArrayProperty("clock-frequency");
                    if (null != data && (cpuFreq = Builder.byteArrayToLong(data, data.length, false) * 1000L) > freq) {
                        freq = cpuFreq;
                    }
                    if (null != (data = cpu.getByteArrayProperty("compatible"))) {
                        for (String s : new String(data, StandardCharsets.UTF_8).split("\u0000")) {
                            if (s.isEmpty()) continue;
                            compatibleStrSet.add(s);
                        }
                    }
                }
                cpu.release();
                cpu = iter.next();
            }
            iter.release();
            List<String> m1compatible = Arrays.asList("ARM,v8", "apple,firestorm", "apple,icestorm");
            compatibleStrSet.retainAll(m1compatible);
            if (compatibleStrSet.size() == m1compatible.size()) {
                type = 0x100000C;
                family = 458787763;
            }
        }
        return Triple.of(type, family, freq);
    }

    @Override
    protected CentralProcessor.ProcessorIdentifier queryProcessorId() {
        String processorID;
        String cpuFamily;
        String cpuModel;
        String cpuStepping;
        String cpuVendor;
        String cpuName = SysctlKit.sysctl("machdep.cpu.brand_string", "");
        long cpuFreq = 0L;
        if (cpuName.startsWith("Apple")) {
            cpuVendor = this.vendor.get();
            cpuStepping = "0";
            cpuModel = "0";
            int type = SysctlKit.sysctl("hw.cputype", 0);
            int family = SysctlKit.sysctl("hw.cpufamily", 0);
            if (family == 1463508716) {
                type = this.typeFamilyFreq.get().getLeft();
                family = this.typeFamilyFreq.get().getMiddle();
            }
            cpuFreq = this.typeFamilyFreq.get().getRight();
            cpuFamily = String.format("0x%08x", family);
            processorID = String.format("%08x%08x", type, family);
        } else {
            cpuVendor = SysctlKit.sysctl("machdep.cpu.vendor", "");
            int i = SysctlKit.sysctl("machdep.cpu.stepping", -1);
            cpuStepping = i < 0 ? "" : Integer.toString(i);
            i = SysctlKit.sysctl("machdep.cpu.model", -1);
            cpuModel = i < 0 ? "" : Integer.toString(i);
            i = SysctlKit.sysctl("machdep.cpu.family", -1);
            cpuFamily = i < 0 ? "" : Integer.toString(i);
            long processorIdBits = 0L;
            processorIdBits |= (long)SysctlKit.sysctl("machdep.cpu.signature", 0);
            processorID = String.format("%016x", processorIdBits |= (SysctlKit.sysctl("machdep.cpu.feature_bits", 0L) & 0xFFFFFFFFFFFFFFFFL) << 32);
        }
        if (cpuFreq == 0L) {
            cpuFreq = SysctlKit.sysctl("hw.cpufrequency", 0L);
        }
        boolean cpu64bit = SysctlKit.sysctl("hw.cpu64bit_capable", 0) != 0;
        return new CentralProcessor.ProcessorIdentifier(cpuVendor, cpuName, cpuFamily, cpuModel, cpuStepping, processorID, cpu64bit, cpuFreq);
    }

    @Override
    protected List<CentralProcessor.LogicalProcessor> initProcessorCounts() {
        int logicalProcessorCount = SysctlKit.sysctl("hw.logicalcpu", 1);
        int physicalProcessorCount = SysctlKit.sysctl("hw.physicalcpu", 1);
        int physicalPackageCount = SysctlKit.sysctl("hw.packages", 1);
        ArrayList<CentralProcessor.LogicalProcessor> logProcs = new ArrayList<CentralProcessor.LogicalProcessor>(logicalProcessorCount);
        for (int i = 0; i < logicalProcessorCount; ++i) {
            logProcs.add(new CentralProcessor.LogicalProcessor(i, i * physicalProcessorCount / logicalProcessorCount, i * physicalPackageCount / logicalProcessorCount));
        }
        return logProcs;
    }

    @Override
    public long[] querySystemCpuLoadTicks() {
        SystemB.HostCpuLoadInfo cpuLoadInfo;
        long[] ticks = new long[CentralProcessor.TickType.values().length];
        int machPort = SystemB.INSTANCE.mach_host_self();
        if (0 != SystemB.INSTANCE.host_statistics(machPort, 3, (Structure)(cpuLoadInfo = new SystemB.HostCpuLoadInfo()), new IntByReference(cpuLoadInfo.size()))) {
            Logger.error("Failed to get System CPU ticks. Error code: {} ", Native.getLastError());
            return ticks;
        }
        ticks[CentralProcessor.TickType.USER.getIndex()] = cpuLoadInfo.cpu_ticks[0];
        ticks[CentralProcessor.TickType.NICE.getIndex()] = cpuLoadInfo.cpu_ticks[3];
        ticks[CentralProcessor.TickType.SYSTEM.getIndex()] = cpuLoadInfo.cpu_ticks[1];
        ticks[CentralProcessor.TickType.IDLE.getIndex()] = cpuLoadInfo.cpu_ticks[2];
        return ticks;
    }

    @Override
    public double[] getSystemLoadAverage(int nelem) {
        if (nelem < 1 || nelem > 3) {
            throw new IllegalArgumentException("Must include from one to three elements.");
        }
        double[] average = new double[nelem];
        int retval = SystemB.INSTANCE.getloadavg(average, nelem);
        if (retval < nelem) {
            Arrays.fill(average, -1.0);
        }
        return average;
    }

    @Override
    public long[] queryCurrentFreq() {
        long[] freq = new long[]{SysctlKit.sysctl("hw.cpufrequency", this.getProcessorIdentifier().getVendorFreq())};
        return freq;
    }

    @Override
    public long queryContextSwitches() {
        SystemB.VMMeter vmstats;
        int machPort = SystemB.INSTANCE.mach_host_self();
        if (0 != SystemB.INSTANCE.host_statistics(machPort, 2, (Structure)(vmstats = new SystemB.VMMeter()), new IntByReference(vmstats.size()))) {
            Logger.error("Failed to update vmstats. Error code: {}", Native.getLastError());
            return -1L;
        }
        return Builder.unsignedIntToLong(vmstats.v_swtch);
    }

    @Override
    public long queryInterrupts() {
        SystemB.VMMeter vmstats;
        int machPort = SystemB.INSTANCE.mach_host_self();
        if (0 != SystemB.INSTANCE.host_statistics(machPort, 2, (Structure)(vmstats = new SystemB.VMMeter()), new IntByReference(vmstats.size()))) {
            Logger.error("Failed to update vmstats. Error code: {}", Native.getLastError());
            return -1L;
        }
        return Builder.unsignedIntToLong(vmstats.v_intr);
    }

    @Override
    public long queryMaxFreq() {
        return SysctlKit.sysctl("hw.cpufrequency_max", this.getProcessorIdentifier().getVendorFreq());
    }

    @Override
    public long[][] queryProcessorCpuLoadTicks() {
        IntByReference procInfoCount;
        PointerByReference procCpuLoadInfo;
        IntByReference procCount;
        long[][] ticks = new long[this.getLogicalProcessorCount()][CentralProcessor.TickType.values().length];
        int machPort = SystemB.INSTANCE.mach_host_self();
        if (0 != SystemB.INSTANCE.host_processor_info(machPort, 2, procCount = new IntByReference(), procCpuLoadInfo = new PointerByReference(), procInfoCount = new IntByReference())) {
            Logger.error("Failed to update CPU Load. Error code: {}", Native.getLastError());
            return ticks;
        }
        int[] cpuTicks = procCpuLoadInfo.getValue().getIntArray(0L, procInfoCount.getValue());
        for (int cpu = 0; cpu < procCount.getValue(); ++cpu) {
            int offset = cpu * 4;
            ticks[cpu][CentralProcessor.TickType.USER.getIndex()] = Builder.getUnsignedInt(cpuTicks[offset + 0]);
            ticks[cpu][CentralProcessor.TickType.NICE.getIndex()] = Builder.getUnsignedInt(cpuTicks[offset + 3]);
            ticks[cpu][CentralProcessor.TickType.SYSTEM.getIndex()] = Builder.getUnsignedInt(cpuTicks[offset + 1]);
            ticks[cpu][CentralProcessor.TickType.IDLE.getIndex()] = Builder.getUnsignedInt(cpuTicks[offset + 2]);
        }
        return ticks;
    }
}

