/*********************************************************************************
 *                                                                               *
 * The MIT License (MIT)                                                         *
 *                                                                               *
 * Copyright (c) 2015-2021 aoju.org OSHI and other contributors.                 *
 *                                                                               *
 * Permission is hereby granted, free of charge, to any person obtaining a copy  *
 * of this software and associated documentation files (the "Software"), to deal *
 * in the Software without restriction, including without limitation the rights  *
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     *
 * copies of the Software, and to permit persons to whom the Software is         *
 * furnished to do so, subject to the following conditions:                      *
 *                                                                               *
 * The above copyright notice and this permission notice shall be included in    *
 * all copies or substantial portions of the Software.                           *
 *                                                                               *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   *
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, *
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     *
 * THE SOFTWARE.                                                                 *
 *                                                                               *
 ********************************************************************************/
package org.aoju.bus.health.linux.hardware;

import com.sun.jna.platform.linux.Udev;
import com.sun.jna.platform.linux.Udev.UdevDevice;
import com.sun.jna.platform.linux.Udev.UdevEnumerate;
import com.sun.jna.platform.linux.Udev.UdevListEntry;
import org.aoju.bus.core.annotation.Immutable;
import org.aoju.bus.core.lang.Normal;
import org.aoju.bus.core.lang.Symbol;
import org.aoju.bus.health.builtin.hardware.AbstractUsbDevice;
import org.aoju.bus.health.builtin.hardware.UsbDevice;

import java.util.*;

/**
 * Linux Usb Device
 *
 * @author Kimi Liu
 * @version 6.2.3
 * @since JDK 1.8+
 */
@Immutable
public class LinuxUsbDevice extends AbstractUsbDevice {

    public LinuxUsbDevice(String name, String vendor, String vendorId, String productId, String serialNumber,
                          String uniqueDeviceId, List<UsbDevice> connectedDevices) {
        super(name, vendor, vendorId, productId, serialNumber, uniqueDeviceId, connectedDevices);
    }

    /**
     * Instantiates a list of {@link UsbDevice} objects, representing
     * devices connected via a usb port (including internal devices).
     * <p>
     * If the value of {@code tree} is true, the top level devices returned from
     * this method are the USB Controllers; connected hubs and devices in its device
     * tree share that controller's bandwidth. If the value of {@code tree} is
     * false, USB devices (not controllers) are listed in a single flat list.
     *
     * @param tree If true, returns a list of controllers, which requires recursive
     *             iteration of connected devices. If false, returns a flat list of
     *             devices excluding controllers.
     * @return a list of {@link  UsbDevice} objects.
     */
    public static List<UsbDevice> getUsbDevices(boolean tree) {
        List<UsbDevice> devices = getUsbDevices();
        if (tree) {
            return devices;
        }
        List<UsbDevice> deviceList = new ArrayList<>();
        // Top level is controllers; they won't be added to the list, but all
        // their connected devices will be
        for (UsbDevice device : devices) {
            deviceList.add(new LinuxUsbDevice(device.getName(), device.getVendor(), device.getVendorId(),
                    device.getProductId(), device.getSerialNumber(), device.getUniqueDeviceId(),
                    Collections.emptyList()));
            addDevicesToList(deviceList, device.getConnectedDevices());
        }
        return deviceList;
    }

    private static List<UsbDevice> getUsbDevices() {
        // Enumerate all usb devices and build information maps
        Udev.UdevContext udev = Udev.INSTANCE.udev_new();
        // Create a list of the devices in the 'usb' subsystem.
        UdevEnumerate enumerate = Udev.INSTANCE.udev_enumerate_new(udev);
        Udev.INSTANCE.udev_enumerate_add_match_subsystem(enumerate, "usb");
        Udev.INSTANCE.udev_enumerate_scan_devices(enumerate);
        UdevListEntry devices = Udev.INSTANCE.udev_enumerate_get_list_entry(enumerate);

        // Build a list of devices with no parent; these will be the roots
        List<String> usbControllers = new ArrayList<>();

        // Maps to store information using device node path as the key
        Map<String, String> nameMap = new HashMap<>();
        Map<String, String> vendorMap = new HashMap<>();
        Map<String, String> vendorIdMap = new HashMap<>();
        Map<String, String> productIdMap = new HashMap<>();
        Map<String, String> serialMap = new HashMap<>();
        Map<String, List<String>> hubMap = new HashMap<>();

        // For each item enumerated, store information in the maps
        for (UdevListEntry dev_list_entry = devices; null != dev_list_entry; dev_list_entry = Udev.INSTANCE
                .udev_list_entry_get_next(dev_list_entry)) {

            // Get the filename of the /sys entry for the device and create a
            // udev_device object (dev) representing it
            String path = Udev.INSTANCE.udev_list_entry_get_name(dev_list_entry);
            UdevDevice dev = Udev.INSTANCE.udev_device_new_from_syspath(udev, path);
            // Ignore interfaces
            if (!"usb_device".equals(Udev.INSTANCE.udev_device_get_devtype(dev))) {
                continue;
            }

            // Use the path as the key for the maps
            String value = Udev.INSTANCE.udev_device_get_sysattr_value(dev, "product");
            if (null != value) {
                nameMap.put(path, value);
            }
            value = Udev.INSTANCE.udev_device_get_sysattr_value(dev, "manufacturer");
            if (null != value) {
                vendorMap.put(path, value);
            }
            value = Udev.INSTANCE.udev_device_get_sysattr_value(dev, "idVendor");
            if (null != value) {
                vendorIdMap.put(path, value);
            }
            value = Udev.INSTANCE.udev_device_get_sysattr_value(dev, "idProduct");
            if (null != value) {
                productIdMap.put(path, value);
            }
            value = Udev.INSTANCE.udev_device_get_sysattr_value(dev, "serial");
            if (null != value) {
                serialMap.put(path, value);
            }
            UdevDevice parent = Udev.INSTANCE.udev_device_get_parent_with_subsystem_devtype(dev, "usb", "usb_device");
            if (null == parent) {
                // This is a controller with no parent, add to list
                usbControllers.add(path);
            } else {
                // Add child path (path variable) to parent's path
                String parentPath = Udev.INSTANCE.udev_device_get_syspath(parent);
                hubMap.computeIfAbsent(parentPath, x -> new ArrayList<>()).add(path);
            }
            Udev.INSTANCE.udev_device_unref(dev);
        }
        // Free the enumerator object
        Udev.INSTANCE.udev_enumerate_unref(enumerate);
        Udev.INSTANCE.udev_unref(udev);

        // Build tree and return
        List<UsbDevice> controllerDevices = new ArrayList<>();
        for (String controller : usbControllers) {
            controllerDevices.add(getDeviceAndChildren(controller, "0000", "0000", nameMap, vendorMap, vendorIdMap,
                    productIdMap, serialMap, hubMap));
        }
        return controllerDevices;
    }

    private static void addDevicesToList(List<UsbDevice> deviceList, List<UsbDevice> list) {
        for (UsbDevice device : list) {
            deviceList.add(device);
            addDevicesToList(deviceList, device.getConnectedDevices());
        }
    }

    /**
     * Recursively creates LinuxUsbDevices by fetching information from maps to
     * populate fields
     *
     * @param devPath      The device node path.
     * @param vid          The default (parent) vendor ID
     * @param pid          The default (parent) product ID
     * @param nameMap      the map of names
     * @param vendorMap    the map of vendors
     * @param vendorIdMap  the map of vendorIds
     * @param productIdMap the map of productIds
     * @param serialMap    the map of serial numbers
     * @param hubMap       the map of hubs
     * @return A LinuxUsbDevice corresponding to this device
     */
    private static LinuxUsbDevice getDeviceAndChildren(String devPath, String vid, String pid,
                                                       Map<String, String> nameMap, Map<String, String> vendorMap, Map<String, String> vendorIdMap,
                                                       Map<String, String> productIdMap, Map<String, String> serialMap, Map<String, List<String>> hubMap) {
        String vendorId = vendorIdMap.getOrDefault(devPath, vid);
        String productId = productIdMap.getOrDefault(devPath, pid);
        List<String> childPaths = hubMap.getOrDefault(devPath, new ArrayList<>());
        List<UsbDevice> usbDevices = new ArrayList<>();
        for (String path : childPaths) {
            usbDevices.add(getDeviceAndChildren(path, vendorId, productId, nameMap, vendorMap, vendorIdMap,
                    productIdMap, serialMap, hubMap));
        }
        Collections.sort(usbDevices);
        return new LinuxUsbDevice(nameMap.getOrDefault(devPath, vendorId + Symbol.C_COLON + productId),
                vendorMap.getOrDefault(devPath, Normal.EMPTY), vendorId, productId, serialMap.getOrDefault(devPath, Normal.EMPTY), devPath,
                usbDevices);
    }

}
