/*********************************************************************************
 *                                                                               *
 * The MIT License (MIT)                                                         *
 *                                                                               *
 * Copyright (c) 2015-2022 aoju.org and other contributors.                      *
 *                                                                               *
 * Permission is hereby granted, free of charge, to any person obtaining a copy  *
 * of this software and associated documentation files (the "Software"), to deal *
 * in the Software without restriction, including without limitation the rights  *
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     *
 * copies of the Software, and to permit persons to whom the Software is         *
 * furnished to do so, subject to the following conditions:                      *
 *                                                                               *
 * The above copyright notice and this permission notice shall be included in    *
 * all copies or substantial portions of the Software.                           *
 *                                                                               *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   *
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, *
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     *
 * THE SOFTWARE.                                                                 *
 *                                                                               *
 ********************************************************************************/
package org.aoju.bus.core.compress;

import org.aoju.bus.core.lang.exception.InstrumentException;
import org.aoju.bus.core.toolkit.FileKit;
import org.aoju.bus.core.toolkit.IoKit;
import org.aoju.bus.core.toolkit.ZipKit;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.Enumeration;
import java.util.function.Consumer;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;

/**
 * Zip文件或流读取器，一般用于Zip文件解压
 *
 * @author Kimi Liu
 * @version 6.3.5
 * @since JDK 1.8+
 */
public class ZipReader implements Closeable {

    private ZipFile zipFile;
    private ZipInputStream in;

    /**
     * 构造
     *
     * @param zipFile 读取的的Zip文件
     * @param charset 编码
     */
    public ZipReader(File zipFile, Charset charset) {
        this.zipFile = ZipKit.zipFile(zipFile, charset);
    }

    /**
     * 构造
     *
     * @param zipFile 读取的的Zip文件
     */
    public ZipReader(ZipFile zipFile) {
        this.zipFile = zipFile;
    }

    /**
     * 构造
     *
     * @param in      读取的的Zip文件流
     * @param charset 编码
     */
    public ZipReader(InputStream in, Charset charset) {
        this.in = new ZipInputStream(in, charset);
    }

    /**
     * 构造
     *
     * @param zin 读取的的Zip文件流
     */
    public ZipReader(ZipInputStream zin) {
        this.in = zin;
    }

    /**
     * 创建{@link ZipReader}
     *
     * @param zipFile 生成的Zip文件
     * @param charset 编码
     * @return {@link ZipReader}
     */
    public static ZipReader of(File zipFile, Charset charset) {
        return new ZipReader(zipFile, charset);
    }

    /**
     * 创建{@link ZipReader}
     *
     * @param in      Zip输入的流，一般为输入文件流
     * @param charset 编码
     * @return {@link ZipReader}
     */
    public static ZipReader of(InputStream in, Charset charset) {
        return new ZipReader(in, charset);
    }

    /**
     * 获取指定路径的文件流
     *
     * @param path 路径
     * @return 文件流
     */
    public InputStream get(String path) {
        if (null != this.zipFile) {
            final ZipFile zipFile = this.zipFile;
            final ZipEntry entry = zipFile.getEntry(path);
            if (null != entry) {
                return ZipKit.get(zipFile, entry);
            }
        } else {
            throw new UnsupportedOperationException("Zip stream mode not support get!");
        }

        return null;
    }

    /**
     * 解压到指定目录中
     *
     * @param outFile 解压到的目录
     * @return 解压的目录
     * @throws InstrumentException IO异常
     */
    public File readTo(File outFile) throws InstrumentException {
        read((zipEntry) -> {
            File outItemFile = FileKit.file(outFile, zipEntry.getName());
            if (zipEntry.isDirectory()) {
                outItemFile.mkdirs();
            } else {
                InputStream in;
                if (null != this.zipFile) {
                    in = ZipKit.get(this.zipFile, zipEntry);
                } else {
                    in = this.in;
                }
                // 文件
                FileKit.writeFromStream(in, outItemFile, false);
            }
        });
        return outFile;
    }

    /**
     * 读取并处理Zip文件中的每一个{@link ZipEntry}
     *
     * @param consumer {@link ZipEntry}处理器
     * @return this
     * @throws InstrumentException IO异常
     */
    public ZipReader read(Consumer<ZipEntry> consumer) throws InstrumentException {
        if (null != this.zipFile) {
            readFromZipFile(consumer);
        } else {
            readFromStream(consumer);
        }
        return this;
    }

    @Override
    public void close() throws InstrumentException {
        if (null != this.zipFile) {
            IoKit.close(this.zipFile);
        } else {
            IoKit.close(this.in);
        }
    }

    /**
     * 读取并处理Zip文件中的每一个{@link ZipEntry}
     *
     * @param consumer {@link ZipEntry}处理器
     */
    private void readFromZipFile(Consumer<ZipEntry> consumer) {
        final Enumeration<? extends ZipEntry> em = zipFile.entries();
        while (em.hasMoreElements()) {
            consumer.accept(em.nextElement());
        }
    }

    /**
     * 读取并处理Zip流中的每一个{@link ZipEntry}
     *
     * @param consumer {@link ZipEntry}处理器
     * @throws InstrumentException IO异常
     */
    private void readFromStream(Consumer<ZipEntry> consumer) throws InstrumentException {
        try {
            ZipEntry zipEntry;
            while (null != (zipEntry = in.getNextEntry())) {
                consumer.accept(zipEntry);
            }
        } catch (IOException e) {
            throw new InstrumentException(e);
        }
    }

}
