/*********************************************************************************
 *                                                                               *
 * The MIT License (MIT)                                                         *
 *                                                                               *
 * Copyright (c) 2015-2022 aoju.org and other contributors.                      *
 *                                                                               *
 * Permission is hereby granted, free of charge, to any person obtaining a copy  *
 * of this software and associated documentation files (the "Software"), to deal *
 * in the Software without restriction, including without limitation the rights  *
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     *
 * copies of the Software, and to permit persons to whom the Software is         *
 * furnished to do so, subject to the following conditions:                      *
 *                                                                               *
 * The above copyright notice and this permission notice shall be included in    *
 * all copies or substantial portions of the Software.                           *
 *                                                                               *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   *
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, *
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     *
 * THE SOFTWARE.                                                                 *
 *                                                                               *
 ********************************************************************************/
package org.aoju.bus.core.map;

import org.aoju.bus.core.lang.Assert;
import org.aoju.bus.core.toolkit.IterKit;
import org.aoju.bus.core.toolkit.ObjectKit;

import java.io.Serializable;
import java.util.*;
import java.util.function.Function;

/**
 * 抽象{@link Table}接口实现
 * 默认实现了：
 * <ul>
 *     <li>{@link #equals(Object)}</li>
 *     <li>{@link #hashCode()}</li>
 *     <li>{@link #toString()}</li>
 *     <li>{@link #values()}</li>
 *     <li>{@link #cellSet()}</li>
 *     <li>{@link #iterator()}</li>
 * </ul>
 *
 * @param <R> 行类型
 * @param <C> 列类型
 * @param <V> 值类型
 * @author Kimi Liu
 * @version 6.5.0
 * @since Java 17+
 */
public abstract class AbstractTable<R, C, V> implements Table<R, C, V> {

    private Collection<V> values;
    private Set<Cell<R, C, V>> cellSet;

    @Override
    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        } else if (obj instanceof Table) {
            final Table<?, ?, ?> that = (Table<?, ?, ?>) obj;
            return this.cellSet().equals(that.cellSet());
        } else {
            return false;
        }
    }

    @Override
    public int hashCode() {
        return cellSet().hashCode();
    }

    @Override
    public String toString() {
        return rowMap().toString();
    }

    @Override
    public Collection<V> values() {
        Collection<V> result = values;
        return (result == null) ? values = new Values() : result;
    }

    @Override
    public Set<Cell<R, C, V>> cellSet() {
        Set<Cell<R, C, V>> result = cellSet;
        return (result == null) ? cellSet = new CellSet() : result;
    }

    @Override
    public Iterator<Cell<R, C, V>> iterator() {
        return new CellIterator();
    }

    /**
     * 简单{@link Cell} 实现
     *
     * @param <R> 行类型
     * @param <C> 列类型
     * @param <V> 值类型
     */
    private static class SimpleCell<R, C, V> implements Cell<R, C, V>, Serializable {

        private static final long serialVersionUID = 1L;

        private final R rowKey;
        private final C columnKey;
        private final V value;

        SimpleCell(R rowKey, C columnKey, V value) {
            this.rowKey = rowKey;
            this.columnKey = columnKey;
            this.value = value;
        }

        @Override
        public R getRowKey() {
            return rowKey;
        }

        @Override
        public C getColumnKey() {
            return columnKey;
        }

        @Override
        public V getValue() {
            return value;
        }

        @Override
        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (obj instanceof Cell) {
                Cell<?, ?, ?> other = (Cell<?, ?, ?>) obj;
                return ObjectKit.equal(rowKey, other.getRowKey())
                        && ObjectKit.equal(columnKey, other.getColumnKey())
                        && ObjectKit.equal(value, other.getValue());
            }
            return false;
        }

        @Override
        public int hashCode() {
            return Objects.hash(rowKey, columnKey, value);
        }

        @Override
        public String toString() {
            return "(" + rowKey + "," + columnKey + ")=" + value;
        }
    }

    private class Values extends AbstractCollection<V> {

        @Override
        public Iterator<V> iterator() {
            return new TransIterator<>(cellSet().iterator(), Cell::getValue);
        }

        @Override
        public boolean contains(Object o) {
            return containsValue((V) o);
        }

        @Override
        public void clear() {
            AbstractTable.this.clear();
        }

        @Override
        public int size() {
            return AbstractTable.this.size();
        }
    }

    private class CellSet extends AbstractSet<Cell<R, C, V>> {
        @Override
        public boolean contains(Object o) {
            if (o instanceof Cell) {
                final Cell<R, C, V> cell = (Cell<R, C, V>) o;
                Map<C, V> row = getRow(cell.getRowKey());
                if (null != row) {
                    return ObjectKit.equals(row.get(cell.getColumnKey()), cell.getValue());
                }
            }
            return false;
        }

        @Override
        public boolean remove(Object o) {
            if (contains(o)) {
                final Cell<R, C, V> cell = (Cell<R, C, V>) o;
                AbstractTable.this.remove(cell.getRowKey(), cell.getColumnKey());
            }
            return false;
        }

        @Override
        public void clear() {
            AbstractTable.this.clear();
        }

        @Override
        public Iterator<Cell<R, C, V>> iterator() {
            return new CellIterator();
        }

        @Override
        public int size() {
            return AbstractTable.this.size();
        }
    }

    /**
     * 基于{@link Cell}的{@link Iterator}实现
     */
    private class CellIterator implements Iterator<Cell<R, C, V>> {
        final Iterator<Map.Entry<R, Map<C, V>>> rowIterator = rowMap().entrySet().iterator();
        Map.Entry<R, Map<C, V>> rowEntry;
        Iterator<Map.Entry<C, V>> columnIterator = IterKit.empty();

        @Override
        public boolean hasNext() {
            return rowIterator.hasNext() || columnIterator.hasNext();
        }

        @Override
        public Cell<R, C, V> next() {
            if (false == columnIterator.hasNext()) {
                rowEntry = rowIterator.next();
                columnIterator = rowEntry.getValue().entrySet().iterator();
            }
            final Map.Entry<C, V> columnEntry = columnIterator.next();
            return new SimpleCell<>(rowEntry.getKey(), columnEntry.getKey(), columnEntry.getValue());
        }

        @Override
        public void remove() {
            columnIterator.remove();
            if (rowEntry.getValue().isEmpty()) {
                rowIterator.remove();
            }
        }
    }

    /**
     * 转换迭代器
     *
     * @param <F> 对象
     * @param <T> 对象
     */
    public class TransIterator<F, T> implements Iterator<T> {

        private final Iterator<? extends F> backingIterator;
        private final Function<? super F, ? extends T> func;

        /**
         * 构造
         *
         * @param backingIterator 源{@link Iterator}
         * @param func            转换函数
         */
        public TransIterator(Iterator<? extends F> backingIterator, Function<? super F, ? extends T> func) {
            this.backingIterator = Assert.notNull(backingIterator);
            this.func = Assert.notNull(func);
        }

        @Override
        public final boolean hasNext() {
            return backingIterator.hasNext();
        }

        @Override
        public final T next() {
            return func.apply(backingIterator.next());
        }

        @Override
        public final void remove() {
            backingIterator.remove();
        }
    }

}
