package org.aoju.bus.pay.provider.wxpay;

import org.aoju.bus.core.lang.Algorithm;
import org.aoju.bus.core.lang.Http;
import org.aoju.bus.core.lang.MediaType;
import org.aoju.bus.core.toolkit.FileKit;
import org.aoju.bus.core.toolkit.StringKit;
import org.aoju.bus.pay.Builder;
import org.aoju.bus.pay.Mode;
import org.aoju.bus.pay.magic.Results;
import org.aoju.bus.pay.metric.HttpKit;
import org.aoju.bus.pay.metric.WxPayKit;
import org.aoju.bus.pay.provider.wxpay.enums.ApiUrl;
import org.aoju.bus.pay.provider.wxpay.enums.WxDomain;

import java.io.File;
import java.io.InputStream;
import java.security.PrivateKey;
import java.util.HashMap;
import java.util.Map;

/**
 * 微信支付
 */
public class WxpayProvider {

    private static final String OS = System.getProperty("os.name") + "/" + System.getProperty("os.version");
    private static final String VERSION = System.getProperty("java.version");

    /**
     * 获取接口请求的 URL
     *
     * @param apiUrl {@link ApiUrl} 支付 API 接口枚举
     * @return {@link String} 返回完整的接口请求URL
     */
    public static String getReqUrl(ApiUrl apiUrl) {
        return getReqUrl(apiUrl, null, false);
    }

    /**
     * 获取接口请求的 URL
     *
     * @param apiUrl    {@link ApiUrl} 支付 API 接口枚举
     * @param isSandBox 是否是沙箱环境
     * @return {@link String} 返回完整的接口请求URL
     */
    public static String getReqUrl(ApiUrl apiUrl, boolean isSandBox) {
        return getReqUrl(apiUrl, null, isSandBox);
    }

    /**
     * 获取接口请求的 URL
     *
     * @param apiUrl    {@link ApiUrl} 支付 API 接口枚举
     * @param wxDomain  {@link WxDomain} 支付 API 接口域名枚举
     * @param isSandBox 是否是沙箱环境
     * @return {@link String} 返回完整的接口请求URL
     */
    public static String getReqUrl(ApiUrl apiUrl, WxDomain wxDomain, boolean isSandBox) {
        if (wxDomain == null) {
            wxDomain = WxDomain.CHINA;
        }
        return wxDomain.getType()
                .concat(isSandBox ? ApiUrl.SAND_BOX_NEW.getType() : "")
                .concat(apiUrl.getType());
    }

    /**
     * 发起请求
     *
     * @param apiUrl 接口 URL
     *               通过 {@link WxpayProvider#getReqUrl(ApiUrl)}
     *               或者 {@link WxpayProvider#getReqUrl(ApiUrl, WxDomain, boolean)} 来获取
     * @param params 接口请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String execution(String apiUrl, Map<String, String> params) {
        return doPost(apiUrl, params);
    }

    /**
     * 发起请求
     *
     * @param apiUrl 接口 URL
     *               通过 {@link WxpayProvider#getReqUrl(ApiUrl)}
     *               或者 {@link WxpayProvider#getReqUrl(ApiUrl, WxDomain, boolean)} 来获取
     * @param params 接口请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String executionByGet(String apiUrl, Map<String, Object> params) {
        return doGet(apiUrl, params);
    }

    /**
     * 发起请求
     *
     * @param apiUrl   接口 URL
     *                 通过 {@link WxpayProvider#getReqUrl(ApiUrl)}
     *                 或者 {@link WxpayProvider#getReqUrl(ApiUrl, WxDomain, boolean)} 来获取
     * @param params   接口请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String execution(String apiUrl, Map<String, String> params, String certPath, String certPass) {
        return doPostSsl(apiUrl, params, certPath, certPass);
    }

    /**
     * 发起请求
     *
     * @param apiUrl   接口 URL
     *                 通过 {@link WxpayProvider#getReqUrl(ApiUrl)}
     *                 或者 {@link WxpayProvider#getReqUrl(ApiUrl, WxDomain, boolean)} 来获取
     * @param params   接口请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @param protocol 协议
     * @return {@link String} 请求返回的结果
     */
    public static String executionByProtocol(String apiUrl, Map<String, String> params, String certPath, String certPass, String protocol) {
        return doPostSslByProtocol(apiUrl, params, certPath, certPass, protocol);
    }

    /**
     * 发起请求
     *
     * @param apiUrl   接口 URL
     *                 通过 {@link WxpayProvider#getReqUrl(ApiUrl)}
     *                 或者 {@link WxpayProvider#getReqUrl(ApiUrl, WxDomain, boolean)} 来获取
     * @param params   接口请求参数
     * @param certPath 证书文件路径
     * @return {@link String} 请求返回的结果
     */
    public static String execution(String apiUrl, Map<String, String> params, String certPath) {
        return doPostSsl(apiUrl, params, certPath);
    }

    /**
     * 发起请求
     *
     * @param apiUrl   接口 URL
     *                 通过 {@link WxpayProvider#getReqUrl(ApiUrl)}
     *                 或者 {@link WxpayProvider#getReqUrl(ApiUrl, WxDomain, boolean)} 来获取
     * @param params   接口请求参数
     * @param certPath 证书文件路径
     * @param protocol 协议
     * @return {@link String} 请求返回的结果
     */
    public static String executionByProtocol(String apiUrl, Map<String, String> params, String certPath, String protocol) {
        return doPostSslByProtocol(apiUrl, params, certPath, protocol);
    }

    /**
     * 发起请求
     *
     * @param apiUrl   接口 URL
     *                 通过 {@link WxpayProvider#getReqUrl(ApiUrl)}
     *                 或者 {@link WxpayProvider#getReqUrl(ApiUrl, WxDomain, boolean)} 来获取
     * @param params   接口请求参数
     * @param certFile 证书文件输入流
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String execution(String apiUrl, Map<String, String> params, InputStream certFile, String certPass) {
        return doPostSsl(apiUrl, params, certFile, certPass);
    }

    /**
     * 发起请求
     *
     * @param apiUrl   接口 URL
     *                 通过 {@link WxpayProvider#getReqUrl(ApiUrl)}
     *                 或者 {@link WxpayProvider#getReqUrl(ApiUrl, WxDomain, boolean)} 来获取
     * @param params   接口请求参数
     * @param certFile 证书文件输入流
     * @param certPass 证书密码
     * @param protocol 协议
     * @return {@link String} 请求返回的结果
     */
    public static String executionByProtocol(String apiUrl, Map<String, String> params, InputStream certFile, String certPass, String protocol) {
        return doPostSslByProtocol(apiUrl, params, certFile, certPass, protocol);
    }

    /**
     * 发起请求
     *
     * @param apiUrl   接口 URL
     *                 通过 {@link WxpayProvider#getReqUrl(ApiUrl)}
     *                 或者 {@link WxpayProvider#getReqUrl(ApiUrl, WxDomain, boolean)} 来获取
     * @param params   接口请求参数
     * @param certFile 证书文件输入流
     * @return {@link String} 请求返回的结果
     */
    public static String execution(String apiUrl, Map<String, String> params, InputStream certFile) {
        return doPostSsl(apiUrl, params, certFile);
    }

    /**
     * 发起请求
     *
     * @param apiUrl   接口 URL
     *                 通过 {@link WxpayProvider#getReqUrl(ApiUrl)}
     *                 或者 {@link WxpayProvider#getReqUrl(ApiUrl, WxDomain, boolean)} 来获取
     * @param params   接口请求参数
     * @param certFile 证书文件输入流
     * @param protocol 协议
     * @return {@link String} 请求返回的结果
     */
    public static String executionByProtocol(String apiUrl, Map<String, String> params, InputStream certFile, String protocol) {
        return doPostSslByProtocol(apiUrl, params, certFile, protocol);
    }

    public static String execution(String apiUrl, Map<String, String> params,
                                   String certPath, String certPass, String filePath) {
        return doUploadSsl(apiUrl, params, certPath, certPass, filePath);
    }

    public static String executionByProtocol(String apiUrl, Map<String, String> params, String certPath, String certPass, String filePath, String protocol) {
        return doUploadSslByProtocol(apiUrl, params, certPath, certPass, filePath, protocol);
    }


    /**
     * V3 接口统一执行入口
     *
     * @param method       {@link Http} 请求方法
     * @param urlPrefix    可通过 {@link WxDomain}来获取
     * @param urlSuffix    可通过 {@link ApiUrl} 来获取，URL挂载参数需要自行拼接
     * @param mchId        商户Id
     * @param serialNo     商户 API 证书序列号
     * @param platSerialNo 平台序列号，接口中包含敏感信息时必传
     * @param keyPath      apiclient_key.pem 证书路径
     * @param body         接口请求参数
     * @param nonceStr     随机字符库
     * @param timestamp    时间戳
     * @param authType     认证类型
     * @param file         文件
     * @return {@link Results} 请求返回的结果
     * @throws Exception 接口执行异常
     */
    public static Results v3(String method, String urlPrefix, String urlSuffix,
                             String mchId, String serialNo, String platSerialNo, String keyPath,
                             String body, String nonceStr, long timestamp, String authType,
                             File file) throws Exception {
        // 构建 Authorization
        String authorization = WxPayKit.buildAuthorization(method, urlSuffix, mchId, serialNo,
                keyPath, body, nonceStr, timestamp, authType);

        if (StringKit.isEmpty(platSerialNo)) {
            platSerialNo = serialNo;
        }

        if (Http.GET.equals(method)) {
            return get(urlPrefix.concat(urlSuffix), authorization, platSerialNo, null);
        } else if (Http.POST.equals(method)) {
            if (FileKit.isNotEmpty(file)) {
                return upload(urlPrefix.concat(urlSuffix), authorization, platSerialNo, body, file);
            }
            return post(urlPrefix.concat(urlSuffix), authorization, platSerialNo, body);
        } else if (Http.DELETE.equals(method)) {
            return delete(urlPrefix.concat(urlSuffix), authorization, platSerialNo, body);
        } else if (Http.PUT.equals(method)) {
            return put(urlPrefix.concat(urlSuffix), authorization, platSerialNo, body);
        }
        return null;
    }

    /**
     * V3 接口统一执行入口
     *
     * @param method       {@link Http} 请求方法
     * @param urlPrefix    可通过 {@link WxDomain}来获取
     * @param urlSuffix    可通过 {@link ApiUrl} 来获取，URL挂载参数需要自行拼接
     * @param mchId        商户Id
     * @param serialNo     商户 API 证书序列号
     * @param platSerialNo 平台序列号，接口中包含敏感信息时必传
     * @param privateKey   商户私钥
     * @param body         接口请求参数
     * @param nonceStr     随机字符库
     * @param timestamp    时间戳
     * @param authType     认证类型
     * @param file         文件
     * @return {@link Results} 请求返回的结果
     * @throws Exception 接口执行异常
     */
    public static Results v3(String method, String urlPrefix, String urlSuffix,
                             String mchId, String serialNo, String platSerialNo, PrivateKey privateKey,
                             String body, String nonceStr, long timestamp, String authType,
                             File file) throws Exception {
        // 构建 Authorization
        String authorization = WxPayKit.buildAuthorization(method, urlSuffix, mchId, serialNo,
                privateKey, body, nonceStr, timestamp, authType);

        if (StringKit.isEmpty(platSerialNo)) {
            platSerialNo = serialNo;
        }

        if (Http.GET.equals(method)) {
            return get(urlPrefix.concat(urlSuffix), authorization, platSerialNo, null);
        } else if (Http.POST.equals(method)) {
            if (FileKit.isNotEmpty(file)) {
                return upload(urlPrefix.concat(urlSuffix), authorization, platSerialNo, body, file);
            }
            return post(urlPrefix.concat(urlSuffix), authorization, platSerialNo, body);
        } else if (Http.DELETE.equals(method)) {
            return delete(urlPrefix.concat(urlSuffix), authorization, platSerialNo, body);
        } else if (Http.PUT.equals(method)) {
            return put(urlPrefix.concat(urlSuffix), authorization, platSerialNo, body);
        }
        return null;
    }

    /**
     * V3 接口统一执行入口
     *
     * @param method       {@link Http} 请求方法
     * @param urlPrefix    可通过 {@link WxDomain}来获取
     * @param urlSuffix    可通过 {@link ApiUrl} 来获取，URL挂载参数需要自行拼接
     * @param mchId        商户Id
     * @param serialNo     商户 API 证书序列号
     * @param platSerialNo 平台序列号
     * @param keyPath      apiclient_key.pem 证书路径
     * @param body         接口请求参数
     * @return {@link Results} 请求返回的结果
     * @throws Exception 接口执行异常
     */
    public static Results v3(String method, String urlPrefix, String urlSuffix, String mchId,
                             String serialNo, String platSerialNo, String keyPath, String body) throws Exception {
        long timestamp = System.currentTimeMillis() / 1000;
        String authType = "WECHATPAY2-SHA256-RSA2048";
        String nonceStr = WxPayKit.generateStr();
        return v3(method, urlPrefix, urlSuffix, mchId, serialNo, platSerialNo, keyPath, body, nonceStr, timestamp, authType, null);
    }

    /**
     * V3 接口统一执行入口
     *
     * @param method       {@link Http} 请求方法
     * @param urlPrefix    可通过 {@link WxDomain}来获取
     * @param urlSuffix    可通过 {@link ApiUrl} 来获取，URL挂载参数需要自行拼接
     * @param mchId        商户Id
     * @param serialNo     商户 API 证书序列号
     * @param platSerialNo 平台序列号
     * @param privateKey   商户私钥
     * @param body         接口请求参数
     * @return {@link Results} 请求返回的结果
     * @throws Exception 接口执行异常
     */
    public static Results v3(String method, String urlPrefix, String urlSuffix, String mchId,
                             String serialNo, String platSerialNo, PrivateKey privateKey, String body) throws Exception {
        long timestamp = System.currentTimeMillis() / 1000;
        String authType = "WECHATPAY2-SHA256-RSA2048";
        String nonceStr = WxPayKit.generateStr();
        return v3(method, urlPrefix, urlSuffix, mchId, serialNo, platSerialNo, privateKey, body, nonceStr, timestamp, authType, null);
    }

    /**
     * V3 接口统一执行入口
     *
     * @param method       {@link Http} 请求方法
     * @param urlPrefix    可通过 {@link WxDomain}来获取
     * @param urlSuffix    可通过 {@link ApiUrl} 来获取，URL挂载参数需要自行拼接
     * @param mchId        商户Id
     * @param serialNo     商户 API 证书序列号
     * @param platSerialNo 平台序列号
     * @param keyPath      apiclient_key.pem 证书路径
     * @param params       Get 接口请求参数
     * @return {@link Results} 请求返回的结果
     * @throws Exception 接口执行异常
     */
    public static Results v3(String method, String urlPrefix, String urlSuffix,
                             String mchId, String serialNo, String platSerialNo, String keyPath,
                             Map<String, String> params) throws Exception {
        long timestamp = System.currentTimeMillis() / 1000;
        String authType = "WECHATPAY2-SHA256-RSA2048";
        String nonceStr = WxPayKit.generateStr();
        if (null != params && !params.keySet().isEmpty()) {
            urlSuffix = urlSuffix.concat("?").concat(Builder.createLinkString(params, true));
        }
        return v3(method, urlPrefix, urlSuffix, mchId, serialNo, platSerialNo, keyPath, "", nonceStr, timestamp, authType, null);
    }

    /**
     * V3 接口统一执行入口
     *
     * @param method       {@link Http} 请求方法
     * @param urlPrefix    可通过 {@link WxDomain}来获取
     * @param urlSuffix    可通过 {@link ApiUrl} 来获取，URL挂载参数需要自行拼接
     * @param mchId        商户Id
     * @param serialNo     商户 API 证书序列号
     * @param platSerialNo 平台序列号
     * @param privateKey   商户私钥
     * @param params       Get 接口请求参数
     * @return {@link Results} 请求返回的结果
     * @throws Exception 接口执行异常
     */
    public static Results v3(String method, String urlPrefix, String urlSuffix,
                             String mchId, String serialNo, String platSerialNo, PrivateKey privateKey,
                             Map<String, String> params) throws Exception {
        long timestamp = System.currentTimeMillis() / 1000;
        String authType = "WECHATPAY2-SHA256-RSA2048";
        String nonceStr = WxPayKit.generateStr();
        if (null != params && !params.keySet().isEmpty()) {
            urlSuffix = urlSuffix.concat("?").concat(Builder.createLinkString(params, true));
        }
        return v3(method, urlPrefix, urlSuffix, mchId, serialNo, platSerialNo, privateKey, "", nonceStr, timestamp, authType, null);
    }

    /**
     * V3 接口统一执行入口
     *
     * @param urlPrefix    可通过 {@link WxDomain}来获取
     * @param urlSuffix    可通过 {@link ApiUrl} 来获取，URL挂载参数需要自行拼接
     * @param mchId        商户Id
     * @param serialNo     商户 API 证书序列号
     * @param platSerialNo 平台序列号
     * @param keyPath      apiclient_key.pem 证书路径
     * @param body         接口请求参数
     * @param file         文件
     * @return {@link Results} 请求返回的结果
     * @throws Exception 接口执行异常
     */
    public static Results v3(String urlPrefix, String urlSuffix, String mchId, String serialNo, String platSerialNo, String keyPath, String body, File file) throws Exception {
        long timestamp = System.currentTimeMillis() / 1000;
        String authType = "WECHATPAY2-SHA256-RSA2048";
        String nonceStr = WxPayKit.generateStr();
        return v3(Http.POST, urlPrefix, urlSuffix, mchId, serialNo, platSerialNo, keyPath, body, nonceStr, timestamp, authType, file);
    }

    /**
     * V3 接口统一执行入口
     *
     * @param urlPrefix    可通过 {@link WxDomain}来获取
     * @param urlSuffix    可通过 {@link ApiUrl} 来获取，URL挂载参数需要自行拼接
     * @param mchId        商户Id
     * @param serialNo     商户 API 证书序列号
     * @param platSerialNo 平台序列号
     * @param privateKey   商户私钥
     * @param body         接口请求参数
     * @param file         文件
     * @return {@link Results} 请求返回的结果
     * @throws Exception 接口执行异常
     */
    public static Results v3(String urlPrefix, String urlSuffix, String mchId, String serialNo,
                             String platSerialNo, PrivateKey privateKey, String body, File file) throws Exception {
        long timestamp = System.currentTimeMillis() / 1000;
        String authType = "WECHATPAY2-SHA256-RSA2048";
        String nonceStr = WxPayKit.generateStr();
        return v3(Http.POST, urlPrefix, urlSuffix, mchId, serialNo, platSerialNo, privateKey, body, nonceStr, timestamp, authType, file);
    }

    /**
     * V3 接口统一执行入口
     *
     * @param method       {@link Http} 请求方法
     * @param urlPrefix    可通过 {@link WxDomain}来获取
     * @param urlSuffix    可通过 {@link ApiUrl} 来获取，URL挂载参数需要自行拼接
     * @param mchId        商户Id
     * @param serialNo     商户 API 证书序列号
     * @param platSerialNo 平台序列号，接口中包含敏感信息时必传
     * @param keyPath      apiclient_key.pem 证书路径
     * @param body         接口请求参数
     * @param nonceStr     随机字符库
     * @param timestamp    时间戳
     * @param authType     认证类型
     * @param file         文件
     * @return {@link Map} 请求返回的结果
     * @throws Exception 接口执行异常
     */
    @Deprecated
    public static Map<String, Object> v3Execution(String method, String urlPrefix, String urlSuffix,
                                                  String mchId, String serialNo, String platSerialNo, String keyPath,
                                                  String body, String nonceStr, long timestamp, String authType,
                                                  File file) throws Exception {
        Results response = v3(method, urlPrefix, urlSuffix, mchId, serialNo, platSerialNo, keyPath, body, nonceStr, timestamp, authType, file);
        return buildResMap(response);
    }

    /**
     * V3 接口统一执行入口
     *
     * @param method    {@link Http} 请求方法
     * @param urlPrefix 可通过 {@link WxDomain}来获取
     * @param urlSuffix 可通过 {@link ApiUrl} 来获取，URL挂载参数需要自行拼接
     * @param mchId     商户Id
     * @param serialNo  商户 API 证书序列号
     * @param keyPath   apiclient_key.pem 证书路径
     * @param body      接口请求参数
     * @return {@link Map} 请求返回的结果
     */
    @Deprecated
    public static Map<String, Object> v3Execution(String method, String urlPrefix, String urlSuffix, String mchId,
                                                  String serialNo, String keyPath, String body) throws Exception {
        Results response = v3(method, urlPrefix, urlSuffix, mchId, serialNo, null, keyPath, body);
        return buildResMap(response);
    }

    /**
     * V3 接口统一执行入口
     *
     * @param method       {@link Http} 请求方法
     * @param urlPrefix    可通过 {@link WxDomain}来获取
     * @param urlSuffix    可通过 {@link ApiUrl} 来获取，URL挂载参数需要自行拼接
     * @param mchId        商户Id
     * @param serialNo     商户 API 证书序列号
     * @param platSerialNo 平台序列号
     * @param keyPath      apiclient_key.pem 证书路径
     * @param body         接口请求参数
     * @return {@link Map} 请求返回的结果
     * @throws Exception 接口执行异常
     */
    @Deprecated
    public static Map<String, Object> v3Execution(String method, String urlPrefix, String urlSuffix, String mchId,
                                                  String serialNo, String platSerialNo, String keyPath, String body) throws Exception {
        Results response = v3(method, urlPrefix, urlSuffix, mchId, serialNo, platSerialNo, keyPath, body);
        return buildResMap(response);
    }

    /**
     * V3 接口统一执行入口
     *
     * @param method       {@link Http} 请求方法
     * @param urlPrefix    可通过 {@link WxDomain}来获取
     * @param urlSuffix    可通过 {@link ApiUrl} 来获取，URL挂载参数需要自行拼接
     * @param mchId        商户Id
     * @param serialNo     商户 API 证书序列号
     * @param platSerialNo 平台序列号
     * @param keyPath      apiclient_key.pem 证书路径
     * @param params       Get 接口请求参数
     * @return {@link Map} 请求返回的结果
     * @throws Exception 接口执行异常
     */
    @Deprecated
    public static Map<String, Object> v3Execution(String method, String urlPrefix, String urlSuffix,
                                                  String mchId, String serialNo, String platSerialNo, String keyPath,
                                                  Map<String, String> params) throws Exception {
        Results response = v3(method, urlPrefix, urlSuffix, mchId, serialNo, platSerialNo, keyPath, params);
        return buildResMap(response);
    }

    /**
     * V3 接口统一执行入口
     *
     * @param method    {@link Http} 请求方法
     * @param urlPrefix 可通过 {@link WxDomain}来获取
     * @param urlSuffix 可通过 {@link ApiUrl} 来获取，URL挂载参数需要自行拼接
     * @param mchId     商户Id
     * @param serialNo  商户 API 证书序列号
     * @param keyPath   apiclient_key.pem 证书路径
     * @param params    Get 接口请求参数
     * @return {@link Map} 请求返回的结果
     * @throws Exception 接口执行异常
     */
    @Deprecated
    public static Map<String, Object> v3Execution(String method, String urlPrefix, String urlSuffix,
                                                  String mchId, String serialNo, String keyPath,
                                                  Map<String, String> params) throws Exception {
        Results response = v3(method, urlPrefix, urlSuffix, mchId, serialNo, null, keyPath, params);
        return buildResMap(response);
    }

    /**
     * V3 接口统一执行入口
     *
     * @param urlPrefix    可通过 {@link WxDomain}来获取
     * @param urlSuffix    可通过 {@link ApiUrl} 来获取，URL挂载参数需要自行拼接
     * @param mchId        商户Id
     * @param serialNo     商户 API 证书序列号
     * @param platSerialNo 平台序列号
     * @param keyPath      apiclient_key.pem 证书路径
     * @param body         接口请求参数
     * @param file         文件
     * @return {@link Map} 请求返回的结果
     * @throws Exception 接口执行异常
     */
    @Deprecated
    public static Map<String, Object> v3Upload(String urlPrefix, String urlSuffix, String mchId, String serialNo, String platSerialNo, String keyPath, String body, File file) throws Exception {
        Results response = v3(urlPrefix, urlSuffix, mchId, serialNo, platSerialNo, keyPath, body, file);
        return buildResMap(response);
    }

    /**
     * V3 接口统一执行入口
     *
     * @param urlPrefix 可通过 {@link WxDomain}来获取
     * @param urlSuffix 可通过 {@link ApiUrl} 来获取，URL挂载参数需要自行拼接
     * @param mchId     商户Id
     * @param serialNo  商户 API 证书序列号
     * @param keyPath   apiclient_key.pem 证书路径
     * @param body      接口请求参数
     * @param file      文件
     * @return {@link Map} 请求返回的结果
     * @throws Exception 接口执行异常
     */
    @Deprecated
    public static Map<String, Object> v3Upload(String urlPrefix, String urlSuffix, String mchId, String serialNo, String keyPath, String body, File file) throws Exception {
        return v3Upload(urlPrefix, urlSuffix, mchId, serialNo, null, keyPath, body, file);
    }

    /**
     * 获取验签秘钥API
     *
     * @param mchId      商户号
     * @param partnerKey API 密钥
     * @param algorithm  签名方式
     * @return {@link String} 请求返回的结果
     */
    public static String getSignKey(String mchId, String partnerKey, Algorithm algorithm) {
        Map<String, String> map = new HashMap<>(3);
        String nonceStr = WxPayKit.generateStr();
        map.put("mch_id", mchId);
        map.put("nonce_str", nonceStr);
        map.put("sign", WxPayKit.createSign(map, partnerKey, algorithm));
        return execution(getReqUrl(ApiUrl.GET_SIGN_KEY), map);
    }

    /**
     * 统一下单
     *
     * @param params 请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String pushOrder(Map<String, String> params) {
        return pushOrder(false, null, params);
    }

    /**
     * 统一下单
     *
     * @param isSandbox 是否是沙盒环境
     * @param params    请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String pushOrder(boolean isSandbox, Map<String, String> params) {
        return pushOrder(isSandbox, null, params);
    }

    /**
     * 统一下单
     *
     * @param isSandbox 是否是沙盒环境
     * @param wxDomain  {@link WxDomain} 支付 API 接口域名枚举
     * @param params    请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String pushOrder(boolean isSandbox, WxDomain wxDomain, Map<String, String> params) {
        return execution(getReqUrl(ApiUrl.UNIFIED_ORDER, wxDomain, isSandbox), params);
    }

    /**
     * 订单查询
     *
     * @param params 请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String orderQuery(Map<String, String> params) {
        return orderQuery(false, null, params);
    }

    /**
     * 订单查询
     *
     * @param isSandbox 是否是沙盒环境
     * @param params    请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String orderQuery(boolean isSandbox, Map<String, String> params) {
        return execution(getReqUrl(ApiUrl.ORDER_QUERY, null, isSandbox), params);
    }

    /**
     * 订单查询
     *
     * @param isSandbox 是否是沙盒环境
     * @param params    请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String orderQuery(boolean isSandbox, WxDomain wxDomain, Map<String, String> params) {
        return execution(getReqUrl(ApiUrl.ORDER_QUERY, wxDomain, isSandbox), params);
    }

    /**
     * 关闭订单
     *
     * @param params 请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String closeOrder(Map<String, String> params) {
        return execution(getReqUrl(ApiUrl.CLOSE_ORDER), params);
    }

    /**
     * 撤销订单
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String orderReverse(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.REVERSE), params, certPath, certPass);
    }

    /**
     * 撤销订单
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String orderReverse(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.REVERSE), params, certFile, certPass);
    }

    /**
     * 申请退款
     *
     * @param isSandbox 是否是沙盒环境
     * @param params    请求参数
     * @param certPath  证书文件路径
     * @param certPass  证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String orderRefund(boolean isSandbox, Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.REFUND, null, isSandbox), params, certPath, certPass);
    }

    /**
     * 申请退款
     *
     * @param isSandbox 是否是沙盒环境
     * @param params    请求参数
     * @param certPath  证书文件路径
     * @param certPass  证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String orderRefundByProtocol(boolean isSandbox, Map<String, String> params, String certPath, String certPass, String protocol) {
        return executionByProtocol(getReqUrl(ApiUrl.REFUND, null, isSandbox), params, certPath, certPass, protocol);
    }

    /**
     * 申请退款
     *
     * @param isSandbox 是否是沙盒环境
     * @param params    请求参数
     * @param certFile  证书文件的 InputStream
     * @param certPass  证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String orderRefund(boolean isSandbox, Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.REFUND, null, isSandbox), params, certFile, certPass);
    }

    /**
     * 申请退款
     *
     * @param isSandbox 是否是沙盒环境
     * @param params    请求参数
     * @param certFile  证书文件的 InputStream
     * @param certPass  证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String orderRefundByProtocol(boolean isSandbox, Map<String, String> params, InputStream certFile, String certPass, String protocol) {
        return executionByProtocol(getReqUrl(ApiUrl.REFUND, null, isSandbox), params, certFile, certPass, protocol);
    }

    /**
     * 查询退款
     *
     * @param isSandbox 是否是沙盒环境
     * @param params    请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String orderRefundQuery(boolean isSandbox, Map<String, String> params) {
        return execution(getReqUrl(ApiUrl.REFUND_QUERY, null, isSandbox), params);
    }

    /**
     * 下载对账单
     *
     * @param isSandbox 是否是沙盒环境
     * @param params    请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String downloadBill(boolean isSandbox, Map<String, String> params) {
        return execution(getReqUrl(ApiUrl.DOWNLOAD_BILL, null, isSandbox), params);
    }

    /**
     * 交易保障
     *
     * @param params 请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String orderReport(Map<String, String> params) {
        return execution(getReqUrl(ApiUrl.REPORT, null, false), params);
    }

    /**
     * 转换短链接
     *
     * @param params 请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String toShortUrl(Map<String, String> params) {
        return execution(getReqUrl(ApiUrl.SHORT_URL, null, false), params);
    }

    /**
     * 授权码查询 openId
     *
     * @param params 请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String authCodeToOpenid(Map<String, String> params) {
        return execution(getReqUrl(ApiUrl.AUTH_CODE_TO_OPENID, null, false), params);
    }

    /**
     * 刷卡支付
     *
     * @param isSandbox 是否是沙盒环境
     * @param params    请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String microPay(boolean isSandbox, Map<String, String> params) {
        return execution(getReqUrl(ApiUrl.MICRO_PAY, null, isSandbox), params);
    }

    /**
     * 企业付款到零钱
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String transfers(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.TRANSFER, null, false), params, certPath, certPass);
    }

    /**
     * 企业付款到零钱
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @param protocol 协议
     * @return {@link String} 请求返回的结果
     */
    public static String transfersByProtocol(Map<String, String> params, String certPath, String certPass, String protocol) {
        return executionByProtocol(getReqUrl(ApiUrl.TRANSFER, null, false), params, certPath, certPass, protocol);
    }

    /**
     * 企业付款到零钱
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String transfers(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.TRANSFER, null, false), params, certFile, certPass);
    }

    /**
     * 企业付款到零钱
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @param protocol 协议
     * @return {@link String} 请求返回的结果
     */
    public static String transfersByProtocol(Map<String, String> params, InputStream certFile, String certPass, String protocol) {
        return executionByProtocol(getReqUrl(ApiUrl.TRANSFER, null, false), params, certFile, certPass, protocol);
    }

    /**
     * 查询企业付款到零钱
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String getTransferInfo(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.GET_TRANSFER_INFO, null, false), params, certPath, certPass);
    }

    /**
     * 查询企业付款到零钱
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String getTransferInfo(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.GET_TRANSFER_INFO, null, false), params, certFile, certPass);
    }

    /**
     * 企业付款到银行
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String payBank(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.TRANSFER_BANK, null, false), params, certPath, certPass);
    }

    /**
     * 企业付款到银行
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @param protocol 协议
     * @return {@link String} 请求返回的结果
     */
    public static String payBankByProtocol(Map<String, String> params, String certPath, String certPass, String protocol) {
        return executionByProtocol(getReqUrl(ApiUrl.TRANSFER_BANK, null, false), params, certPath, certPass, protocol);
    }

    /**
     * 企业付款到银行
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String payBank(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.TRANSFER_BANK, null, false), params, certFile, certPass);
    }

    /**
     * 企业付款到银行
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @param protocol 协议
     * @return {@link String} 请求返回的结果
     */
    public static String payBankByProtocol(Map<String, String> params, InputStream certFile, String certPass, String protocol) {
        return executionByProtocol(getReqUrl(ApiUrl.TRANSFER_BANK, null, false), params, certFile, certPass, protocol);
    }

    /**
     * 查询企业付款到银行
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String queryBank(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.GET_TRANSFER_BANK_INFO, null, false), params, certPath, certPass);
    }

    /**
     * 查询企业付款到银行
     *
     * @param params   请求参数
     * @param certFile 证书文件的  InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String queryBank(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.GET_TRANSFER_BANK_INFO, null, false), params, certFile, certPass);
    }

    /**
     * 获取 RSA 加密公钥
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String getPublicKey(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.GET_PUBLIC_KEY, WxDomain.FRAUD, false), params, certPath, certPass);
    }

    /**
     * 获取 RSA 加密公钥
     *
     * @param params   请求参数
     * @param certFile 证书文件的   InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String getPublicKey(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.GET_PUBLIC_KEY, WxDomain.FRAUD, false), params, certFile, certPass);
    }

    /**
     * 公众号纯签约
     *
     * @param params 请求参数
     * @param mode   商户平台模式
     * @return {@link String} 请求返回的结果
     */
    public static String entrustWeb(Map<String, Object> params, Mode mode) {
        if (mode == Mode.BUSINESS_MODEL) {
            return executionByGet(getReqUrl(ApiUrl.ENTRUST_WEB), params);
        } else {
            return executionByGet(getReqUrl(ApiUrl.PARTNER_ENTRUST_WEB), params);
        }
    }


    /**
     * APP 纯签约
     *
     * @param params 请求参数
     * @param mode   商户平台模式
     * @return {@link String} 请求返回的结果
     */
    public static String preEntrustWeb(Map<String, Object> params, Mode mode) {
        if (mode == Mode.BUSINESS_MODEL) {
            return executionByGet(getReqUrl(ApiUrl.PRE_ENTRUST_WEB), params);
        } else {
            return executionByGet(getReqUrl(ApiUrl.PARTNER_PRE_ENTRUST_WEB), params);
        }
    }


    /**
     * H5 纯签约
     *
     * @param params 请求参数
     * @param mode   商户平台模式
     * @return {@link String} 请求返回的结果
     */
    public static String h5EntrustWeb(Map<String, Object> params, Mode mode) {
        if (mode == Mode.BUSINESS_MODEL) {
            return executionByGet(getReqUrl(ApiUrl.H5_ENTRUST_WEB), params);
        } else {
            return executionByGet(getReqUrl(ApiUrl.PARTNER_H5_ENTRUST_WEB), params);
        }
    }

    /**
     * 支付中签约
     *
     * @param params 请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String contractOrder(Map<String, String> params) {
        return execution(getReqUrl(ApiUrl.PAY_CONTRACT_ORDER), params);
    }

    /**
     * 查询签约关系
     *
     * @param params 请求参数
     * @param mode   商户平台模式
     * @return {@link String} 请求返回的结果
     */
    public static String queryContract(Map<String, String> params, Mode mode) {
        if (mode == Mode.BUSINESS_MODEL) {
            return execution(getReqUrl(ApiUrl.QUERY_ENTRUST_CONTRACT), params);
        } else {
            return execution(getReqUrl(ApiUrl.PARTNER_QUERY_ENTRUST_CONTRACT), params);
        }
    }

    /**
     * 申请扣款
     *
     * @param params 请求参数
     * @param mode   商户平台模式
     * @return {@link String} 请求返回的结果
     */
    public static String papPayApply(Map<String, String> params, Mode mode) {
        if (mode == Mode.BUSINESS_MODEL) {
            return execution(getReqUrl(ApiUrl.PAP_PAY_APPLY), params);
        } else {
            return execution(getReqUrl(ApiUrl.PARTNER_PAP_PAY_APPLY), params);
        }
    }

    /**
     * 申请解约
     *
     * @param params 请求参数
     * @param mode   商户平台模式
     * @return {@link String} 请求返回的结果
     */
    public static String deleteContract(Map<String, String> params, Mode mode) {
        if (mode == Mode.BUSINESS_MODEL) {
            return execution(getReqUrl(ApiUrl.DELETE_ENTRUST_CONTRACT), params);
        } else {
            return execution(getReqUrl(ApiUrl.PARTNER_DELETE_ENTRUST_CONTRACT), params);
        }
    }

    /**
     * 查询签约关系对账单
     *
     * @param params 请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String contractBill(Map<String, String> params, Mode mode) {
        if (mode == Mode.BUSINESS_MODEL) {
            return execution(getReqUrl(ApiUrl.QUERY_ENTRUST_CONTRACT), params);
        } else {
            return execution(getReqUrl(ApiUrl.PARTNER_QUERY_ENTRUST_CONTRACT), params);
        }
    }

    /**
     * 查询代扣订单
     *
     * @param params 请求参数
     * @param mode   商户平台模式
     * @return {@link String} 请求返回的结果
     */
    public static String papOrderQuery(Map<String, String> params, Mode mode) {
        if (mode == Mode.BUSINESS_MODEL) {
            return execution(getReqUrl(ApiUrl.PAP_ORDER_QUERY), params);
        } else {
            return execution(getReqUrl(ApiUrl.PARTNER_PAP_ORDER_QUERY), params);
        }
    }

    /**
     * 请求单次分账
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String profitSharing(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.PROFIT_SHARING), params, certPath, certPass);
    }

    /**
     * 请求单次分账
     *
     * @param params   请求参数
     * @param certFile 证书文件的  InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String profitSharing(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.PROFIT_SHARING), params, certFile, certPass);
    }

    /**
     * 请求多次分账
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String multiProfitSharing(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.MULTI_PROFIT_SHARING), params, certPath, certPass);
    }

    /**
     * 请求多次分账
     *
     * @param params   请求参数
     * @param certFile 证书文件的  InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String multiProfitSharing(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.MULTI_PROFIT_SHARING), params, certFile, certPass);
    }

    /**
     * 查询分账结果
     *
     * @param params 请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String profitSharingQuery(Map<String, String> params) {
        return execution(getReqUrl(ApiUrl.PROFIT_SHARING_QUERY), params);
    }

    /**
     * 添加分账接收方
     *
     * @param params 请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String profitSharingAddReceiver(Map<String, String> params) {
        return execution(getReqUrl(ApiUrl.PROFITS_HARING_ADD_RECEIVER), params);
    }

    /**
     * 删除分账接收方
     *
     * @param params 请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String profitSharingRemoveReceiver(Map<String, String> params) {
        return execution(getReqUrl(ApiUrl.PROFIT_SHARING_REMOVE_RECEIVER), params);
    }

    /**
     * 完结分账
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String profitSharingFinish(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.PROFIT_SHARING_FINISH), params, certPath, certPass);
    }

    /**
     * 完结分账
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String profitSharingFinish(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.PROFIT_SHARING_FINISH), params, certFile, certPass);
    }

    /**
     * 分账回退
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String profitSharingReturn(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.PROFIT_SHARING_RETURN), params, certPath, certPass);
    }

    /**
     * 分账回退
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String profitSharingReturn(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.PROFIT_SHARING_RETURN), params, certFile, certPass);
    }

    /**
     * 分账回退结果查询
     *
     * @param params 请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String profitSharingReturnQuery(Map<String, String> params) {
        return execution(getReqUrl(ApiUrl.PROFIT_SHARING_RETURN_QUERY), params);
    }

    /**
     * 发放代金券
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String sendCoupon(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.SEND_COUPON), params, certPath, certPass);
    }

    /**
     * 发放代金券
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String sendCoupon(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.SEND_COUPON), params, certFile, certPass);
    }

    /**
     * 查询代金券批次
     *
     * @param params 请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String queryCouponStock(Map<String, String> params) {
        return execution(getReqUrl(ApiUrl.QUERY_COUPON_STOCK), params);
    }

    /**
     * 查询代金券信息
     *
     * @param params 请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String queryCouponsInfo(Map<String, String> params) {
        return execution(getReqUrl(ApiUrl.QUERY_COUPONS_INFO), params);
    }

    /**
     * 拉取订单评价数据
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String batchQueryComment(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.BATCH_QUERY_COMMENT), params, certPath, certPass);
    }

    /**
     * 拉取订单评价数据
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String batchQueryComment(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.BATCH_QUERY_COMMENT), params, certFile, certPass);
    }

    /**
     * 支付押金（人脸支付）
     *
     * @param params 请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String depositFacePay(Map<String, String> params) {
        return execution(getReqUrl(ApiUrl.DEPOSIT_FACE_PAY), params);
    }

    /**
     * 支付押金（付款码支付）
     *
     * @param params 请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String depositMicroPay(Map<String, String> params) {
        return execution(getReqUrl(ApiUrl.DEPOSIT_MICRO_PAY), params);
    }

    /**
     * 查询订单
     *
     * @param params 请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String depositOrderQuery(Map<String, String> params) {
        return execution(getReqUrl(ApiUrl.DEPOSIT_ORDER_QUERY), params);
    }

    /**
     * 撤销订单
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String depositReverse(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.DEPOSIT_REVERSE), params, certPath, certPass);
    }

    /**
     * 撤销订单
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String depositReverse(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.DEPOSIT_REVERSE), params, certFile, certPass);
    }

    /**
     * 消费押金
     *
     * @param params   请求参数
     * @param certPath 证书文件的目录
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String depositConsume(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.DEPOSIT_CONSUME), params, certPath, certPass);
    }

    /**
     * 消费押金
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String depositConsume(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.DEPOSIT_CONSUME), params, certFile, certPass);
    }

    /**
     * 申请退款（押金）
     *
     * @param params   请求参数
     * @param certPath 证书文件的目录
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String depositRefund(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.DEPOSIT_REFUND), params, certPath, certPass);
    }

    /**
     * 申请退款（押金）
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String depositRefund(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.DEPOSIT_REFUND), params, certFile, certPass);
    }

    /**
     * 查询退款（押金）
     *
     * @param params 请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String depositRefundQuery(Map<String, String> params) {
        return execution(getReqUrl(ApiUrl.DEPOSIT_REFUND_QUERY), params);
    }

    /**
     * 下载资金账单
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String downloadFundFlow(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.DOWNLOAD_FUND_FLOW), params, certPath, certPass);
    }

    /**
     * 下载资金账单
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String downloadFundFlow(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.DOWNLOAD_FUND_FLOW), params, certFile, certPass);
    }

    /**
     * 刷脸设备获取设备调用凭证
     *
     * @param params 请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String getAuthInfo(Map<String, String> params) {
        return execution(getReqUrl(ApiUrl.GET_AUTH_INFO, WxDomain.PAY_APP, false), params);
    }

    /**
     * 刷脸支付
     *
     * @param params 请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String facePay(Map<String, String> params) {
        return execution(getReqUrl(ApiUrl.FACE_PAY), params);
    }

    /**
     * 查询刷脸支付订单
     *
     * @param params 请求参数
     * @return {@link String} 请求返回的结果
     */
    public static String facePayQuery(Map<String, String> params) {
        return execution(getReqUrl(ApiUrl.FACE_PAY_QUERY), params);
    }

    /**
     * 刷脸支付撤销订单
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String facePayReverse(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.FACE_PAY_REVERSE), params, certPath, certPass);
    }

    /**
     * 刷脸支付撤销订单
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String facePayReverse(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.FACE_PAY_REVERSE), params, certFile, certPass);
    }

    /**
     * 发放普通红包
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String sendRedPack(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.SEND_RED_PACK), params, certPath, certPass);
    }

    /**
     * 发放普通红包
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @param protocol 协议
     * @return {@link String} 请求返回的结果
     */
    public static String sendRedPackByProtocol(Map<String, String> params, String certPath, String certPass, String protocol) {
        return executionByProtocol(getReqUrl(ApiUrl.SEND_RED_PACK), params, certPath, certPass, protocol);
    }

    /**
     * 发放普通红包
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String sendRedPack(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.SEND_RED_PACK), params, certFile, certPass);
    }

    /**
     * 发放普通红包
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @param protocol 协议
     * @return {@link String} 请求返回的结果
     */
    public static String sendRedPackByProtocol(Map<String, String> params, InputStream certFile, String certPass, String protocol) {
        return executionByProtocol(getReqUrl(ApiUrl.SEND_RED_PACK), params, certFile, certPass, protocol);
    }

    /**
     * 发放裂变红包
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String sendGroupRedPack(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.SEND_GROUP_RED_PACK), params, certPath, certPass);
    }

    /**
     * 发放裂变红包
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @param protocol 协议
     * @return {@link String} 请求返回的结果
     */
    public static String sendGroupRedPackByProtocol(Map<String, String> params, String certPath, String certPass, String protocol) {
        return executionByProtocol(getReqUrl(ApiUrl.SEND_GROUP_RED_PACK), params, certPath, certPass, protocol);
    }

    /**
     * 发放裂变红包
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String sendGroupRedPack(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.SEND_GROUP_RED_PACK), params, certFile, certPass);
    }

    /**
     * 发放裂变红包
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @param protocol 协议
     * @return {@link String} 请求返回的结果
     */
    public static String sendGroupRedPackByProtocol(Map<String, String> params, InputStream certFile, String certPass, String protocol) {
        return executionByProtocol(getReqUrl(ApiUrl.SEND_GROUP_RED_PACK), params, certFile, certPass, protocol);
    }

    /**
     * 查询红包记录
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String getHbInfo(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.GET_HB_INFO), params, certPath, certPass);
    }

    /**
     * 查询红包记录
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String getHbInfo(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.GET_HB_INFO), params, certFile, certPass);
    }

    /**
     * 小程序发放红包接口
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String sendMiniProgramRedPack(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.SEND_MINI_PROGRAM_HB), params, certPath, certPass);
    }

    /**
     * 小程序发放红包接口
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @param protocol 协议
     * @return {@link String} 请求返回的结果
     */
    public static String sendMiniProgramRedPackByProtocol(Map<String, String> params, String certPath, String certPass, String protocol) {
        return executionByProtocol(getReqUrl(ApiUrl.SEND_MINI_PROGRAM_HB), params, certPath, certPass, protocol);
    }

    /**
     * 小程序发放红包接口
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String sendMiniProgramRedPack(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.SEND_MINI_PROGRAM_HB), params, certFile, certPass);
    }

    /**
     * 小程序发放红包接口
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @param protocol 协议
     * @return {@link String} 请求返回的结果
     */
    public static String sendMiniProgramRedPackByProtocol(Map<String, String> params, InputStream certFile, String certPass, String protocol) {
        return executionByProtocol(getReqUrl(ApiUrl.SEND_MINI_PROGRAM_HB), params, certFile, certPass, protocol);
    }

    /**
     * 发放企业红包
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String sendWorkWxRedPack(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.SEND_WORK_WX_RED_PACK), params, certPath, certPass);
    }

    /**
     * 发放企业红包
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @param protocol 协议
     * @return {@link String} 请求返回的结果
     */
    public static String sendWorkWxRedPackByProtocol(Map<String, String> params, String certPath, String certPass, String protocol) {
        return executionByProtocol(getReqUrl(ApiUrl.SEND_WORK_WX_RED_PACK), params, certPath, certPass, protocol);
    }

    /**
     * 发放企业红包
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String sendWorkWxRedPack(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.SEND_WORK_WX_RED_PACK), params, certFile, certPass);
    }

    /**
     * 发放企业红包
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @param protocol 协议
     * @return {@link String} 请求返回的结果
     */
    public static String sendWorkWxRedPackByProtocol(Map<String, String> params, InputStream certFile, String certPass, String protocol) {
        return executionByProtocol(getReqUrl(ApiUrl.SEND_WORK_WX_RED_PACK), params, certFile, certPass, protocol);
    }

    /**
     * 查询向员工付款记录
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String queryWorkWxRedPack(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.QUERY_WORK_WX_RED_PACK), params, certPath, certPass);
    }

    /**
     * 查询向员工付款记录
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @param protocol 协议
     * @return {@link String} 请求返回的结果
     */
    public static String queryWorkWxRedPackByProtocol(Map<String, String> params, String certPath, String certPass, String protocol) {
        return executionByProtocol(getReqUrl(ApiUrl.QUERY_WORK_WX_RED_PACK), params, certPath, certPass, protocol);
    }

    /**
     * 查询向员工付款记录
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String queryWorkWxRedPack(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.QUERY_WORK_WX_RED_PACK), params, certFile, certPass);
    }

    /**
     * 查询向员工付款记录
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @param protocol 协议
     * @return {@link String} 请求返回的结果
     */
    public static String queryWorkWxRedPackByProtocol(Map<String, String> params, InputStream certFile, String certPass, String protocol) {
        return executionByProtocol(getReqUrl(ApiUrl.QUERY_WORK_WX_RED_PACK), params, certFile, certPass, protocol);
    }

    /**
     * 向员工付款
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String trans2pocket(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.PAY_WWS_TRANS_2_POCKET), params, certPath, certPass);
    }

    /**
     * 向员工付款
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @param protocol 协议
     * @return {@link String} 请求返回的结果
     */
    public static String trans2pocketByProtocol(Map<String, String> params, String certPath, String certPass, String protocol) {
        return executionByProtocol(getReqUrl(ApiUrl.PAY_WWS_TRANS_2_POCKET), params, certPath, certPass, protocol);
    }

    /**
     * 向员工付款
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String trans2pocket(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.PAY_WWS_TRANS_2_POCKET), params, certFile, certPass);
    }

    /**
     * 向员工付款
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @param protocol 协议
     * @return {@link String} 请求返回的结果
     */
    public static String trans2pocketByProtocol(Map<String, String> params, InputStream certFile, String certPass, String protocol) {
        return executionByProtocol(getReqUrl(ApiUrl.PAY_WWS_TRANS_2_POCKET), params, certFile, certPass, protocol);
    }

    /**
     * 查询向员工付款记录
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String queryTrans2pocket(Map<String, String> params, String certPath, String certPass) {
        return execution(getReqUrl(ApiUrl.QUERY_WWS_TRANS_2_POCKET), params, certPath, certPass);
    }

    /**
     * 查询向员工付款记录
     *
     * @param params   请求参数
     * @param certPath 证书文件路径
     * @param certPass 证书密码
     * @param protocol 协议
     * @return {@link String} 请求返回的结果
     */
    public static String queryTrans2pocketByProtocol(Map<String, String> params, String certPath, String certPass, String protocol) {
        return executionByProtocol(getReqUrl(ApiUrl.QUERY_WWS_TRANS_2_POCKET), params, certPath, certPass, protocol);
    }

    /**
     * 查询向员工付款记录
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @return {@link String} 请求返回的结果
     */
    public static String queryTrans2pocket(Map<String, String> params, InputStream certFile, String certPass) {
        return execution(getReqUrl(ApiUrl.QUERY_WWS_TRANS_2_POCKET), params, certFile, certPass);
    }

    /**
     * 查询向员工付款记录
     *
     * @param params   请求参数
     * @param certFile 证书文件的 InputStream
     * @param certPass 证书密码
     * @param protocol 协议
     * @return {@link String} 请求返回的结果
     */
    public static String queryTrans2pocket(Map<String, String> params, InputStream certFile, String certPass, String protocol) {
        return executionByProtocol(getReqUrl(ApiUrl.QUERY_WWS_TRANS_2_POCKET), params, certFile, certPass, protocol);
    }

    /**
     * @param url    请求url
     * @param params 请求参数
     * @return {@link String}    请求返回的结果
     */
    public static String doGet(String url, Map<String, Object> params) {
        return HttpKit.get(url, params);
    }

    /**
     * get 请求
     *
     * @param url     请求url
     * @param params  请求参数
     * @param headers 请求头
     * @return {@link Results}    请求返回的结果
     */
    public static Results get(String url, Map<String, Object> params, Map<String, String> headers) {
        return HttpKit.get(url, params, headers);
    }

    /**
     * get 请求
     *
     * @param url           请求url
     * @param authorization 授权信息
     * @param serialNumber  公钥证书序列号
     * @param params        请求参数
     * @return {@link Results}    请求返回的结果
     */
    public static Results get(String url, String authorization, String serialNumber, Map<String, Object> params) {
        return get(url, params, getHeaders(authorization, serialNumber));
    }

    /**
     * post 请求
     *
     * @param url     请求url
     * @param data    请求参数
     * @param headers 请求头
     * @return {@link Results}    请求返回的结果
     */
    public static Results post(String url, String data, Map<String, String> headers) {
        return HttpKit.post(url, data, headers);
    }

    /**
     * post 请求
     *
     * @param url           请求url
     * @param authorization 授权信息
     * @param serialNumber  公钥证书序列号
     * @param data          请求参数
     * @return {@link Results}    请求返回的结果
     */
    public static Results post(String url, String authorization, String serialNumber, String data) {
        return post(url, data, getHeaders(authorization, serialNumber));
    }

    /**
     * delete 请求
     *
     * @param url     请求url
     * @param data    请求参数
     * @param headers 请求头
     * @return {@link Results}    请求返回的结果
     */
    public static Results delete(String url, String data, Map<String, String> headers) {
        return HttpKit.delete(url, data, headers);
    }

    /**
     * delete 请求
     *
     * @param url           请求url
     * @param authorization 授权信息
     * @param serialNumber  公钥证书序列号
     * @param data          请求参数
     * @return {@link Results}    请求返回的结果
     */
    public static Results delete(String url, String authorization, String serialNumber, String data) {
        return delete(url, data, getHeaders(authorization, serialNumber));
    }

    /**
     * upload 请求
     *
     * @param url     请求url
     * @param params  请求参数
     * @param headers 请求头
     * @return {@link Results}    请求返回的结果
     */
    public static Results upload(String url, Map<String, Object> params, Map<String, String> headers) {
        return HttpKit.post(url, params, headers);
    }

    /**
     * upload 请求
     *
     * @param url           请求url
     * @param authorization 授权信息
     * @param serialNumber  公钥证书序列号
     * @param data          请求参数
     * @param file          上传文件
     * @return {@link Results}    请求返回的结果
     */
    public static Results upload(String url, String authorization, String serialNumber, String data, File file) {
        Map<String, Object> paramMap = new HashMap<>(2);
        paramMap.put("file", file);
        paramMap.put("meta", data);
        return upload(url, paramMap, getUploadHeaders(authorization, serialNumber));
    }


    /**
     * put 请求
     *
     * @param url     请求url
     * @param data    请求参数
     * @param headers 请求头
     * @return {@link Results}    请求返回的结果
     */
    public static Results put(String url, String data, Map<String, String> headers) {
        return HttpKit.put(url, data, headers);
    }

    /**
     * put 请求
     *
     * @param url           请求url
     * @param authorization 授权信息
     * @param serialNumber  公钥证书序列号
     * @param data          请求参数
     * @return {@link Results}    请求返回的结果
     */
    public static Results put(String url, String authorization, String serialNumber, String data) {
        return put(url, data, getHeaders(authorization, serialNumber));
    }

    public static String doPost(String url, Map<String, String> params) {
        return HttpKit.post(url, WxPayKit.toXml(params));
    }

    public static String doPostSsl(String url, Map<String, String> params, String certPath, String certPass) {
        return HttpKit.post(url, WxPayKit.toXml(params), certPath, certPass);
    }

    public static String doPostSslByProtocol(String url, Map<String, String> params, String certPath, String certPass, String protocol) {
        return HttpKit.post(url, WxPayKit.toXml(params), certPath, certPass, protocol);
    }

    public static String doPostSsl(String url, Map<String, String> params, InputStream certFile, String certPass) {
        return HttpKit.post(url, WxPayKit.toXml(params), certFile, certPass);
    }

    public static String doPostSslByProtocol(String url, Map<String, String> params, InputStream certFile, String certPass, String protocol) {
        return HttpKit.post(url, WxPayKit.toXml(params), certFile, certPass, protocol);
    }

    public static String doPostSsl(String url, Map<String, String> params, String certPath) {
        if (params.isEmpty() || !params.containsKey("mch_id")) {
            throw new RuntimeException("请求参数中必须包含 mch_id，如接口参考中不包 mch_id， 请使用其他同名构造方法。");
        }
        String certPass = params.get("mch_id");
        return doPostSsl(url, params, certPath, certPass);
    }

    public static String doPostSslByProtocol(String url, Map<String, String> params, String certPath, String protocol) {
        if (params.isEmpty() || !params.containsKey("mch_id")) {
            throw new RuntimeException("请求参数中必须包含 mch_id，如接口参考中不包 mch_id， 请使用其他同名构造方法。");
        }
        String certPass = params.get("mch_id");
        return doPostSslByProtocol(url, params, certPath, certPass, protocol);
    }

    public static String doPostSsl(String url, Map<String, String> params, InputStream certFile) {
        if (params.isEmpty() || !params.containsKey("mch_id")) {
            throw new RuntimeException("请求参数中必须包含 mch_id，如接口参考中不包 mch_id， 请使用其他同名构造方法。");
        }
        String certPass = params.get("mch_id");
        return doPostSsl(url, params, certFile, certPass);
    }

    public static String doPostSslByProtocol(String url, Map<String, String> params, InputStream certFile, String protocol) {
        if (params.isEmpty() || !params.containsKey("mch_id")) {
            throw new RuntimeException("请求参数中必须包含 mch_id，如接口参考中不包 mch_id， 请使用其他同名构造方法。");
        }
        String certPass = params.get("mch_id");
        return doPostSslByProtocol(url, params, certFile, certPass, protocol);
    }

    public static String doUploadSsl(String url, Map<String, String> params, String certPath, String certPass, String filePath) {
        return HttpKit.upload(url, WxPayKit.toXml(params), certPath, certPass, filePath);
    }

    public static String doUploadSslByProtocol(String url, Map<String, String> params, String certPath, String certPass, String filePath, String protocol) {
        return HttpKit.upload(url, WxPayKit.toXml(params), certPath, certPass, filePath, protocol);
    }

    public static String doUploadSsl(String url, Map<String, String> params, String certPath, String filePath) {
        if (params.isEmpty() || !params.containsKey("mch_id")) {
            throw new RuntimeException("请求参数中必须包含 mch_id，如接口参考中不包 mch_id， 请使用其他同名构造方法。");
        }
        String certPass = params.get("mch_id");
        return doUploadSsl(url, params, certPath, certPass, filePath);
    }

    public static String doUploadSslByProtocol(String url, Map<String, String> params, String certPath, String filePath, String protocol) {
        if (params.isEmpty() || !params.containsKey("mch_id")) {
            throw new RuntimeException("请求参数中必须包含 mch_id，如接口参考中不包 mch_id， 请使用其他同名构造方法。");
        }
        String certPass = params.get("mch_id");
        return doUploadSslByProtocol(url, params, certPath, certPass, filePath, protocol);
    }

    public static Map<String, String> getBaseHeaders(String authorization) {
        String userAgent = String.format(
                "WeChatPay-IJPay-HttpClient/%s (%s) Java/%s",
                WxpayProvider.class.getPackage().getImplementationVersion(),
                OS,
                VERSION == null ? "Unknown" : VERSION);

        Map<String, String> headers = new HashMap<>(5);
        headers.put("Accept", MediaType.APPLICATION_JSON);
        headers.put("Authorization", authorization);
        headers.put("User-Agent", userAgent);
        return headers;
    }

    public static Map<String, String> getHeaders(String authorization, String serialNumber) {
        Map<String, String> headers = getBaseHeaders(authorization);
        headers.put("Content-Type", MediaType.APPLICATION_JSON);
        if (StringKit.isNotEmpty(serialNumber)) {
            headers.put("Wechatpay-Serial", serialNumber);
        }
        return headers;
    }

    public static Map<String, String> getUploadHeaders(String authorization, String serialNumber) {
        Map<String, String> headers = getBaseHeaders(authorization);
        headers.put("Content-Type", "multipart/form-data;boundary=\"boundary\"");
        if (StringKit.isNotEmpty(serialNumber)) {
            headers.put("Wechatpay-Serial", serialNumber);
        }
        return headers;
    }

    /**
     * 构建返回参数
     *
     * @param response {@link Results}
     * @return {@link Map}
     */
    public static Map<String, Object> buildResMap(Results response) {
        if (response == null) {
            return null;
        }
        Map<String, Object> map = new HashMap<>(6);
        String timestamp = response.getHeader("Wechatpay-Timestamp");
        String nonceStr = response.getHeader("Wechatpay-Nonce");
        String serialNo = response.getHeader("Wechatpay-Serial");
        String signature = response.getHeader("Wechatpay-Signature");
        String body = response.getBody();
        int status = response.getStatus();
        map.put("timestamp", timestamp);
        map.put("nonceStr", nonceStr);
        map.put("serialNumber", serialNo);
        map.put("signature", signature);
        map.put("body", body);
        map.put("status", status);
        return map;
    }

}
