/*
 * The MIT License
 *
 * Copyright (c) 2017, aoju.org All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package io.swagger.models.parameters;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import io.swagger.models.properties.*;
import org.aoju.bus.logger.Logger;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * @author Kimi Liu
 * @version 3.0.9
 * @since JDK 1.8
 */
@JsonPropertyOrder({
        "name", "in", "description", "required", "type",
        "items", "collectionFormat", "default", "maximum",
        "exclusiveMaximum", "minimum", "exclusiveMinimum",
        "maxLength", "minLength", "pattern", "maxItems",
        "minItems", "uniqueItems", "multipleOf"
})
public abstract class AbstractSerializableParameter<T extends AbstractSerializableParameter<T>> extends AbstractParameter
        implements SerializableParameter {

    /**
     * See http://json-schema.org/latest/json-schema-validation.html#anchor26
     */
    public Integer maxLength;
    /**
     * See http://json-schema.org/latest/json-schema-validation.html#anchor29
     */
    public Integer minLength;
    /**
     * See http://json-schema.org/latest/json-schema-validation.html#anchor33
     */
    public String pattern;
    /**
     * See http://json-schema.org/latest/json-schema-validation.html#anchor49
     */
    public Boolean uniqueItems;
    /**
     * See http://json-schema.org/latest/json-schema-validation.html#anchor14
     */
    public Number multipleOf;
    protected String type;
    protected String format;
    protected String collectionFormat;
    protected Property items;
    protected Boolean exclusiveMaximum;
    protected BigDecimal maximum;
    protected Boolean exclusiveMinimum;
    protected BigDecimal minimum;
    protected String example;
    protected Boolean allowEmptyValue;
    @JsonIgnore
    protected List<String> _enum;
    @JsonIgnore
    protected String defaultValue;
    private Integer maxItems;
    private Integer minItems;

    public T property(Property property) {
        this.setProperty(property);
        return castThis();
    }

    public T type(String type) {
        this.setType(type);
        return castThis();
    }

    public T format(String format) {
        this.setFormat(format);
        return castThis();
    }

    public T description(String description) {
        this.setDescription(description);
        return castThis();
    }

    public T name(String name) {
        this.setName(name);
        return castThis();
    }

    public T required(boolean required) {
        this.setRequired(required);
        return castThis();
    }

    public T collectionFormat(String collectionFormat) {
        this.setCollectionFormat(collectionFormat);
        return castThis();
    }

    public T example(String example) {
        this.setExample(example);
        return castThis();
    }

    public T allowEmptyValue(Boolean allowEmpty) {
        this.setAllowEmptyValue(allowEmpty);
        return castThis();
    }

    public T readOnly(Boolean readOnly) {
        this.setReadOnly(readOnly);
        return castThis();
    }

    @JsonIgnore
    protected String getDefaultCollectionFormat() {
        return "csv";
    }

    public T items(Property items) {
        this.items = items;
        return castThis();
    }

    public T _enum(List<String> value) {
        this._enum = value;
        return castThis();
    }

    @JsonIgnore
    @Override
    public List<String> getEnum() {
        return _enum;
    }

    @Override
    public void setEnum(List<String> _enum) {
        this._enum = _enum;
    }

    @JsonProperty("enum")
    @Override
    public List<Object> getEnumValue() {
        if (_enum == null) return null;
        if (_enum.isEmpty()) return Collections.emptyList();
        List<Object> oList = new ArrayList<Object>(_enum.size());

        if (BaseIntegerProperty.TYPE.equals(type) || DecimalProperty.TYPE.equals(type)) {
            for (String s : _enum) {
                try {
                    if ("int32".equals(format)) {
                        oList.add(Integer.valueOf(s));
                    } else if ("int64".equals(format)) {
                        oList.add(Long.valueOf(s));
                    } else if ("double".equals(format)) {
                        oList.add(Double.valueOf(s));
                    } else if ("float".equals(format)) {
                        oList.add(Float.valueOf(s));
                    } else if (BaseIntegerProperty.TYPE.equals(type)) {
                        oList.add(Integer.valueOf(s));
                    } else if (DecimalProperty.TYPE.equals(type)) {
                        oList.add(Double.valueOf(s));
                    }
                } catch (NumberFormatException e) {
                    Logger.warn(String.format("Illegal enum value %s for parameter type %s", s, type), e);
                    oList.add(s);
                }
            }
        } else if ((type == null || "array".equals(type)) && items != null) {
            for (String s : _enum) {
                try {
                    if (items instanceof StringProperty) {
                        oList.add(s);
                    } else if (items instanceof IntegerProperty) {
                        oList.add(Integer.valueOf(s));
                    } else if (items instanceof LongProperty) {
                        oList.add(Long.valueOf(s));
                    } else if (items instanceof FloatProperty) {
                        oList.add(Float.valueOf(s));
                    } else if (items instanceof DoubleProperty) {
                        oList.add(Double.valueOf(s));
                    } else if (items instanceof BaseIntegerProperty) {
                        oList.add(Integer.valueOf(s));
                    } else if (items instanceof DecimalProperty) {
                        oList.add(Double.valueOf(s));
                    } else {
                        oList.add(s);
                    }
                } catch (NumberFormatException e) {
                    Logger.warn(String.format("Illegal enum value %s for parameter type %s", s, type), e);
                    oList.add(s);
                }
            }

        } else {
            for (String s : _enum) {
                oList.add(s);
            }
        }
        return oList;
    }

    @Override
    public void setEnumValue(List<?> enumValue) {
        if (enumValue == null) {
            this._enum = null;
            return;
        }
        if (enumValue.isEmpty()) {
            this._enum = Collections.emptyList();
            return;
        }
        List<String> sList = new ArrayList<String>(enumValue.size());
        for (Object item : enumValue) {
            sList.add(item.toString());
        }
        this._enum = sList;
    }

    @Override
    public Property getItems() {
        return items;
    }

    @Override
    public void setItems(Property items) {
        this.items = items;
    }

    @Override
    public String getFormat() {
        return format;
    }

    @Override
    public void setFormat(String format) {
        this.format = format;
    }

    @Override
    public String getType() {
        return type;
    }

    @Override
    public void setType(String type) {
        this.type = type;
        if (ArrayProperty.isType(type)) {
            if (getCollectionFormat() == null) {
                setCollectionFormat(getDefaultCollectionFormat());
            }
        } else {
            setCollectionFormat(null);
        }
    }

    @Override
    public String getCollectionFormat() {
        return collectionFormat;
    }

    @Override
    public void setCollectionFormat(String collectionFormat) {
        this.collectionFormat = collectionFormat;
    }

    public void setProperty(Property property) {
        setType(property.getType());
        this.format = property.getFormat();
        if (property instanceof StringProperty) {
            final StringProperty string = (StringProperty) property;
            setEnum(string.getEnum());
        } else if (property instanceof IntegerProperty) {
            setEnumValue(((IntegerProperty) property).getEnum());
        } else if (property instanceof LongProperty) {
            setEnumValue(((LongProperty) property).getEnum());
        } else if (property instanceof FloatProperty) {
            setEnumValue(((FloatProperty) property).getEnum());
        } else if (property instanceof DoubleProperty) {
            setEnumValue(((DoubleProperty) property).getEnum());
        } else if (property instanceof ArrayProperty) {
            final ArrayProperty array = (ArrayProperty) property;
            setItems(array.getItems());
        }
    }

    public Object getDefaultValue() {
        if (defaultValue == null || defaultValue.isEmpty()) {
            return null;
        }

        // don't return a default value if types fail to convert
        try {
            if ("integer".equals(this.type)) {
                return new Integer(defaultValue);
            }
            if ("number".equals(this.type)) {
                return new BigDecimal(defaultValue);
            }
        } catch (Exception e) {
            return null;
        }

        return defaultValue;
    }

    public void setDefaultValue(String defaultValue) {
        this.defaultValue = defaultValue;
    }

    public Object getDefault() {
        if (defaultValue == null || defaultValue.isEmpty()) {
            return null;
        }
        try {
            if (BaseIntegerProperty.TYPE.equals(type)) {
                return Long.valueOf(defaultValue);
            } else if (DecimalProperty.TYPE.equals(type)) {
                return Double.valueOf(defaultValue);
            } else if (BooleanProperty.TYPE.equals(type)) {
                if ("true".equalsIgnoreCase(defaultValue) || "false".equalsIgnoreCase(defaultValue)) {
                    return Boolean.valueOf(defaultValue);
                }
            }
        } catch (NumberFormatException e) {
            Logger.warn(String.format("Illegal DefaultValue %s for parameter type %s", defaultValue, type), e);
        }
        return defaultValue;
    }

    public void setDefault(Object defaultValue) {
        this.defaultValue = defaultValue == null ? null : defaultValue.toString();
    }

    @Override
    public void setExclusiveMaximum(Boolean exclusiveMaximum) {
        this.exclusiveMaximum = exclusiveMaximum;
    }

    @Override
    public BigDecimal getMaximum() {
        return maximum;
    }

    @Override
    public void setMaximum(BigDecimal maximum) {
        this.maximum = maximum;
    }

    @Override
    public Boolean isExclusiveMinimum() {
        return exclusiveMinimum;
    }

    @Override
    public void setExclusiveMinimum(Boolean exclusiveMinimum) {
        this.exclusiveMinimum = exclusiveMinimum;
    }

    @Override
    public BigDecimal getMinimum() {
        return minimum;
    }

    @Override
    public void setMinimum(BigDecimal minimum) {
        this.minimum = minimum;
    }

    @Override
    public Integer getMaxItems() {
        return maxItems;
    }

    @Override
    public void setMaxItems(Integer maxItems) {
        this.maxItems = maxItems;
    }

    @Override
    public Integer getMinItems() {
        return minItems;
    }

    @Override
    public void setMinItems(Integer minItems) {
        this.minItems = minItems;
    }

    @Override
    public Boolean getAllowEmptyValue() {
        return allowEmptyValue;
    }

    @Override
    public void setAllowEmptyValue(Boolean allowEmptyValue) {
        this.allowEmptyValue = allowEmptyValue;
    }

    @JsonProperty("x-example")
    public Object getExample() {
        if (example == null || example.isEmpty()) {
            return null;
        }
        try {
            if (BaseIntegerProperty.TYPE.equals(type)) {
                return Long.valueOf(example);
            } else if (DecimalProperty.TYPE.equals(type)) {
                return Double.valueOf(example);
            } else if (BooleanProperty.TYPE.equals(type)) {
                if ("true".equalsIgnoreCase(example) || "false".equalsIgnoreCase(defaultValue)) {
                    return Boolean.valueOf(example);
                }
            }
        } catch (NumberFormatException e) {
            Logger.warn(String.format("Illegal DefaultValue %s for parameter type %s", defaultValue, type), e);
        }
        return example;
    }

    public void setExample(String example) {
        this.example = example;
    }

    @Override
    public Integer getMaxLength() {
        return maxLength;
    }

    @Override
    public void setMaxLength(Integer maxLength) {
        this.maxLength = maxLength;
    }

    @Override
    public Integer getMinLength() {
        return minLength;
    }

    @Override
    public void setMinLength(Integer minLength) {
        this.minLength = minLength;
    }

    @Override
    public String getPattern() {
        return pattern;
    }

    @Override
    public void setPattern(String pattern) {
        this.pattern = pattern;
    }

    @Override
    public Boolean isUniqueItems() {
        return uniqueItems;
    }

    @Override
    public void setUniqueItems(Boolean uniqueItems) {
        this.uniqueItems = uniqueItems;
    }

    @Override
    public Number getMultipleOf() {
        return multipleOf;
    }

    @Override
    public void setMultipleOf(Number multipleOf) {
        this.multipleOf = multipleOf;
    }

    @Override
    public Boolean isExclusiveMaximum() {
        return exclusiveMaximum;
    }

    @JsonIgnore
    private T castThis() {
        final T result = (T) this;
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) return true;
        if (!super.equals(obj)) return false;
        if (getClass() != obj.getClass()) return false;
        AbstractSerializableParameter<?> other = (AbstractSerializableParameter<?>) obj;
        if (_enum == null) {
            if (other._enum != null) return false;
        } else if (!_enum.equals(other._enum)) return false;
        if (collectionFormat == null) {
            if (other.collectionFormat != null) return false;
        } else if (!collectionFormat.equals(other.collectionFormat)) return false;
        if (defaultValue == null) {
            if (other.defaultValue != null) return false;
        } else if (!defaultValue.equals(other.defaultValue)) return false;
        if (example == null) {
            if (other.example != null) return false;
        } else if (!example.equals(other.example)) return false;
        if (exclusiveMaximum == null) {
            if (other.exclusiveMaximum != null) return false;
        } else if (!exclusiveMaximum.equals(other.exclusiveMaximum)) return false;
        if (exclusiveMinimum == null) {
            if (other.exclusiveMinimum != null) return false;
        } else if (!exclusiveMinimum.equals(other.exclusiveMinimum)) return false;
        if (format == null) {
            if (other.format != null) return false;
        } else if (!format.equals(other.format)) return false;
        if (items == null) {
            if (other.items != null) return false;
        } else if (!items.equals(other.items)) return false;
        if (maxItems == null) {
            if (other.maxItems != null) return false;
        } else if (!maxItems.equals(other.maxItems)) return false;
        if (maxLength == null) {
            if (other.maxLength != null) return false;
        } else if (!maxLength.equals(other.maxLength)) return false;
        if (maximum == null) {
            if (other.maximum != null) return false;
        } else if (!maximum.equals(other.maximum)) return false;
        if (minItems == null) {
            if (other.minItems != null) return false;
        } else if (!minItems.equals(other.minItems)) return false;
        if (minLength == null) {
            if (other.minLength != null) return false;
        } else if (!minLength.equals(other.minLength)) return false;
        if (minimum == null) {
            if (other.minimum != null) return false;
        } else if (!minimum.equals(other.minimum)) return false;
        if (multipleOf == null) {
            if (other.multipleOf != null) return false;
        } else if (!multipleOf.equals(other.multipleOf)) return false;
        if (pattern == null) {
            if (other.pattern != null) return false;
        } else if (!pattern.equals(other.pattern)) return false;
        if (type == null) {
            if (other.type != null) return false;
        } else if (!type.equals(other.type)) return false;
        if (uniqueItems == null) {
            if (other.uniqueItems != null) {
                return false;
            }
        } else if (!uniqueItems.equals(other.uniqueItems)) {
            return false;
        }
        if (allowEmptyValue != null ? !allowEmptyValue.equals(other.allowEmptyValue) : other.allowEmptyValue != null) {
            return false;
        }
        return true;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = super.hashCode();
        result = prime * result + ((_enum == null) ? 0 : _enum.hashCode());
        result = prime * result + ((collectionFormat == null) ? 0 : collectionFormat.hashCode());
        result = prime * result + ((defaultValue == null) ? 0 : defaultValue.hashCode());
        result = prime * result + ((example == null) ? 0 : example.hashCode());
        result = prime * result + ((exclusiveMaximum == null) ? 0 : exclusiveMaximum.hashCode());
        result = prime * result + ((exclusiveMinimum == null) ? 0 : exclusiveMinimum.hashCode());
        result = prime * result + ((format == null) ? 0 : format.hashCode());
        result = prime * result + ((items == null) ? 0 : items.hashCode());
        result = prime * result + ((maxItems == null) ? 0 : maxItems.hashCode());
        result = prime * result + ((maxLength == null) ? 0 : maxLength.hashCode());
        result = prime * result + ((maximum == null) ? 0 : maximum.hashCode());
        result = prime * result + ((minItems == null) ? 0 : minItems.hashCode());
        result = prime * result + ((minLength == null) ? 0 : minLength.hashCode());
        result = prime * result + ((minimum == null) ? 0 : minimum.hashCode());
        result = prime * result + ((multipleOf == null) ? 0 : multipleOf.hashCode());
        result = prime * result + ((pattern == null) ? 0 : pattern.hashCode());
        result = prime * result + ((type == null) ? 0 : type.hashCode());
        result = prime * result + ((uniqueItems == null) ? 0 : uniqueItems.hashCode());
        result = prime * result + (allowEmptyValue != null ? allowEmptyValue.hashCode() : 0);
        return result;
    }

}
