/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.core.memory;

import java.io.EOFException;
import java.io.IOException;
import java.io.UTFDataFormatException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;
import org.apache.flink.core.memory.DataInputView;
import org.apache.flink.core.memory.DataOutputView;
import org.apache.flink.core.memory.MemorySegment;
import org.apache.flink.core.memory.MemorySegmentWritable;
import org.apache.flink.core.memory.MemoryUtils;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sun.misc.Unsafe;

public class DataOutputSerializer
implements DataOutputView,
MemorySegmentWritable {
    private static final Logger LOG = LoggerFactory.getLogger(DataOutputSerializer.class);
    private static final int PRUNE_BUFFER_THRESHOLD = 0x500000;
    private final byte[] startBuffer;
    private byte[] buffer;
    private int position;
    private ByteBuffer wrapper;
    private static final Unsafe UNSAFE = MemoryUtils.UNSAFE;
    private static final long BASE_OFFSET = UNSAFE.arrayBaseOffset(byte[].class);
    private static final boolean LITTLE_ENDIAN = MemoryUtils.NATIVE_BYTE_ORDER == ByteOrder.LITTLE_ENDIAN;

    public DataOutputSerializer(int startSize) {
        if (startSize < 1) {
            throw new IllegalArgumentException();
        }
        this.startBuffer = new byte[startSize];
        this.buffer = this.startBuffer;
        this.wrapper = ByteBuffer.wrap(this.buffer);
    }

    public ByteBuffer wrapAsByteBuffer() {
        this.wrapper.position(0);
        this.wrapper.limit(this.position);
        return this.wrapper;
    }

    @Deprecated
    public byte[] getByteArray() {
        return this.getSharedBuffer();
    }

    public byte[] getSharedBuffer() {
        return this.buffer;
    }

    public byte[] getCopyOfBuffer() {
        return Arrays.copyOf(this.buffer, this.position);
    }

    public void clear() {
        this.position = 0;
    }

    public int length() {
        return this.position;
    }

    public void pruneBuffer() {
        this.clear();
        if (this.buffer.length > 0x500000) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Releasing serialization buffer of " + this.buffer.length + " bytes.");
            }
            this.buffer = this.startBuffer;
            this.wrapper = ByteBuffer.wrap(this.buffer);
        }
    }

    public String toString() {
        return String.format("[pos=%d cap=%d]", this.position, this.buffer.length);
    }

    @Override
    public void write(int b) throws IOException {
        if (this.position >= this.buffer.length) {
            this.resize(1);
        }
        this.buffer[this.position++] = (byte)(b & 0xFF);
    }

    @Override
    public void write(byte[] b) throws IOException {
        this.write(b, 0, b.length);
    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        if (len < 0 || off > b.length - len) {
            throw new ArrayIndexOutOfBoundsException();
        }
        if (this.position > this.buffer.length - len) {
            this.resize(len);
        }
        System.arraycopy(b, off, this.buffer, this.position, len);
        this.position += len;
    }

    @Override
    public void write(MemorySegment segment, int off, int len) throws IOException {
        if (len < 0 || off > segment.size() - len) {
            throw new ArrayIndexOutOfBoundsException();
        }
        if (this.position > this.buffer.length - len) {
            this.resize(len);
        }
        segment.get(off, this.buffer, this.position, len);
        this.position += len;
    }

    @Override
    public void writeBoolean(boolean v) throws IOException {
        this.write(v ? 1 : 0);
    }

    @Override
    public void writeByte(int v) throws IOException {
        this.write(v);
    }

    @Override
    public void writeBytes(String s) throws IOException {
        int sLen = s.length();
        if (this.position >= this.buffer.length - sLen) {
            this.resize(sLen);
        }
        for (int i = 0; i < sLen; ++i) {
            this.writeByte(s.charAt(i));
        }
        this.position += sLen;
    }

    @Override
    public void writeChar(int v) throws IOException {
        if (this.position >= this.buffer.length - 1) {
            this.resize(2);
        }
        this.buffer[this.position++] = (byte)(v >> 8);
        this.buffer[this.position++] = (byte)v;
    }

    @Override
    public void writeChars(String s) throws IOException {
        int sLen = s.length();
        if (this.position >= this.buffer.length - 2 * sLen) {
            this.resize(2 * sLen);
        }
        for (int i = 0; i < sLen; ++i) {
            this.writeChar(s.charAt(i));
        }
    }

    @Override
    public void writeDouble(double v) throws IOException {
        this.writeLong(Double.doubleToLongBits(v));
    }

    @Override
    public void writeFloat(float v) throws IOException {
        this.writeInt(Float.floatToIntBits(v));
    }

    @Override
    public void writeInt(int v) throws IOException {
        if (this.position >= this.buffer.length - 3) {
            this.resize(4);
        }
        if (LITTLE_ENDIAN) {
            v = Integer.reverseBytes(v);
        }
        UNSAFE.putInt(this.buffer, BASE_OFFSET + (long)this.position, v);
        this.position += 4;
    }

    @Override
    public void writeLong(long v) throws IOException {
        if (this.position >= this.buffer.length - 7) {
            this.resize(8);
        }
        if (LITTLE_ENDIAN) {
            v = Long.reverseBytes(v);
        }
        UNSAFE.putLong(this.buffer, BASE_OFFSET + (long)this.position, v);
        this.position += 8;
    }

    @Override
    public void writeShort(int v) throws IOException {
        if (this.position >= this.buffer.length - 1) {
            this.resize(2);
        }
        this.buffer[this.position++] = (byte)(v >>> 8 & 0xFF);
        this.buffer[this.position++] = (byte)(v & 0xFF);
    }

    @Override
    public void writeUTF(String str) throws IOException {
        int i;
        char c;
        int strlen = str.length();
        int utflen = 0;
        for (int i2 = 0; i2 < strlen; ++i2) {
            c = str.charAt(i2);
            if (c >= '\u0001' && c <= '\u007f') {
                ++utflen;
                continue;
            }
            if (c > '\u07ff') {
                utflen += 3;
                continue;
            }
            utflen += 2;
        }
        if (utflen > 65535) {
            throw new UTFDataFormatException("Encoded string is too long: " + utflen);
        }
        if (this.position > this.buffer.length - utflen - 2) {
            this.resize(utflen + 2);
        }
        byte[] bytearr = this.buffer;
        int count = this.position;
        bytearr[count++] = (byte)(utflen >>> 8 & 0xFF);
        bytearr[count++] = (byte)(utflen & 0xFF);
        for (i = 0; i < strlen && (c = str.charAt(i)) >= '\u0001' && c <= '\u007f'; ++i) {
            bytearr[count++] = (byte)c;
        }
        while (i < strlen) {
            c = str.charAt(i);
            if (c >= '\u0001' && c <= '\u007f') {
                bytearr[count++] = (byte)c;
            } else if (c > '\u07ff') {
                bytearr[count++] = (byte)(0xE0 | c >> 12 & 0xF);
                bytearr[count++] = (byte)(0x80 | c >> 6 & 0x3F);
                bytearr[count++] = (byte)(0x80 | c & 0x3F);
            } else {
                bytearr[count++] = (byte)(0xC0 | c >> 6 & 0x1F);
                bytearr[count++] = (byte)(0x80 | c & 0x3F);
            }
            ++i;
        }
        this.position = count;
    }

    private void resize(int minCapacityAdd) throws IOException {
        byte[] nb;
        int newLen = Math.max(this.buffer.length * 2, this.buffer.length + minCapacityAdd);
        try {
            nb = new byte[newLen];
        }
        catch (NegativeArraySizeException e) {
            throw new IOException("Serialization failed because the record length would exceed 2GB (max addressable array size in Java).");
        }
        catch (OutOfMemoryError e) {
            if (newLen > this.buffer.length + minCapacityAdd) {
                newLen = this.buffer.length + minCapacityAdd;
                try {
                    nb = new byte[newLen];
                }
                catch (OutOfMemoryError ee) {
                    throw new IOException("Failed to serialize element. Serialized size (> " + newLen + " bytes) exceeds JVM heap space", ee);
                }
            }
            throw new IOException("Failed to serialize element. Serialized size (> " + newLen + " bytes) exceeds JVM heap space", e);
        }
        System.arraycopy(this.buffer, 0, nb, 0, this.position);
        this.buffer = nb;
        this.wrapper = ByteBuffer.wrap(this.buffer);
    }

    @Override
    public void skipBytesToWrite(int numBytes) throws IOException {
        if (this.buffer.length - this.position < numBytes) {
            throw new EOFException("Could not skip " + numBytes + " bytes.");
        }
        this.position += numBytes;
    }

    @Override
    public void write(DataInputView source, int numBytes) throws IOException {
        if (this.buffer.length - this.position < numBytes) {
            throw new EOFException("Could not write " + numBytes + " bytes. Buffer overflow.");
        }
        source.readFully(this.buffer, this.position, numBytes);
        this.position += numBytes;
    }

    public void setPosition(int position) {
        Preconditions.checkArgument(position >= 0 && position <= this.position, "Position out of bounds.");
        this.position = position;
    }
}

