/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.memory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.ConcurrentModificationException;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import javax.annotation.Nullable;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.core.memory.MemorySegment;
import org.apache.flink.core.memory.MemorySegmentFactory;
import org.apache.flink.core.memory.MemoryType;
import org.apache.flink.runtime.memory.MemoryAllocationException;
import org.apache.flink.runtime.memory.MemoryReservationException;
import org.apache.flink.runtime.memory.OpaqueMemoryResource;
import org.apache.flink.runtime.memory.SharedResources;
import org.apache.flink.runtime.util.KeyedBudgetManager;
import org.apache.flink.util.MathUtils;
import org.apache.flink.util.Preconditions;
import org.apache.flink.util.function.LongFunctionWithException;
import org.apache.flink.util.function.ThrowingRunnable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MemoryManager {
    private static final Logger LOG = LoggerFactory.getLogger(MemoryManager.class);
    public static final int DEFAULT_PAGE_SIZE = 32768;
    public static final int MIN_PAGE_SIZE = 4096;
    private final Map<Object, Set<MemorySegment>> allocatedSegments;
    private final Map<Object, Map<MemoryType, Long>> reservedMemory;
    private final KeyedBudgetManager<MemoryType> budgetByType;
    private final SharedResources sharedResources;
    private volatile boolean isShutDown;

    public MemoryManager(Map<MemoryType, Long> memorySizeByType, int pageSize) {
        for (Map.Entry<MemoryType, Long> sizeForType : memorySizeByType.entrySet()) {
            MemoryManager.sanityCheck(sizeForType.getValue(), pageSize, sizeForType.getKey());
        }
        this.allocatedSegments = new ConcurrentHashMap<Object, Set<MemorySegment>>();
        this.reservedMemory = new ConcurrentHashMap<Object, Map<MemoryType, Long>>();
        this.budgetByType = new KeyedBudgetManager<MemoryType>(memorySizeByType, pageSize);
        this.sharedResources = new SharedResources();
        MemoryManager.verifyIntTotalNumberOfPages(memorySizeByType, this.budgetByType.maxTotalNumberOfPages());
        LOG.debug("Initialized MemoryManager with total memory size {} ({}), page size {}.", new Object[]{this.budgetByType.totalAvailableBudget(), memorySizeByType, pageSize});
    }

    private static void sanityCheck(long memorySize, int pageSize, MemoryType memoryType) {
        Preconditions.checkNotNull((Object)memoryType);
        Preconditions.checkArgument((memorySize >= 0L ? 1 : 0) != 0, (Object)"Size of total memory must be non-negative.");
        Preconditions.checkArgument((pageSize >= 4096 ? 1 : 0) != 0, (String)"The page size must be at least %d bytes.", (Object[])new Object[]{4096});
        Preconditions.checkArgument((boolean)MathUtils.isPowerOf2((long)pageSize), (Object)"The given page size is not a power of two.");
    }

    private static void verifyIntTotalNumberOfPages(Map<MemoryType, Long> memorySizeByType, long numberOfPagesLong) {
        Preconditions.checkArgument((numberOfPagesLong <= Integer.MAX_VALUE ? 1 : 0) != 0, (String)"The given number of memory bytes (%d: %s) corresponds to more than MAX_INT pages.", (Object[])new Object[]{numberOfPagesLong, memorySizeByType});
    }

    public void shutdown() {
        if (!this.isShutDown) {
            this.isShutDown = true;
            this.reservedMemory.clear();
            this.budgetByType.releaseAll();
            for (Set<MemorySegment> segments : this.allocatedSegments.values()) {
                for (MemorySegment seg : segments) {
                    seg.free();
                }
                segments.clear();
            }
            this.allocatedSegments.clear();
        }
    }

    @VisibleForTesting
    public boolean isShutdown() {
        return this.isShutDown;
    }

    public boolean verifyEmpty() {
        return this.budgetByType.totalAvailableBudget() == this.budgetByType.maxTotalBudget();
    }

    @Deprecated
    public List<MemorySegment> allocatePages(Object owner, int numPages) throws MemoryAllocationException {
        ArrayList<MemorySegment> segments = new ArrayList<MemorySegment>(numPages);
        this.allocatePages(owner, segments, numPages);
        return segments;
    }

    @Deprecated
    public void allocatePages(Object owner, Collection<MemorySegment> target, int numberOfPages) throws MemoryAllocationException {
        this.allocatePages(AllocationRequest.newBuilder(owner).ofAllTypes().numberOfPages(numberOfPages).withOutput(target).build());
    }

    public Collection<MemorySegment> allocatePages(AllocationRequest request) throws MemoryAllocationException {
        KeyedBudgetManager.AcquisitionResult<MemoryType> acquiredBudget;
        Object owner = request.getOwner();
        Collection target = request.output;
        int numberOfPages = request.getNumberOfPages();
        Preconditions.checkNotNull((Object)owner, (String)"The memory owner must not be null.");
        Preconditions.checkState((!this.isShutDown ? 1 : 0) != 0, (Object)"Memory manager has been shut down.");
        if (target instanceof ArrayList) {
            ((ArrayList)target).ensureCapacity(numberOfPages);
        }
        if ((acquiredBudget = this.budgetByType.acquirePagedBudget(request.getTypes(), numberOfPages)).isFailure()) {
            throw new MemoryAllocationException(String.format("Could not allocate %d pages. Only %d pages are remaining.", numberOfPages, acquiredBudget.getTotalAvailableForAllQueriedKeys()));
        }
        this.allocatedSegments.compute(owner, (o, currentSegmentsForOwner) -> {
            Set segmentsForOwner = currentSegmentsForOwner == null ? new HashSet(numberOfPages) : currentSegmentsForOwner;
            for (MemoryType memoryType : acquiredBudget.getAcquiredPerKey().keySet()) {
                for (long i = acquiredBudget.getAcquiredPerKey().get(memoryType).longValue(); i > 0L; --i) {
                    MemorySegment segment = this.allocateManagedSegment(memoryType, owner);
                    target.add(segment);
                    segmentsForOwner.add(segment);
                }
            }
            return segmentsForOwner;
        });
        Preconditions.checkState((!this.isShutDown ? 1 : 0) != 0, (Object)"Memory manager has been concurrently shut down.");
        return target;
    }

    public void release(MemorySegment segment) {
        Preconditions.checkState((!this.isShutDown ? 1 : 0) != 0, (Object)"Memory manager has been shut down.");
        if (segment == null || segment.getOwner() == null) {
            return;
        }
        try {
            this.allocatedSegments.computeIfPresent(segment.getOwner(), (o, segsForOwner) -> {
                segment.free();
                if (segsForOwner.remove(segment)) {
                    this.budgetByType.releasePageForKey(MemoryManager.getSegmentType(segment));
                }
                return segsForOwner.isEmpty() ? null : segsForOwner;
            });
        }
        catch (Throwable t) {
            throw new RuntimeException("Error removing book-keeping reference to allocated memory segment.", t);
        }
    }

    public void release(Collection<MemorySegment> segments) {
        if (segments == null) {
            return;
        }
        Preconditions.checkState((!this.isShutDown ? 1 : 0) != 0, (Object)"Memory manager has been shut down.");
        EnumMap<MemoryType, Long> releasedMemory = new EnumMap<MemoryType, Long>(MemoryType.class);
        boolean successfullyReleased = false;
        do {
            Iterator<MemorySegment> segmentsIterator = segments.iterator();
            try {
                MemorySegment segment = null;
                while (segment == null && segmentsIterator.hasNext()) {
                    segment = segmentsIterator.next();
                }
                while (segment != null) {
                    segment = this.releaseSegmentsForOwnerUntilNextOwner(segment, segmentsIterator, releasedMemory);
                }
                segments.clear();
                successfullyReleased = true;
            }
            catch (ConcurrentModificationException | NoSuchElementException runtimeException) {
                // empty catch block
            }
        } while (!successfullyReleased);
        this.budgetByType.releaseBudgetForKeys(releasedMemory);
    }

    private MemorySegment releaseSegmentsForOwnerUntilNextOwner(MemorySegment firstSeg, Iterator<MemorySegment> segmentsIterator, EnumMap<MemoryType, Long> releasedMemory) {
        AtomicReference nextOwnerMemorySegment = new AtomicReference();
        Object owner = firstSeg.getOwner();
        this.allocatedSegments.compute(owner, (o, segsForOwner) -> {
            this.freeSegment(firstSeg, (Collection<MemorySegment>)segsForOwner, releasedMemory);
            while (segmentsIterator.hasNext()) {
                MemorySegment segment = (MemorySegment)segmentsIterator.next();
                try {
                    if (segment == null || segment.isFreed()) continue;
                    Object nextOwner = segment.getOwner();
                    if (nextOwner != owner) {
                        nextOwnerMemorySegment.set(segment);
                        break;
                    }
                    this.freeSegment(segment, (Collection<MemorySegment>)segsForOwner, releasedMemory);
                }
                catch (Throwable t) {
                    throw new RuntimeException("Error removing book-keeping reference to allocated memory segment.", t);
                }
            }
            return segsForOwner == null || segsForOwner.isEmpty() ? null : segsForOwner;
        });
        return (MemorySegment)nextOwnerMemorySegment.get();
    }

    private void freeSegment(MemorySegment segment, @Nullable Collection<MemorySegment> segments, EnumMap<MemoryType, Long> releasedMemory) {
        segment.free();
        if (segments != null && segments.remove(segment)) {
            this.releaseSegment(segment, releasedMemory);
        }
    }

    public void releaseAll(Object owner) {
        if (owner == null) {
            return;
        }
        Preconditions.checkState((!this.isShutDown ? 1 : 0) != 0, (Object)"Memory manager has been shut down.");
        Set<MemorySegment> segments = this.allocatedSegments.remove(owner);
        if (segments == null || segments.isEmpty()) {
            return;
        }
        EnumMap<MemoryType, Long> releasedMemory = new EnumMap<MemoryType, Long>(MemoryType.class);
        for (MemorySegment segment : segments) {
            segment.free();
            this.releaseSegment(segment, releasedMemory);
        }
        this.budgetByType.releaseBudgetForKeys(releasedMemory);
        segments.clear();
    }

    public void reserveMemory(Object owner, MemoryType memoryType, long size) throws MemoryReservationException {
        this.checkMemoryReservationPreconditions(owner, memoryType, size);
        if (size == 0L) {
            return;
        }
        long acquiredMemory = this.budgetByType.acquireBudgetForKey(memoryType, size);
        if (acquiredMemory < size) {
            throw new MemoryReservationException(String.format("Could not allocate %d bytes. Only %d bytes are remaining.", size, acquiredMemory));
        }
        this.reservedMemory.compute(owner, (o, reservations) -> {
            EnumMap<MemoryType, Long> newReservations = reservations;
            if (reservations == null) {
                newReservations = new EnumMap<MemoryType, Long>(MemoryType.class);
                newReservations.put(memoryType, size);
            } else {
                reservations.compute(memoryType, (mt, currentlyReserved) -> currentlyReserved == null ? size : currentlyReserved + size);
            }
            return newReservations;
        });
        Preconditions.checkState((!this.isShutDown ? 1 : 0) != 0, (Object)"Memory manager has been concurrently shut down.");
    }

    public void releaseMemory(Object owner, MemoryType memoryType, long size) {
        this.checkMemoryReservationPreconditions(owner, memoryType, size);
        if (size == 0L) {
            return;
        }
        this.reservedMemory.compute(owner, (o, reservations) -> {
            if (reservations != null) {
                reservations.compute(memoryType, (mt, currentlyReserved) -> {
                    long newReservedMemory = 0L;
                    if (currentlyReserved != null) {
                        if (currentlyReserved < size) {
                            LOG.warn("Trying to release more memory {} than it was reserved {} so far for the owner {}", new Object[]{size, currentlyReserved, owner});
                        }
                        newReservedMemory = this.releaseAndCalculateReservedMemory(size, memoryType, (long)currentlyReserved);
                    }
                    return newReservedMemory == 0L ? null : Long.valueOf(newReservedMemory);
                });
            }
            return reservations == null || reservations.isEmpty() ? null : reservations;
        });
    }

    private long releaseAndCalculateReservedMemory(long memoryToFree, MemoryType memoryType, long currentlyReserved) {
        long effectiveMemoryToRelease = Math.min(currentlyReserved, memoryToFree);
        this.budgetByType.releaseBudgetForKey(memoryType, effectiveMemoryToRelease);
        return currentlyReserved - effectiveMemoryToRelease;
    }

    private void checkMemoryReservationPreconditions(Object owner, MemoryType memoryType, long size) {
        Preconditions.checkNotNull((Object)owner, (String)"The memory owner must not be null.");
        Preconditions.checkNotNull((Object)memoryType, (String)"The memory type must not be null.");
        Preconditions.checkState((!this.isShutDown ? 1 : 0) != 0, (Object)"Memory manager has been shut down.");
        Preconditions.checkArgument((size >= 0L ? 1 : 0) != 0, (String)"The memory size (%s) has to have non-negative size", (Object[])new Object[]{size});
    }

    public void releaseAllMemory(Object owner, MemoryType memoryType) {
        this.checkMemoryReservationPreconditions(owner, memoryType, 0L);
        this.reservedMemory.compute(owner, (o, reservations) -> {
            Long size;
            if (reservations != null && (size = (Long)reservations.remove(memoryType)) != null) {
                this.budgetByType.releaseBudgetForKey(memoryType, size);
            }
            return reservations == null || reservations.isEmpty() ? null : reservations;
        });
    }

    public <T extends AutoCloseable> OpaqueMemoryResource<T> getSharedMemoryResourceForManagedMemory(String type, LongFunctionWithException<T, Exception> initializer) throws Exception {
        return this.getSharedMemoryResourceForManagedMemory(type, initializer, 1.0);
    }

    public <T extends AutoCloseable> OpaqueMemoryResource<T> getSharedMemoryResourceForManagedMemory(String type, LongFunctionWithException<T, Exception> initializer, double fractionToInitializeWith) throws Exception {
        long numBytes = this.computeMemorySize(fractionToInitializeWith);
        LongFunctionWithException reserveAndInitialize = size -> {
            try {
                this.reserveMemory(type, MemoryType.OFF_HEAP, size);
            }
            catch (MemoryReservationException e) {
                throw new MemoryAllocationException("Could not created the shared memory resource of size " + size + ". Not enough memory left to reserve from the slot's managed memory.", e);
            }
            return (AutoCloseable)initializer.apply(size);
        };
        Consumer<Long> releaser = size -> this.releaseMemory(type, MemoryType.OFF_HEAP, (long)size);
        Object leaseHolder = new Object();
        SharedResources.ResourceAndSize resource = this.sharedResources.getOrAllocateSharedResource(type, leaseHolder, reserveAndInitialize, numBytes);
        long size2 = resource.size();
        ThrowingRunnable disposer = () -> this.sharedResources.release(type, leaseHolder, releaser);
        return new OpaqueMemoryResource(resource.resourceHandle(), size2, (ThrowingRunnable<Exception>)disposer);
    }

    public <T extends AutoCloseable> OpaqueMemoryResource<T> getExternalSharedMemoryResource(String type, LongFunctionWithException<T, Exception> initializer, long numBytes) throws Exception {
        Object leaseHolder = new Object();
        SharedResources.ResourceAndSize<T> resource = this.sharedResources.getOrAllocateSharedResource(type, leaseHolder, initializer, numBytes);
        ThrowingRunnable disposer = () -> this.sharedResources.release(type, leaseHolder);
        return new OpaqueMemoryResource<T>(resource.resourceHandle(), resource.size(), (ThrowingRunnable<Exception>)disposer);
    }

    public int getPageSize() {
        return (int)this.budgetByType.getDefaultPageSize();
    }

    public long getMemorySize() {
        return this.budgetByType.maxTotalBudget();
    }

    public long getMemorySizeByType(MemoryType memoryType) {
        return this.budgetByType.maxTotalBudgetForKey(memoryType);
    }

    public long availableMemory(MemoryType memoryType) {
        return this.budgetByType.availableBudgetForKey(memoryType);
    }

    public int computeNumberOfPages(double fraction) {
        if (fraction <= 0.0 || fraction > 1.0) {
            throw new IllegalArgumentException("The fraction of memory to allocate must within (0, 1].");
        }
        return (int)((double)this.budgetByType.maxTotalNumberOfPages() * fraction);
    }

    public long computeMemorySize(double fraction) {
        Preconditions.checkArgument((fraction > 0.0 && fraction <= 1.0 ? 1 : 0) != 0, (String)"The fraction of memory to allocate must within (0, 1], was: %s", (Object[])new Object[]{fraction});
        return (long)((double)this.budgetByType.maxTotalBudget() * fraction);
    }

    private MemorySegment allocateManagedSegment(MemoryType memoryType, Object owner) {
        switch (memoryType) {
            case HEAP: {
                return MemorySegmentFactory.allocateUnpooledSegment((int)this.getPageSize(), (Object)owner);
            }
            case OFF_HEAP: {
                return MemorySegmentFactory.allocateOffHeapUnsafeMemory((int)this.getPageSize(), (Object)owner);
            }
        }
        throw new IllegalArgumentException("unrecognized memory type: " + memoryType);
    }

    private void releaseSegment(MemorySegment segment, EnumMap<MemoryType, Long> releasedMemory) {
        releasedMemory.compute(MemoryManager.getSegmentType(segment), (t, v) -> v == null ? (long)this.getPageSize() : v + (long)this.getPageSize());
    }

    private static MemoryType getSegmentType(MemorySegment segment) {
        return segment.isOffHeap() ? MemoryType.OFF_HEAP : MemoryType.HEAP;
    }

    public static MemoryManager forDefaultPageSize(long size) {
        HashMap<MemoryType, Long> memorySizes = new HashMap<MemoryType, Long>();
        memorySizes.put(MemoryType.OFF_HEAP, size);
        return new MemoryManager(memorySizes, 32768);
    }

    public static class Builder {
        private final Object owner;
        private Collection<MemorySegment> output = new ArrayList<MemorySegment>();
        private int numberOfPages = 1;
        private Set<MemoryType> types = EnumSet.noneOf(MemoryType.class);

        public Builder(Object owner) {
            this.owner = owner;
        }

        public Builder withOutput(Collection<MemorySegment> output) {
            this.output = output;
            return this;
        }

        public Builder numberOfPages(int numberOfPages) {
            this.numberOfPages = numberOfPages;
            return this;
        }

        public Builder ofType(MemoryType type) {
            this.types.add(type);
            return this;
        }

        public Builder ofAllTypes() {
            this.types = EnumSet.allOf(MemoryType.class);
            return this;
        }

        public AllocationRequest build() {
            return new AllocationRequest(this.owner, this.output, this.numberOfPages, this.types);
        }
    }

    public static class AllocationRequest {
        private final Object owner;
        private final Collection<MemorySegment> output;
        private final int numberOfPages;
        private final Set<MemoryType> types;

        private AllocationRequest(Object owner, Collection<MemorySegment> output, int numberOfPages, Set<MemoryType> types) {
            this.owner = owner;
            this.output = output;
            this.numberOfPages = numberOfPages;
            this.types = types;
        }

        public Object getOwner() {
            return this.owner;
        }

        public int getNumberOfPages() {
            return this.numberOfPages;
        }

        public Set<MemoryType> getTypes() {
            return Collections.unmodifiableSet(this.types);
        }

        public static Builder newBuilder(Object owner) {
            return new Builder(owner);
        }

        public static AllocationRequest ofAllTypes(Object owner, int numberOfPages) {
            return AllocationRequest.newBuilder(owner).ofAllTypes().numberOfPages(numberOfPages).build();
        }

        public static AllocationRequest ofType(Object owner, int numberOfPages, MemoryType type) {
            return AllocationRequest.newBuilder(owner).ofType(type).numberOfPages(numberOfPages).build();
        }
    }
}

