/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.util;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Collection;
import java.util.Optional;
import java.util.function.Supplier;
import javax.annotation.Nullable;
import org.apache.flink.util.ExceptionUtils;
import org.apache.flink.util.FlinkRuntimeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public enum JavaGcCleanerWrapper {

    private static final Logger LOG = LoggerFactory.getLogger(JavaGcCleanerWrapper.class);
    private static final Class<?>[] LEGACY_WAIT_FOR_REFERENCE_PROCESSING_ARG_TYPES = new Class[]{Boolean.TYPE};
    private static final Object[] LEGACY_WAIT_FOR_REFERENCE_PROCESSING_ARGS = new Object[]{false};
    private static final Class<?>[] JAVA9_WAIT_FOR_REFERENCE_PROCESSING_ARG_TYPES = new Class[0];
    private static final Object[] JAVA9_WAIT_FOR_REFERENCE_PROCESSING_ARGS = new Object[0];
    private static final Collection<CleanerProvider> CLEANER_PROVIDERS = Arrays.asList(JavaGcCleanerWrapper.createLegacyCleanerProvider(), JavaGcCleanerWrapper.createJava9CleanerProvider());
    private static final CleanerManager CLEANER_MANAGER = JavaGcCleanerWrapper.findGcCleanerManager();

    private static CleanerProvider createLegacyCleanerProvider() {
        String name = "Legacy (before Java 9) cleaner";
        ReflectionUtils reflectionUtils = new ReflectionUtils(name + " provider");
        String cleanerClassName = "sun.misc.Cleaner";
        return new CleanerProvider(name, new CleanerFactoryProvider(name, reflectionUtils, cleanerClassName, Optional::empty, "create", cleanerClassName, "clean"), new PendingCleanersRunnerProvider(name, reflectionUtils, "tryHandlePending", LEGACY_WAIT_FOR_REFERENCE_PROCESSING_ARGS, LEGACY_WAIT_FOR_REFERENCE_PROCESSING_ARG_TYPES));
    }

    private static CleanerProvider createJava9CleanerProvider() {
        String name = "New Java 9+ cleaner";
        ReflectionUtils reflectionUtils = new ReflectionUtils(name + " provider");
        String cleanerClassName = "java.lang.ref.Cleaner";
        return new CleanerProvider(name, new CleanerFactoryProvider(name, reflectionUtils, cleanerClassName, () -> {
            Class cleanerClass = reflectionUtils.findClass(cleanerClassName);
            Method cleanerCreateMethod = reflectionUtils.findMethod(cleanerClass, "create", new Class[0]);
            try {
                return Optional.of(cleanerCreateMethod.invoke(null, new Object[0]));
            }
            catch (IllegalAccessException | InvocationTargetException e) {
                throw new FlinkRuntimeException("Failed to create a Java 9 Cleaner", e);
            }
        }, "register", "java.lang.ref.Cleaner$Cleanable", "clean"), new PendingCleanersRunnerProvider(name, reflectionUtils, "waitForReferenceProcessing", JAVA9_WAIT_FOR_REFERENCE_PROCESSING_ARGS, JAVA9_WAIT_FOR_REFERENCE_PROCESSING_ARG_TYPES));
    }

    private static CleanerManager findGcCleanerManager() {
        CleanerManager foundCleanerManager = null;
        Throwable t = null;
        for (CleanerProvider cleanerProvider : CLEANER_PROVIDERS) {
            try {
                foundCleanerManager = cleanerProvider.createCleanerManager();
                break;
            }
            catch (Throwable e) {
                t = ExceptionUtils.firstOrSuppressed(e, t);
            }
        }
        if (foundCleanerManager == null) {
            String errorMessage = String.format("Failed to find GC Cleaner among available providers: %s", CLEANER_PROVIDERS);
            throw new Error(errorMessage, t);
        }
        return foundCleanerManager;
    }

    public static Runnable createCleaner(Object owner, Runnable cleanOperation) {
        return JavaGcCleanerWrapper.CLEANER_MANAGER.create(owner, cleanOperation);
    }

    public static boolean tryRunPendingCleaners() throws InterruptedException {
        return JavaGcCleanerWrapper.CLEANER_MANAGER.tryRunPendingCleaners();
    }

    private static class ReflectionUtils {
        private final String logPrefix;

        private ReflectionUtils(String logPrefix) {
            this.logPrefix = logPrefix;
        }

        private Class<?> findClass(String className) {
            try {
                return Class.forName(className);
            }
            catch (ClassNotFoundException e) {
                throw new FlinkRuntimeException(String.format("%s: Failed to find %s class", this.logPrefix, className.split("\\.")[0]), e);
            }
        }

        private Method findMethod(Class<?> cl, String methodName, Class<?> ... parameterTypes) {
            try {
                return cl.getDeclaredMethod(methodName, parameterTypes);
            }
            catch (NoSuchMethodException e) {
                throw new FlinkRuntimeException(String.format("%s: Failed to find %s#%s method", this.logPrefix, cl.getSimpleName(), methodName), e);
            }
        }
    }

    private static class PendingCleanersRunner {
        private final Method waitForReferenceProcessingMethod;
        private final Object[] waitForReferenceProcessingArgs;

        private PendingCleanersRunner(Method waitForReferenceProcessingMethod, Object[] waitForReferenceProcessingArgs) {
            this.waitForReferenceProcessingMethod = waitForReferenceProcessingMethod;
            this.waitForReferenceProcessingArgs = waitForReferenceProcessingArgs;
        }

        private boolean tryRunPendingCleaners() throws InterruptedException {
            try {
                return (Boolean)this.waitForReferenceProcessingMethod.invoke(null, this.waitForReferenceProcessingArgs);
            }
            catch (IllegalAccessException | InvocationTargetException e) {
                PendingCleanersRunner.throwIfCauseIsInterruptedException(e);
                return (Boolean)PendingCleanersRunner.throwInvocationError(e, this.waitForReferenceProcessingMethod);
            }
        }

        private static void throwIfCauseIsInterruptedException(Throwable t) throws InterruptedException {
            if (t.getCause() instanceof InterruptedException) {
                throw (InterruptedException)t.getCause();
            }
        }

        private static <T> T throwInvocationError(Throwable t, Method method) {
            String message = String.format("FATAL UNEXPECTED - Failed to invoke %s#%s", "java.lang.ref.Reference", method.getName());
            LOG.error(message, t);
            throw new Error(message, t);
        }
    }

    private static class PendingCleanersRunnerProvider {
        private static final String REFERENCE_CLASS = "java.lang.ref.Reference";
        private final String cleanerName;
        private final ReflectionUtils reflectionUtils;
        private final String waitForReferenceProcessingName;
        private final Object[] waitForReferenceProcessingArgs;
        private final Class<?>[] waitForReferenceProcessingArgTypes;

        private PendingCleanersRunnerProvider(String cleanerName, ReflectionUtils reflectionUtils, String waitForReferenceProcessingName, Object[] waitForReferenceProcessingArgs, Class<?>[] waitForReferenceProcessingArgTypes) {
            this.cleanerName = cleanerName;
            this.reflectionUtils = reflectionUtils;
            this.waitForReferenceProcessingName = waitForReferenceProcessingName;
            this.waitForReferenceProcessingArgs = waitForReferenceProcessingArgs;
            this.waitForReferenceProcessingArgTypes = waitForReferenceProcessingArgTypes;
        }

        private PendingCleanersRunner createPendingCleanersRunner() {
            Class referenceClass = this.reflectionUtils.findClass(REFERENCE_CLASS);
            Method waitForReferenceProcessingMethod = this.reflectionUtils.findMethod(referenceClass, this.waitForReferenceProcessingName, this.waitForReferenceProcessingArgTypes);
            waitForReferenceProcessingMethod.setAccessible(true);
            return new PendingCleanersRunner(waitForReferenceProcessingMethod, this.waitForReferenceProcessingArgs);
        }

        public String toString() {
            return "Pending " + this.cleanerName + "s runner provider";
        }
    }

    private static class CleanerFactory {
        private final String cleanerName;
        @Nullable
        private final Object cleaner;
        private final Method cleanableCreationMethod;
        private final Method cleanMethod;

        private CleanerFactory(String cleanerName, @Nullable Object cleaner, Method cleanableCreationMethod, Method cleanMethod) {
            this.cleanerName = cleanerName;
            this.cleaner = cleaner;
            this.cleanableCreationMethod = cleanableCreationMethod;
            this.cleanMethod = cleanMethod;
        }

        private Runnable create(Object owner, Runnable cleanupOperation) {
            Object cleanable;
            try {
                cleanable = this.cleanableCreationMethod.invoke(this.cleaner, owner, cleanupOperation);
            }
            catch (IllegalAccessException | InvocationTargetException e) {
                throw new Error("Failed to create a " + this.cleanerName, e);
            }
            String ownerString = owner.toString();
            return () -> {
                try {
                    this.cleanMethod.invoke(cleanable, new Object[0]);
                }
                catch (IllegalAccessException | InvocationTargetException e) {
                    String message = String.format("FATAL UNEXPECTED - Failed to invoke a %s for %s", this.cleanerName, ownerString);
                    LOG.error(message, (Throwable)e);
                    throw new Error(message, e);
                }
            };
        }
    }

    private static class CleanerFactoryProvider {
        private final String cleanerName;
        private final ReflectionUtils reflectionUtils;
        private final String cleanerClassName;
        private final Supplier<Optional<Object>> cleanerSupplier;
        private final String cleanableCreationMethodName;
        private final String cleanableClassName;
        private final String cleanMethodName;

        private CleanerFactoryProvider(String cleanerName, ReflectionUtils reflectionUtils, String cleanerClassName, Supplier<Optional<Object>> cleanerSupplier, String cleanableCreationMethodName, String cleanableClassName, String cleanMethodName) {
            this.cleanerName = cleanerName;
            this.reflectionUtils = reflectionUtils;
            this.cleanerClassName = cleanerClassName;
            this.cleanerSupplier = cleanerSupplier;
            this.cleanableCreationMethodName = cleanableCreationMethodName;
            this.cleanableClassName = cleanableClassName;
            this.cleanMethodName = cleanMethodName;
        }

        private CleanerFactory createCleanerFactory() {
            Class cleanerClass = this.reflectionUtils.findClass(this.cleanerClassName);
            Method cleanableCreationMethod = this.reflectionUtils.findMethod(cleanerClass, this.cleanableCreationMethodName, new Class[]{Object.class, Runnable.class});
            Class cleanableClass = this.reflectionUtils.findClass(this.cleanableClassName);
            Method cleanMethod = this.reflectionUtils.findMethod(cleanableClass, this.cleanMethodName, new Class[0]);
            return new CleanerFactory(this.cleanerName, this.cleanerSupplier.get().orElse(null), cleanableCreationMethod, cleanMethod);
        }

        public String toString() {
            return this.cleanerName + " factory provider using " + this.cleanerClassName;
        }
    }

    private static class CleanerManager {
        private final String cleanerName;
        private final CleanerFactory cleanerFactory;
        private final PendingCleanersRunner pendingCleanersRunner;

        private CleanerManager(String cleanerName, CleanerFactory cleanerFactory, PendingCleanersRunner pendingCleanersRunner) {
            this.cleanerName = cleanerName;
            this.cleanerFactory = cleanerFactory;
            this.pendingCleanersRunner = pendingCleanersRunner;
        }

        private Runnable create(Object owner, Runnable cleanOperation) {
            return this.cleanerFactory.create(owner, cleanOperation);
        }

        private boolean tryRunPendingCleaners() throws InterruptedException {
            return this.pendingCleanersRunner.tryRunPendingCleaners();
        }

        public String toString() {
            return this.cleanerName + " manager";
        }
    }

    private static class CleanerProvider {
        private final String cleanerName;
        private final CleanerFactoryProvider cleanerFactoryProvider;
        private final PendingCleanersRunnerProvider pendingCleanersRunnerProvider;

        private CleanerProvider(String cleanerName, CleanerFactoryProvider cleanerFactoryProvider, PendingCleanersRunnerProvider pendingCleanersRunnerProvider) {
            this.cleanerName = cleanerName;
            this.cleanerFactoryProvider = cleanerFactoryProvider;
            this.pendingCleanersRunnerProvider = pendingCleanersRunnerProvider;
        }

        private CleanerManager createCleanerManager() {
            return new CleanerManager(this.cleanerName, this.cleanerFactoryProvider.createCleanerFactory(), this.pendingCleanersRunnerProvider.createPendingCleanersRunner());
        }

        public String toString() {
            return this.cleanerName + " provider";
        }
    }
}

