/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.memory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.ConcurrentModificationException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import javax.annotation.Nullable;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.core.memory.MemorySegment;
import org.apache.flink.core.memory.MemorySegmentFactory;
import org.apache.flink.runtime.memory.MemoryAllocationException;
import org.apache.flink.runtime.memory.MemoryReservationException;
import org.apache.flink.runtime.memory.OpaqueMemoryResource;
import org.apache.flink.runtime.memory.SharedResources;
import org.apache.flink.runtime.memory.UnsafeMemoryBudget;
import org.apache.flink.util.MathUtils;
import org.apache.flink.util.Preconditions;
import org.apache.flink.util.function.LongFunctionWithException;
import org.apache.flink.util.function.ThrowingRunnable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MemoryManager {
    private static final Logger LOG = LoggerFactory.getLogger(MemoryManager.class);
    public static final int DEFAULT_PAGE_SIZE = 32768;
    public static final int MIN_PAGE_SIZE = 4096;
    private final Map<Object, Set<MemorySegment>> allocatedSegments;
    private final Map<Object, Long> reservedMemory;
    private final long pageSize;
    private final long totalNumberOfPages;
    private final UnsafeMemoryBudget memoryBudget;
    private final SharedResources sharedResources;
    private volatile boolean isShutDown;

    public MemoryManager(long memorySize, int pageSize) {
        MemoryManager.sanityCheck(memorySize, pageSize);
        this.pageSize = pageSize;
        this.memoryBudget = new UnsafeMemoryBudget(memorySize);
        this.totalNumberOfPages = memorySize / (long)pageSize;
        this.allocatedSegments = new ConcurrentHashMap<Object, Set<MemorySegment>>();
        this.reservedMemory = new ConcurrentHashMap<Object, Long>();
        this.sharedResources = new SharedResources();
        MemoryManager.verifyIntTotalNumberOfPages(memorySize, this.totalNumberOfPages);
        LOG.debug("Initialized MemoryManager with total memory size {} and page size {}.", (Object)memorySize, (Object)pageSize);
    }

    private static void sanityCheck(long memorySize, int pageSize) {
        Preconditions.checkArgument((memorySize >= 0L ? 1 : 0) != 0, (Object)"Size of total memory must be non-negative.");
        Preconditions.checkArgument((pageSize >= 4096 ? 1 : 0) != 0, (String)"The page size must be at least %d bytes.", (Object[])new Object[]{4096});
        Preconditions.checkArgument((boolean)MathUtils.isPowerOf2((long)pageSize), (Object)"The given page size is not a power of two.");
    }

    private static void verifyIntTotalNumberOfPages(long memorySize, long numberOfPagesLong) {
        Preconditions.checkArgument((numberOfPagesLong <= Integer.MAX_VALUE ? 1 : 0) != 0, (String)"The given number of memory bytes (%d) corresponds to more than MAX_INT pages (%d > %d).", (Object[])new Object[]{memorySize, numberOfPagesLong, Integer.MAX_VALUE});
    }

    public void shutdown() {
        if (!this.isShutDown) {
            this.isShutDown = true;
            this.reservedMemory.clear();
            for (Set<MemorySegment> segments : this.allocatedSegments.values()) {
                for (MemorySegment seg : segments) {
                    seg.free();
                }
                segments.clear();
            }
            this.allocatedSegments.clear();
        }
    }

    @VisibleForTesting
    public boolean isShutdown() {
        return this.isShutDown;
    }

    public boolean verifyEmpty() {
        return this.memoryBudget.verifyEmpty();
    }

    public List<MemorySegment> allocatePages(Object owner, int numPages) throws MemoryAllocationException {
        ArrayList<MemorySegment> segments = new ArrayList<MemorySegment>(numPages);
        this.allocatePages(owner, segments, numPages);
        return segments;
    }

    public void allocatePages(Object owner, Collection<MemorySegment> target, int numberOfPages) throws MemoryAllocationException {
        Preconditions.checkNotNull((Object)owner, (String)"The memory owner must not be null.");
        Preconditions.checkState((!this.isShutDown ? 1 : 0) != 0, (Object)"Memory manager has been shut down.");
        Preconditions.checkArgument(((long)numberOfPages <= this.totalNumberOfPages ? 1 : 0) != 0, (String)"Cannot allocate more segments %d than the max number %d", (Object[])new Object[]{numberOfPages, this.totalNumberOfPages});
        if (target instanceof ArrayList) {
            ((ArrayList)target).ensureCapacity(numberOfPages);
        }
        long memoryToReserve = (long)numberOfPages * this.pageSize;
        try {
            this.memoryBudget.reserveMemory(memoryToReserve);
        }
        catch (MemoryReservationException e) {
            throw new MemoryAllocationException(String.format("Could not allocate %d pages", numberOfPages), e);
        }
        Runnable pageCleanup = this::releasePage;
        this.allocatedSegments.compute(owner, (o, currentSegmentsForOwner) -> {
            Set segmentsForOwner = currentSegmentsForOwner == null ? new HashSet(numberOfPages) : currentSegmentsForOwner;
            for (long i = (long)numberOfPages; i > 0L; --i) {
                MemorySegment segment = MemorySegmentFactory.allocateOffHeapUnsafeMemory((int)this.getPageSize(), (Object)owner, (Runnable)pageCleanup);
                target.add(segment);
                segmentsForOwner.add(segment);
            }
            return segmentsForOwner;
        });
        Preconditions.checkState((!this.isShutDown ? 1 : 0) != 0, (Object)"Memory manager has been concurrently shut down.");
    }

    private void releasePage() {
        this.memoryBudget.releaseMemory(this.getPageSize());
    }

    public void release(MemorySegment segment) {
        Preconditions.checkState((!this.isShutDown ? 1 : 0) != 0, (Object)"Memory manager has been shut down.");
        if (segment == null || segment.getOwner() == null) {
            return;
        }
        try {
            this.allocatedSegments.computeIfPresent(segment.getOwner(), (o, segsForOwner) -> {
                segment.free();
                segsForOwner.remove(segment);
                return segsForOwner.isEmpty() ? null : segsForOwner;
            });
        }
        catch (Throwable t) {
            throw new RuntimeException("Error removing book-keeping reference to allocated memory segment.", t);
        }
    }

    public void release(Collection<MemorySegment> segments) {
        if (segments == null) {
            return;
        }
        Preconditions.checkState((!this.isShutDown ? 1 : 0) != 0, (Object)"Memory manager has been shut down.");
        boolean successfullyReleased = false;
        do {
            Iterator<MemorySegment> segmentsIterator = segments.iterator();
            try {
                MemorySegment segment = null;
                while (segment == null && segmentsIterator.hasNext()) {
                    segment = segmentsIterator.next();
                }
                while (segment != null) {
                    segment = this.releaseSegmentsForOwnerUntilNextOwner(segment, segmentsIterator);
                }
                segments.clear();
                successfullyReleased = true;
            }
            catch (ConcurrentModificationException | NoSuchElementException runtimeException) {
                // empty catch block
            }
        } while (!successfullyReleased);
    }

    private MemorySegment releaseSegmentsForOwnerUntilNextOwner(MemorySegment firstSeg, Iterator<MemorySegment> segmentsIterator) {
        AtomicReference nextOwnerMemorySegment = new AtomicReference();
        Object owner = firstSeg.getOwner();
        this.allocatedSegments.compute(owner, (o, segsForOwner) -> {
            MemoryManager.freeSegment(firstSeg, segsForOwner);
            while (segmentsIterator.hasNext()) {
                MemorySegment segment = (MemorySegment)segmentsIterator.next();
                try {
                    if (segment == null || segment.isFreed()) continue;
                    Object nextOwner = segment.getOwner();
                    if (nextOwner != owner) {
                        nextOwnerMemorySegment.set(segment);
                        break;
                    }
                    MemoryManager.freeSegment(segment, segsForOwner);
                }
                catch (Throwable t) {
                    throw new RuntimeException("Error removing book-keeping reference to allocated memory segment.", t);
                }
            }
            return segsForOwner == null || segsForOwner.isEmpty() ? null : segsForOwner;
        });
        return (MemorySegment)nextOwnerMemorySegment.get();
    }

    private static void freeSegment(MemorySegment segment, @Nullable Collection<MemorySegment> segments) {
        segment.free();
        if (segments != null) {
            segments.remove(segment);
        }
    }

    public void releaseAll(Object owner) {
        if (owner == null) {
            return;
        }
        Preconditions.checkState((!this.isShutDown ? 1 : 0) != 0, (Object)"Memory manager has been shut down.");
        Set<MemorySegment> segments = this.allocatedSegments.remove(owner);
        if (segments == null || segments.isEmpty()) {
            return;
        }
        for (MemorySegment segment : segments) {
            segment.free();
        }
        segments.clear();
    }

    public void reserveMemory(Object owner, long size) throws MemoryReservationException {
        this.checkMemoryReservationPreconditions(owner, size);
        if (size == 0L) {
            return;
        }
        this.memoryBudget.reserveMemory(size);
        this.reservedMemory.compute(owner, (o, memoryReservedForOwner) -> memoryReservedForOwner == null ? size : memoryReservedForOwner + size);
        Preconditions.checkState((!this.isShutDown ? 1 : 0) != 0, (Object)"Memory manager has been concurrently shut down.");
    }

    public void releaseMemory(Object owner, long size) {
        this.checkMemoryReservationPreconditions(owner, size);
        if (size == 0L) {
            return;
        }
        this.reservedMemory.compute(owner, (o, currentlyReserved) -> {
            long newReservedMemory = 0L;
            if (currentlyReserved != null) {
                if (currentlyReserved < size) {
                    LOG.warn("Trying to release more memory {} than it was reserved {} so far for the owner {}", new Object[]{size, currentlyReserved, owner});
                }
                newReservedMemory = this.releaseAndCalculateReservedMemory(size, (long)currentlyReserved);
            }
            return newReservedMemory == 0L ? null : Long.valueOf(newReservedMemory);
        });
    }

    private long releaseAndCalculateReservedMemory(long memoryToFree, long currentlyReserved) {
        long effectiveMemoryToRelease = Math.min(currentlyReserved, memoryToFree);
        this.memoryBudget.releaseMemory(effectiveMemoryToRelease);
        return currentlyReserved - effectiveMemoryToRelease;
    }

    private void checkMemoryReservationPreconditions(Object owner, long size) {
        Preconditions.checkNotNull((Object)owner, (String)"The memory owner must not be null.");
        Preconditions.checkState((!this.isShutDown ? 1 : 0) != 0, (Object)"Memory manager has been shut down.");
        Preconditions.checkArgument((size >= 0L ? 1 : 0) != 0, (String)"The memory size (%s) has to have non-negative size", (Object[])new Object[]{size});
    }

    public void releaseAllMemory(Object owner) {
        this.checkMemoryReservationPreconditions(owner, 0L);
        Long memoryReservedForOwner = this.reservedMemory.remove(owner);
        if (memoryReservedForOwner != null) {
            this.memoryBudget.releaseMemory(memoryReservedForOwner);
        }
    }

    public <T extends AutoCloseable> OpaqueMemoryResource<T> getSharedMemoryResourceForManagedMemory(String type, LongFunctionWithException<T, Exception> initializer) throws Exception {
        return this.getSharedMemoryResourceForManagedMemory(type, initializer, 1.0);
    }

    public <T extends AutoCloseable> OpaqueMemoryResource<T> getSharedMemoryResourceForManagedMemory(String type, LongFunctionWithException<T, Exception> initializer, double fractionToInitializeWith) throws Exception {
        long numBytes = this.computeMemorySize(fractionToInitializeWith);
        LongFunctionWithException reserveAndInitialize = size -> {
            try {
                this.reserveMemory(type, size);
            }
            catch (MemoryReservationException e) {
                throw new MemoryAllocationException("Could not created the shared memory resource of size " + size + ". Not enough memory left to reserve from the slot's managed memory.", e);
            }
            return (AutoCloseable)initializer.apply(size);
        };
        Consumer<Long> releaser = size -> this.releaseMemory(type, (long)size);
        Object leaseHolder = new Object();
        SharedResources.ResourceAndSize resource = this.sharedResources.getOrAllocateSharedResource(type, leaseHolder, reserveAndInitialize, numBytes);
        long size2 = resource.size();
        ThrowingRunnable disposer = () -> this.sharedResources.release(type, leaseHolder, releaser);
        return new OpaqueMemoryResource(resource.resourceHandle(), size2, (ThrowingRunnable<Exception>)disposer);
    }

    public <T extends AutoCloseable> OpaqueMemoryResource<T> getExternalSharedMemoryResource(String type, LongFunctionWithException<T, Exception> initializer, long numBytes) throws Exception {
        Object leaseHolder = new Object();
        SharedResources.ResourceAndSize<T> resource = this.sharedResources.getOrAllocateSharedResource(type, leaseHolder, initializer, numBytes);
        ThrowingRunnable disposer = () -> this.sharedResources.release(type, leaseHolder);
        return new OpaqueMemoryResource<T>(resource.resourceHandle(), resource.size(), (ThrowingRunnable<Exception>)disposer);
    }

    public int getPageSize() {
        return (int)this.pageSize;
    }

    public long getMemorySize() {
        return this.memoryBudget.getTotalMemorySize();
    }

    public long availableMemory() {
        return this.memoryBudget.getAvailableMemorySize();
    }

    public int computeNumberOfPages(double fraction) {
        if (fraction <= 0.0 || fraction > 1.0) {
            throw new IllegalArgumentException("The fraction of memory to allocate must within (0, 1].");
        }
        return (int)((double)this.totalNumberOfPages * fraction);
    }

    public long computeMemorySize(double fraction) {
        Preconditions.checkArgument((fraction > 0.0 && fraction <= 1.0 ? 1 : 0) != 0, (String)"The fraction of memory to allocate must within (0, 1], was: %s", (Object[])new Object[]{fraction});
        return (long)Math.floor((double)this.memoryBudget.getTotalMemorySize() * fraction);
    }

    public static MemoryManager forDefaultPageSize(long size) {
        return new MemoryManager(size, 32768);
    }
}

