/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.streaming.runtime.io;

import java.io.IOException;
import java.util.Arrays;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.flink.annotation.Internal;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.runtime.checkpoint.CheckpointException;
import org.apache.flink.runtime.checkpoint.CheckpointFailureReason;
import org.apache.flink.runtime.checkpoint.channel.InputChannelInfo;
import org.apache.flink.runtime.io.network.api.CancelCheckpointMarker;
import org.apache.flink.runtime.io.network.api.CheckpointBarrier;
import org.apache.flink.runtime.io.network.partition.consumer.InputGate;
import org.apache.flink.runtime.jobgraph.tasks.AbstractInvokable;
import org.apache.flink.streaming.runtime.io.CheckpointBarrierHandler;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Internal
public class CheckpointBarrierAligner
extends CheckpointBarrierHandler {
    private static final Logger LOG = LoggerFactory.getLogger(CheckpointBarrierAligner.class);
    private final Map<InputChannelInfo, Boolean> blockedChannels;
    private final int totalNumberOfInputChannels;
    private final String taskName;
    private long currentCheckpointId = -1L;
    private int numBarriersReceived;
    private int numClosedChannels;
    private long startOfAlignmentTimestamp;
    private long latestAlignmentDurationNanos;
    private final InputGate[] inputGates;

    CheckpointBarrierAligner(String taskName, AbstractInvokable toNotifyOnCheckpoint, InputGate ... inputGates) {
        super(toNotifyOnCheckpoint);
        this.taskName = taskName;
        this.inputGates = inputGates;
        this.blockedChannels = Arrays.stream(inputGates).flatMap(gate -> gate.getChannelInfos().stream()).collect(Collectors.toMap(Function.identity(), info -> false));
        this.totalNumberOfInputChannels = this.blockedChannels.size();
    }

    @Override
    public void abortPendingCheckpoint(long checkpointId, CheckpointException exception) throws IOException {
        if (checkpointId > this.currentCheckpointId && this.isCheckpointPending()) {
            this.releaseBlocksAndResetBarriers();
            this.notifyAbort(this.currentCheckpointId, exception);
        }
    }

    @Override
    public void releaseBlocksAndResetBarriers() throws IOException {
        LOG.debug("{}: End of stream alignment, feeding buffered data back.", (Object)this.taskName);
        for (Map.Entry<InputChannelInfo, Boolean> blockedChannel : this.blockedChannels.entrySet()) {
            if (blockedChannel.getValue().booleanValue()) {
                this.resumeConsumption(blockedChannel.getKey());
            }
            blockedChannel.setValue(false);
        }
        this.numBarriersReceived = 0;
        if (this.startOfAlignmentTimestamp > 0L) {
            this.latestAlignmentDurationNanos = System.nanoTime() - this.startOfAlignmentTimestamp;
            this.startOfAlignmentTimestamp = 0L;
        }
    }

    @Override
    public boolean isBlocked(InputChannelInfo channelInfo) {
        return this.blockedChannels.get(channelInfo);
    }

    @Override
    public void processBarrier(CheckpointBarrier receivedBarrier, InputChannelInfo channelInfo) throws Exception {
        long barrierId = receivedBarrier.getId();
        if (this.totalNumberOfInputChannels == 1) {
            this.resumeConsumption(channelInfo);
            if (barrierId > this.currentCheckpointId) {
                this.currentCheckpointId = barrierId;
                this.notifyCheckpoint(receivedBarrier, this.latestAlignmentDurationNanos);
            }
            return;
        }
        if (this.isCheckpointPending()) {
            if (barrierId == this.currentCheckpointId) {
                this.onBarrier(channelInfo);
            } else if (barrierId > this.currentCheckpointId) {
                LOG.warn("{}: Received checkpoint barrier for checkpoint {} before completing current checkpoint {}. Skipping current checkpoint.", new Object[]{this.taskName, barrierId, this.currentCheckpointId});
                this.notifyAbort(this.currentCheckpointId, new CheckpointException("Barrier id: " + barrierId, CheckpointFailureReason.CHECKPOINT_DECLINED_SUBSUMED));
                this.releaseBlocksAndResetBarriers();
                this.beginNewAlignment(barrierId, channelInfo, receivedBarrier.getTimestamp());
            } else {
                this.resumeConsumption(channelInfo);
            }
        } else if (barrierId > this.currentCheckpointId) {
            this.beginNewAlignment(barrierId, channelInfo, receivedBarrier.getTimestamp());
        } else {
            this.resumeConsumption(channelInfo);
        }
        if (this.numBarriersReceived + this.numClosedChannels == this.totalNumberOfInputChannels) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("{}: Received all barriers, triggering checkpoint {} at {}.", new Object[]{this.taskName, receivedBarrier.getId(), receivedBarrier.getTimestamp()});
            }
            this.releaseBlocksAndResetBarriers();
            this.notifyCheckpoint(receivedBarrier, this.latestAlignmentDurationNanos);
        }
    }

    protected void beginNewAlignment(long checkpointId, InputChannelInfo channelInfo, long checkpointTimestamp) throws IOException {
        this.markCheckpointStart(checkpointTimestamp);
        this.currentCheckpointId = checkpointId;
        this.onBarrier(channelInfo);
        this.startOfAlignmentTimestamp = System.nanoTime();
        if (LOG.isDebugEnabled()) {
            LOG.debug("{}: Starting stream alignment for checkpoint {}.", (Object)this.taskName, (Object)checkpointId);
        }
    }

    protected void onBarrier(InputChannelInfo channelInfo) throws IOException {
        if (!this.blockedChannels.get(channelInfo).booleanValue()) {
            this.blockedChannels.put(channelInfo, true);
            ++this.numBarriersReceived;
            if (LOG.isDebugEnabled()) {
                LOG.debug("{}: Received barrier from channel {}.", (Object)this.taskName, (Object)channelInfo);
            }
        } else {
            throw new IOException("Stream corrupt: Repeated barrier for same checkpoint on input " + channelInfo);
        }
    }

    @Override
    public void processCancellationBarrier(CancelCheckpointMarker cancelBarrier) throws Exception {
        long barrierId = cancelBarrier.getCheckpointId();
        if (this.totalNumberOfInputChannels == 1) {
            if (barrierId > this.currentCheckpointId) {
                this.currentCheckpointId = barrierId;
                this.notifyAbortOnCancellationBarrier(barrierId);
            }
            return;
        }
        if (this.isCheckpointPending()) {
            if (barrierId == this.currentCheckpointId) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("{}: Checkpoint {} canceled, aborting alignment.", (Object)this.taskName, (Object)barrierId);
                }
                this.releaseBlocksAndResetBarriers();
                this.notifyAbortOnCancellationBarrier(barrierId);
            } else if (barrierId > this.currentCheckpointId) {
                LOG.warn("{}: Received cancellation barrier for checkpoint {} before completing current checkpoint {}. Skipping current checkpoint.", new Object[]{this.taskName, barrierId, this.currentCheckpointId});
                this.releaseBlocksAndResetBarriers();
                this.currentCheckpointId = barrierId;
                this.startOfAlignmentTimestamp = 0L;
                this.latestAlignmentDurationNanos = 0L;
                this.notifyAbortOnCancellationBarrier(barrierId);
            }
        } else if (barrierId > this.currentCheckpointId) {
            this.currentCheckpointId = barrierId;
            this.startOfAlignmentTimestamp = 0L;
            this.latestAlignmentDurationNanos = 0L;
            if (LOG.isDebugEnabled()) {
                LOG.debug("{}: Checkpoint {} canceled, skipping alignment.", (Object)this.taskName, (Object)barrierId);
            }
            this.notifyAbortOnCancellationBarrier(barrierId);
        }
    }

    @Override
    public void processEndOfPartition() throws Exception {
        ++this.numClosedChannels;
        if (this.isCheckpointPending()) {
            this.notifyAbort(this.currentCheckpointId, new CheckpointException(CheckpointFailureReason.CHECKPOINT_DECLINED_INPUT_END_OF_STREAM));
            this.releaseBlocksAndResetBarriers();
        }
    }

    @Override
    public long getLatestCheckpointId() {
        return this.currentCheckpointId;
    }

    @Override
    public long getAlignmentDurationNanos() {
        if (this.startOfAlignmentTimestamp <= 0L) {
            return this.latestAlignmentDurationNanos;
        }
        return System.nanoTime() - this.startOfAlignmentTimestamp;
    }

    @Override
    protected boolean isCheckpointPending() {
        return this.numBarriersReceived > 0;
    }

    private void resumeConsumption(InputChannelInfo channelInfo) throws IOException {
        InputGate inputGate = this.inputGates[channelInfo.getGateIdx()];
        Preconditions.checkState((!inputGate.isFinished() ? 1 : 0) != 0, (Object)"InputGate already finished.");
        inputGate.resumeConsumption(channelInfo.getInputChannelIdx());
    }

    @VisibleForTesting
    public int getNumClosedChannels() {
        return this.numClosedChannels;
    }

    public String toString() {
        return String.format("%s: last checkpoint: %d, current barriers: %d, closed channels: %d", this.taskName, this.currentCheckpointId, this.numBarriersReceived, this.numClosedChannels);
    }
}

