/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletionService;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CreateFlag;
import org.apache.hadoop.fs.StreamCapabilities;
import org.apache.hadoop.hdfs.DFSClient;
import org.apache.hadoop.hdfs.DFSOutputStream;
import org.apache.hadoop.hdfs.DFSPacket;
import org.apache.hadoop.hdfs.DFSUtilClient;
import org.apache.hadoop.hdfs.ExceptionLastSeen;
import org.apache.hadoop.hdfs.StripedDataStreamer;
import org.apache.hadoop.hdfs.client.HdfsDataOutputStream;
import org.apache.hadoop.hdfs.protocol.DatanodeID;
import org.apache.hadoop.hdfs.protocol.DatanodeInfo;
import org.apache.hadoop.hdfs.protocol.ErasureCodingPolicy;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.protocol.HdfsFileStatus;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.protocol.LocatedStripedBlock;
import org.apache.hadoop.hdfs.protocol.datatransfer.BlockConstructionStage;
import org.apache.hadoop.hdfs.util.StripedBlockUtil;
import org.apache.hadoop.io.ByteBufferPool;
import org.apache.hadoop.io.ElasticByteBufferPool;
import org.apache.hadoop.io.MultipleIOException;
import org.apache.hadoop.io.erasurecode.CodecUtil;
import org.apache.hadoop.io.erasurecode.ErasureCoderOptions;
import org.apache.hadoop.io.erasurecode.rawcoder.RawErasureEncoder;
import org.apache.hadoop.util.DataChecksum;
import org.apache.hadoop.util.Progressable;
import org.apache.hadoop.util.Time;
import org.apache.htrace.core.TraceScope;

@InterfaceAudience.Private
public class DFSStripedOutputStream
extends DFSOutputStream
implements StreamCapabilities {
    private static final ByteBufferPool BUFFER_POOL = new ElasticByteBufferPool();
    private final ExceptionLastSeen exceptionLastSeen = new ExceptionLastSeen();
    private final Coordinator coordinator;
    private final CellBuffers cellBuffers;
    private final ErasureCodingPolicy ecPolicy;
    private final RawErasureEncoder encoder;
    private final List<StripedDataStreamer> streamers;
    private final DFSPacket[] currentPackets;
    private final int cellSize;
    private final int numAllBlocks;
    private final int numDataBlocks;
    private ExtendedBlock currentBlockGroup;
    private final String[] favoredNodes;
    private final List<StripedDataStreamer> failedStreamers;
    private final Map<Integer, Integer> corruptBlockCountMap;
    private ExecutorService flushAllExecutor;
    private CompletionService<Void> flushAllExecutorCompletionService;
    private int blockGroupIndex;

    DFSStripedOutputStream(DFSClient dfsClient, String src, HdfsFileStatus stat, EnumSet<CreateFlag> flag, Progressable progress, DataChecksum checksum, String[] favoredNodes) throws IOException {
        super(dfsClient, src, stat, flag, progress, checksum, favoredNodes, false);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Creating DFSStripedOutputStream for " + src);
        }
        this.ecPolicy = stat.getErasureCodingPolicy();
        int numParityBlocks = this.ecPolicy.getNumParityUnits();
        this.cellSize = this.ecPolicy.getCellSize();
        this.numDataBlocks = this.ecPolicy.getNumDataUnits();
        this.numAllBlocks = this.numDataBlocks + numParityBlocks;
        this.favoredNodes = favoredNodes;
        this.failedStreamers = new ArrayList<StripedDataStreamer>();
        this.corruptBlockCountMap = new LinkedHashMap<Integer, Integer>();
        this.flushAllExecutor = Executors.newFixedThreadPool(this.numAllBlocks);
        this.flushAllExecutorCompletionService = new ExecutorCompletionService<Void>(this.flushAllExecutor);
        ErasureCoderOptions coderOptions = new ErasureCoderOptions(this.numDataBlocks, numParityBlocks);
        this.encoder = CodecUtil.createRawEncoder((Configuration)dfsClient.getConfiguration(), (String)this.ecPolicy.getCodecName(), (ErasureCoderOptions)coderOptions);
        this.coordinator = new Coordinator(this.numAllBlocks);
        this.cellBuffers = new CellBuffers(numParityBlocks);
        this.streamers = new ArrayList<StripedDataStreamer>(this.numAllBlocks);
        for (short i = 0; i < this.numAllBlocks; i = (short)(i + 1)) {
            StripedDataStreamer streamer = new StripedDataStreamer(stat, dfsClient, src, progress, checksum, this.cachingStrategy, this.byteArrayManager, favoredNodes, i, this.coordinator, this.getAddBlockFlags());
            this.streamers.add(streamer);
        }
        this.currentPackets = new DFSPacket[this.streamers.size()];
        this.setCurrentStreamer(0);
    }

    private boolean useDirectBuffer() {
        return this.encoder.preferDirectBuffer();
    }

    StripedDataStreamer getStripedDataStreamer(int i) {
        return this.streamers.get(i);
    }

    int getCurrentIndex() {
        return this.getCurrentStreamer().getIndex();
    }

    private synchronized StripedDataStreamer getCurrentStreamer() {
        return (StripedDataStreamer)this.streamer;
    }

    private synchronized StripedDataStreamer setCurrentStreamer(int newIdx) {
        int oldIdx;
        if (this.streamer != null && (oldIdx = this.streamers.indexOf((Object)this.getCurrentStreamer())) >= 0) {
            this.currentPackets[oldIdx] = this.currentPacket;
        }
        this.streamer = this.getStripedDataStreamer(newIdx);
        this.currentPacket = this.currentPackets[newIdx];
        this.adjustChunkBoundary();
        return this.getCurrentStreamer();
    }

    private static void encode(RawErasureEncoder encoder, int numData, ByteBuffer[] buffers) throws IOException {
        ByteBuffer[] dataBuffers = new ByteBuffer[numData];
        ByteBuffer[] parityBuffers = new ByteBuffer[buffers.length - numData];
        System.arraycopy(buffers, 0, dataBuffers, 0, dataBuffers.length);
        System.arraycopy(buffers, numData, parityBuffers, 0, parityBuffers.length);
        encoder.encode(dataBuffers, parityBuffers);
    }

    private Set<StripedDataStreamer> checkStreamers() throws IOException {
        HashSet<StripedDataStreamer> newFailed = new HashSet<StripedDataStreamer>();
        for (StripedDataStreamer s : this.streamers) {
            if (s.isHealthy() || this.failedStreamers.contains((Object)s)) continue;
            newFailed.add(s);
        }
        int failCount = this.failedStreamers.size() + newFailed.size();
        if (LOG.isDebugEnabled()) {
            LOG.debug("checkStreamers: " + this.streamers);
            LOG.debug("healthy streamer count=" + (this.numAllBlocks - failCount));
            LOG.debug("original failed streamers: " + this.failedStreamers);
            LOG.debug("newly failed streamers: " + newFailed);
        }
        if (failCount > this.numAllBlocks - this.numDataBlocks) {
            throw new IOException("Failed: the number of failed blocks = " + failCount + " > the number of parity blocks = " + (this.numAllBlocks - this.numDataBlocks));
        }
        return newFailed;
    }

    private void handleCurrentStreamerFailure(String err, Exception e) throws IOException {
        this.currentPacket = null;
        this.handleStreamerFailure(err, e, this.getCurrentStreamer());
    }

    private void handleStreamerFailure(String err, Exception e, StripedDataStreamer streamer) throws IOException {
        LOG.warn("Failed: " + err + ", " + (Object)((Object)this), (Throwable)e);
        streamer.getErrorState().setInternalError();
        streamer.close(true);
        this.checkStreamers();
        this.currentPackets[streamer.getIndex()] = null;
    }

    private void replaceFailedStreamers() {
        assert (this.streamers.size() == this.numAllBlocks);
        int currentIndex = this.getCurrentIndex();
        assert (currentIndex == 0);
        for (short i = 0; i < this.numAllBlocks; i = (short)((short)(i + 1))) {
            StripedDataStreamer oldStreamer = this.getStripedDataStreamer(i);
            if (oldStreamer.isHealthy()) continue;
            LOG.info("replacing previously failed streamer " + (Object)((Object)oldStreamer));
            StripedDataStreamer streamer = new StripedDataStreamer(oldStreamer.stat, this.dfsClient, this.src, oldStreamer.progress, oldStreamer.checksum4WriteBlock, this.cachingStrategy, this.byteArrayManager, this.favoredNodes, i, this.coordinator, this.getAddBlockFlags());
            this.streamers.set(i, streamer);
            this.currentPackets[i] = null;
            if (i == currentIndex) {
                this.streamer = streamer;
                this.currentPacket = null;
            }
            streamer.start();
        }
    }

    private void waitEndBlocks(int i) throws IOException {
        while (this.getStripedDataStreamer(i).isHealthy()) {
            ExtendedBlock b = (ExtendedBlock)this.coordinator.endBlocks.takeWithTimeout(i);
            if (b == null) continue;
            StripedBlockUtil.checkBlocks(this.currentBlockGroup, i, b);
            return;
        }
    }

    private DatanodeInfo[] getExcludedNodes() {
        ArrayList<DatanodeInfo> excluded = new ArrayList<DatanodeInfo>();
        for (StripedDataStreamer streamer : this.streamers) {
            for (DatanodeInfo e : streamer.getExcludedNodes()) {
                if (e == null) continue;
                excluded.add(e);
            }
        }
        return excluded.toArray(new DatanodeInfo[excluded.size()]);
    }

    private void allocateNewBlock() throws IOException {
        if (this.currentBlockGroup != null) {
            for (int i = 0; i < this.numAllBlocks; ++i) {
                this.waitEndBlocks(i);
            }
        }
        this.failedStreamers.clear();
        DatanodeInfo[] excludedNodes = this.getExcludedNodes();
        LOG.debug("Excluding DataNodes when allocating new block: " + Arrays.asList(excludedNodes));
        this.replaceFailedStreamers();
        LOG.debug("Allocating new block group. The previous block group: " + this.currentBlockGroup);
        LocatedBlock lb = DFSStripedOutputStream.addBlock(excludedNodes, this.dfsClient, this.src, this.currentBlockGroup, this.fileId, this.favoredNodes, this.getAddBlockFlags());
        assert (lb.isStriped());
        this.currentBlockGroup = lb.getBlock();
        ++this.blockGroupIndex;
        LocatedBlock[] blocks = StripedBlockUtil.parseStripedBlockGroup((LocatedStripedBlock)lb, this.cellSize, this.numDataBlocks, this.numAllBlocks - this.numDataBlocks);
        for (int i = 0; i < blocks.length; ++i) {
            StripedDataStreamer si = this.getStripedDataStreamer(i);
            assert (si.isHealthy());
            if (blocks[i] == null) {
                assert (i >= this.numDataBlocks);
                LOG.warn("Cannot allocate parity block(index={}, policy={}). Not enough datanodes? Exclude nodes={}", new Object[]{i, this.ecPolicy.getName(), excludedNodes});
                si.getLastException().set(new IOException("Failed to get parity block, index=" + i));
                si.getErrorState().setInternalError();
                si.close(true);
                continue;
            }
            this.coordinator.getFollowingBlocks().offer(i, blocks[i]);
        }
    }

    private boolean shouldEndBlockGroup() {
        return this.currentBlockGroup != null && this.currentBlockGroup.getNumBytes() == this.blockSize * (long)this.numDataBlocks;
    }

    @Override
    protected synchronized void writeChunk(byte[] bytes, int offset, int len, byte[] checksum, int ckoff, int cklen) throws IOException {
        boolean cellFull;
        int index = this.getCurrentIndex();
        int pos = this.cellBuffers.addTo(index, bytes, offset, len);
        boolean bl = cellFull = pos == this.cellSize;
        if (this.currentBlockGroup == null || this.shouldEndBlockGroup()) {
            this.allocateNewBlock();
        }
        this.currentBlockGroup.setNumBytes(this.currentBlockGroup.getNumBytes() + (long)len);
        StripedDataStreamer current = this.getCurrentStreamer();
        if (current.isHealthy()) {
            try {
                super.writeChunk(bytes, offset, len, checksum, ckoff, cklen);
            }
            catch (Exception e) {
                this.handleCurrentStreamerFailure("offset=" + offset + ", length=" + len, e);
            }
        }
        if (cellFull) {
            int next = index + 1;
            if (next == this.numDataBlocks) {
                this.cellBuffers.flipDataBuffers();
                this.writeParityCells();
                next = 0;
                if (this.shouldEndBlockGroup()) {
                    this.flushAllInternals();
                    this.checkStreamerFailures();
                    for (int i = 0; i < this.numAllBlocks; ++i) {
                        StripedDataStreamer s = this.setCurrentStreamer(i);
                        if (!s.isHealthy()) continue;
                        try {
                            this.endBlock();
                            continue;
                        }
                        catch (IOException iOException) {
                            // empty catch block
                        }
                    }
                } else {
                    this.checkStreamerFailures();
                }
            }
            this.setCurrentStreamer(next);
        }
    }

    @Override
    synchronized void enqueueCurrentPacketFull() throws IOException {
        LOG.debug("enqueue full {}, src={}, bytesCurBlock={}, blockSize={}, appendChunk={}, {}", new Object[]{this.currentPacket, this.src, this.getStreamer().getBytesCurBlock(), this.blockSize, this.getStreamer().getAppendChunk(), this.getStreamer()});
        this.enqueueCurrentPacket();
        this.adjustChunkBoundary();
    }

    private boolean isStreamerWriting(int streamerIndex) {
        long length;
        long l = length = this.currentBlockGroup == null ? 0L : this.currentBlockGroup.getNumBytes();
        if (length == 0L) {
            return false;
        }
        if (streamerIndex >= this.numDataBlocks) {
            return true;
        }
        int numCells = (int)((length - 1L) / (long)this.cellSize + 1L);
        return streamerIndex < numCells;
    }

    private Set<StripedDataStreamer> markExternalErrorOnStreamers() {
        HashSet<StripedDataStreamer> healthySet = new HashSet<StripedDataStreamer>();
        for (int i = 0; i < this.numAllBlocks; ++i) {
            StripedDataStreamer streamer = this.getStripedDataStreamer(i);
            if (!streamer.isHealthy() || !this.isStreamerWriting(i)) continue;
            Preconditions.checkState((streamer.getStage() == BlockConstructionStage.DATA_STREAMING ? 1 : 0) != 0, (Object)("streamer: " + (Object)((Object)streamer)));
            streamer.setExternalError();
            healthySet.add(streamer);
        }
        return healthySet;
    }

    private void checkStreamerFailures() throws IOException {
        Set<StripedDataStreamer> newFailed = this.checkStreamers();
        if (newFailed.size() == 0) {
            return;
        }
        this.flushAllInternals();
        newFailed = this.checkStreamers();
        while (newFailed.size() > 0) {
            this.failedStreamers.addAll(newFailed);
            this.coordinator.clearFailureStates();
            this.corruptBlockCountMap.put(this.blockGroupIndex, this.failedStreamers.size());
            Set<StripedDataStreamer> healthySet = this.markExternalErrorOnStreamers();
            ExtendedBlock newBG = this.updateBlockForPipeline(healthySet);
            newFailed = this.waitCreatingStreamers(healthySet);
            if (newFailed.size() + this.failedStreamers.size() > this.numAllBlocks - this.numDataBlocks) {
                throw new IOException("Data streamers failed while creating new block streams: " + newFailed + ". There are not enough healthy streamers.");
            }
            for (StripedDataStreamer failedStreamer : newFailed) {
                assert (!failedStreamer.isHealthy());
            }
            if (newFailed.size() == 0) {
                for (StripedDataStreamer streamer : healthySet) {
                    assert (streamer.isHealthy());
                    streamer.getErrorState().reset();
                }
                this.updatePipeline(newBG);
            }
            for (int i = 0; i < this.numAllBlocks; ++i) {
                this.coordinator.offerStreamerUpdateResult(i, newFailed.size() == 0);
            }
        }
    }

    private int checkStreamerUpdates(Set<StripedDataStreamer> failed, Set<StripedDataStreamer> streamers) {
        for (StripedDataStreamer streamer : streamers) {
            if (this.coordinator.updateStreamerMap.containsKey((Object)streamer) || streamer.isHealthy() || this.coordinator.getNewBlocks().peek(streamer.getIndex()) == null) continue;
            failed.add(streamer);
        }
        return this.coordinator.updateStreamerMap.size() + failed.size();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Set<StripedDataStreamer> waitCreatingStreamers(Set<StripedDataStreamer> healthyStreamers) throws IOException {
        HashSet<StripedDataStreamer> failed = new HashSet<StripedDataStreamer>();
        int expectedNum = healthyStreamers.size();
        long waitInterval = 1000L;
        Iterator iterator = this.coordinator;
        synchronized (iterator) {
            long l;
            for (long remaingTime = (socketTimeout = (long)this.dfsClient.getConf().getSocketTimeout()) > 0L ? socketTimeout / 2L : Long.MAX_VALUE; this.checkStreamerUpdates(failed, healthyStreamers) < expectedNum && remaingTime > 0L; remaingTime -= Time.monotonicNow() - l) {
                try {
                    l = Time.monotonicNow();
                    this.coordinator.wait(1000L);
                    continue;
                }
                catch (InterruptedException interruptedException) {
                    throw DFSUtilClient.toInterruptedIOException("Interrupted when waiting for results of updating striped streamers", interruptedException);
                }
            }
        }
        iterator = this.coordinator;
        synchronized (iterator) {
            for (StripedDataStreamer streamer : healthyStreamers) {
                if (this.coordinator.updateStreamerMap.containsKey((Object)streamer)) continue;
                LOG.info("close the slow stream " + (Object)((Object)streamer));
                streamer.setStreamerAsClosed();
                failed.add(streamer);
            }
        }
        for (Map.Entry entry : this.coordinator.updateStreamerMap.entrySet()) {
            if (((Boolean)entry.getValue()).booleanValue()) continue;
            failed.add((StripedDataStreamer)((Object)entry.getKey()));
        }
        for (StripedDataStreamer stripedDataStreamer : failed) {
            healthyStreamers.remove((Object)stripedDataStreamer);
        }
        return failed;
    }

    private ExtendedBlock updateBlockForPipeline(Set<StripedDataStreamer> healthyStreamers) throws IOException {
        LocatedBlock updated = this.dfsClient.namenode.updateBlockForPipeline(this.currentBlockGroup, this.dfsClient.clientName);
        long newGS = updated.getBlock().getGenerationStamp();
        ExtendedBlock newBlock = new ExtendedBlock(this.currentBlockGroup);
        newBlock.setGenerationStamp(newGS);
        LocatedBlock[] updatedBlks = StripedBlockUtil.parseStripedBlockGroup((LocatedStripedBlock)updated, this.cellSize, this.numDataBlocks, this.numAllBlocks - this.numDataBlocks);
        for (int i = 0; i < this.numAllBlocks; ++i) {
            StripedDataStreamer si = this.getStripedDataStreamer(i);
            if (!healthyStreamers.contains((Object)si)) continue;
            LocatedBlock lb = new LocatedBlock(new ExtendedBlock(newBlock), null, null, null, -1L, updated.isCorrupt(), null);
            lb.setBlockToken(updatedBlks[i].getBlockToken());
            this.coordinator.getNewBlocks().offer(i, lb);
        }
        return newBlock;
    }

    private void updatePipeline(ExtendedBlock newBG) throws IOException {
        DatanodeID[] newNodes = new DatanodeInfo[this.numAllBlocks];
        String[] newStorageIDs = new String[this.numAllBlocks];
        for (int i = 0; i < this.numAllBlocks; ++i) {
            StripedDataStreamer streamer = this.getStripedDataStreamer(i);
            DatanodeInfo[] nodes = streamer.getNodes();
            String[] storageIDs = streamer.getStorageIDs();
            if (streamer.isHealthy() && nodes != null && storageIDs != null) {
                newNodes[i] = nodes[0];
                newStorageIDs[i] = storageIDs[0];
                continue;
            }
            newNodes[i] = new DatanodeInfo.DatanodeInfoBuilder().setNodeID(DatanodeID.EMPTY_DATANODE_ID).build();
            newStorageIDs[i] = "";
        }
        long sentBytes = this.currentBlockGroup.getNumBytes();
        long ackedBytes = this.getAckedLength();
        Preconditions.checkState((ackedBytes <= sentBytes ? 1 : 0) != 0, (Object)("Acked:" + ackedBytes + ", Sent:" + sentBytes));
        this.currentBlockGroup.setNumBytes(ackedBytes);
        newBG.setNumBytes(ackedBytes);
        this.dfsClient.namenode.updatePipeline(this.dfsClient.clientName, this.currentBlockGroup, newBG, newNodes, newStorageIDs);
        this.currentBlockGroup = newBG;
        this.currentBlockGroup.setNumBytes(sentBytes);
    }

    private List<Long> getBlockLengths() {
        ArrayList<Long> blockLengths = new ArrayList<Long>(this.numAllBlocks);
        for (int i = 0; i < this.numAllBlocks; ++i) {
            StripedDataStreamer streamer = this.getStripedDataStreamer(i);
            long numBytes = -1L;
            if (streamer.isHealthy() && streamer.getBlock() != null) {
                numBytes = streamer.getBlock().getNumBytes();
            }
            blockLengths.add(numBytes);
        }
        return blockLengths;
    }

    private long getAckedLength() {
        int idx;
        long sentBytes = this.currentBlockGroup.getNumBytes();
        long numFullStripes = sentBytes / (long)this.numDataBlocks / (long)this.cellSize;
        long fullStripeLength = numFullStripes * (long)this.numDataBlocks * (long)this.cellSize;
        assert (fullStripeLength <= sentBytes) : "Full stripe length can't be greater than the block group length";
        long ackedLength = 0L;
        List<Long> blockLengths = Collections.unmodifiableList(this.getBlockLengths());
        ArrayList<Long> sortedBlockLengths = new ArrayList<Long>(blockLengths);
        Collections.sort(sortedBlockLengths);
        if (numFullStripes > 0L) {
            int offset = sortedBlockLengths.size() - this.numDataBlocks;
            ackedLength = (Long)sortedBlockLengths.get(offset) * (long)this.numDataBlocks;
        }
        if (ackedLength < fullStripeLength) {
            return ackedLength;
        }
        if (ackedLength == sentBytes) {
            return ackedLength;
        }
        int numFullDataCells = (int)((sentBytes - fullStripeLength) / (long)this.cellSize);
        int partialLength = (int)(sentBytes - fullStripeLength) % this.cellSize;
        int numPartialDataCells = partialLength == 0 ? 0 : 1;
        int numEmptyDataCells = this.numDataBlocks - numFullDataCells - numPartialDataCells;
        int parityLength = numFullDataCells > 0 ? this.cellSize : partialLength;
        long fullStripeBlockOffset = fullStripeLength / (long)this.numDataBlocks;
        long[] expectedBlockLengths = new long[this.numAllBlocks];
        for (idx = 0; idx < numFullDataCells; ++idx) {
            expectedBlockLengths[idx] = fullStripeBlockOffset + (long)this.cellSize;
        }
        while (idx < numFullDataCells + numPartialDataCells) {
            expectedBlockLengths[idx] = fullStripeBlockOffset + (long)partialLength;
            ++idx;
        }
        while (idx < numFullDataCells + numPartialDataCells + numEmptyDataCells) {
            expectedBlockLengths[idx] = fullStripeBlockOffset;
            ++idx;
        }
        while (idx < this.numAllBlocks) {
            expectedBlockLengths[idx] = fullStripeBlockOffset + (long)parityLength;
            ++idx;
        }
        int numBlocksWithCorrectLength = 0;
        for (int i = 0; i < this.numAllBlocks; ++i) {
            if (blockLengths.get(i) != expectedBlockLengths[i]) continue;
            ++numBlocksWithCorrectLength;
        }
        if (numBlocksWithCorrectLength >= this.numDataBlocks) {
            ackedLength = sentBytes;
        }
        return ackedLength;
    }

    private int stripeDataSize() {
        return this.numDataBlocks * this.cellSize;
    }

    @Override
    public boolean hasCapability(String capability) {
        return false;
    }

    @Override
    public void hflush() {
        LOG.debug("DFSStripedOutputStream does not support hflush. Caller should check StreamCapabilities before calling.");
    }

    @Override
    public void hsync() {
        LOG.debug("DFSStripedOutputStream does not support hsync. Caller should check StreamCapabilities before calling.");
    }

    @Override
    public void hsync(EnumSet<HdfsDataOutputStream.SyncFlag> syncFlags) {
        LOG.debug("DFSStripedOutputStream does not support hsync {}. Caller should check StreamCapabilities before calling.", syncFlags);
    }

    @Override
    protected synchronized void start() {
        for (StripedDataStreamer streamer : this.streamers) {
            streamer.start();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    void abort() throws IOException {
        MultipleIOException.Builder b = new MultipleIOException.Builder();
        DFSStripedOutputStream dFSStripedOutputStream = this;
        synchronized (dFSStripedOutputStream) {
            if (this.isClosed()) {
                return;
            }
            this.exceptionLastSeen.set(new IOException("Lease timeout of " + this.dfsClient.getConf().getHdfsTimeout() / 1000 + " seconds expired."));
            try {
                this.closeThreads(true);
            }
            catch (IOException e) {
                b.add((Throwable)e);
            }
        }
        this.dfsClient.endFileLease(this.fileId);
        IOException ioe = b.build();
        if (ioe != null) {
            throw ioe;
        }
    }

    @Override
    boolean isClosed() {
        if (this.closed) {
            return true;
        }
        for (StripedDataStreamer s : this.streamers) {
            if (s.streamerClosed()) continue;
            return false;
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    protected void closeThreads(boolean force) throws IOException {
        MultipleIOException.Builder b = new MultipleIOException.Builder();
        try {
            for (StripedDataStreamer streamer : this.streamers) {
                try {
                    streamer.close(force);
                    streamer.join();
                    streamer.closeSocket();
                }
                catch (Exception e) {
                    try {
                        this.handleStreamerFailure("force=" + force, e, streamer);
                    }
                    catch (IOException ioe) {
                        b.add((Throwable)ioe);
                    }
                }
                finally {
                    streamer.setSocketToNull();
                }
            }
        }
        finally {
            this.setClosed();
        }
        IOException ioe = b.build();
        if (ioe != null) {
            throw ioe;
        }
    }

    private boolean generateParityCellsForLastStripe() {
        long currentBlockGroupBytes = this.currentBlockGroup == null ? 0L : this.currentBlockGroup.getNumBytes();
        long lastStripeSize = currentBlockGroupBytes % (long)this.stripeDataSize();
        if (lastStripeSize == 0L) {
            return false;
        }
        long parityCellSize = lastStripeSize < (long)this.cellSize ? lastStripeSize : (long)this.cellSize;
        ByteBuffer[] buffers = this.cellBuffers.getBuffers();
        for (int i = 0; i < this.numAllBlocks; ++i) {
            int position = buffers[i].position();
            assert ((long)position <= parityCellSize) : "If an internal block is smaller than parity block, then its last cell should be small than last parity cell";
            int j = 0;
            while ((long)j < parityCellSize - (long)position) {
                buffers[i].put((byte)0);
                ++j;
            }
            buffers[i].flip();
        }
        return true;
    }

    void writeParityCells() throws IOException {
        ByteBuffer[] buffers = this.cellBuffers.getBuffers();
        if (!this.checkAnyParityStreamerIsHealthy()) {
            return;
        }
        DFSStripedOutputStream.encode(this.encoder, this.numDataBlocks, buffers);
        for (int i = this.numDataBlocks; i < this.numAllBlocks; ++i) {
            this.writeParity(i, buffers[i], this.cellBuffers.getChecksumArray(i));
        }
        this.cellBuffers.clear();
    }

    private boolean checkAnyParityStreamerIsHealthy() {
        for (int i = this.numDataBlocks; i < this.numAllBlocks; ++i) {
            if (!this.streamers.get(i).isHealthy()) continue;
            return true;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Skips encoding and writing parity cells as there are no healthy parity data streamers: " + this.streamers);
        }
        return false;
    }

    void writeParity(int index, ByteBuffer buffer, byte[] checksumBuf) throws IOException {
        StripedDataStreamer current = this.setCurrentStreamer(index);
        int len = buffer.limit();
        long oldBytes = current.getBytesCurBlock();
        if (current.isHealthy()) {
            try {
                DataChecksum sum = this.getDataChecksum();
                if (buffer.isDirect()) {
                    ByteBuffer directCheckSumBuf = BUFFER_POOL.getBuffer(true, checksumBuf.length);
                    sum.calculateChunkedSums(buffer, directCheckSumBuf);
                    directCheckSumBuf.get(checksumBuf);
                    BUFFER_POOL.putBuffer(directCheckSumBuf);
                } else {
                    sum.calculateChunkedSums(buffer.array(), 0, len, checksumBuf, 0);
                }
                for (int i = 0; i < len; i += sum.getBytesPerChecksum()) {
                    int chunkLen = Math.min(sum.getBytesPerChecksum(), len - i);
                    int ckOffset = i / sum.getBytesPerChecksum() * this.getChecksumSize();
                    super.writeChunk(buffer, chunkLen, checksumBuf, ckOffset, this.getChecksumSize());
                }
            }
            catch (Exception e) {
                this.handleCurrentStreamerFailure("oldBytes=" + oldBytes + ", len=" + len, e);
            }
        }
    }

    @Override
    void setClosed() {
        super.setClosed();
        for (int i = 0; i < this.numAllBlocks; ++i) {
            this.getStripedDataStreamer(i).release();
        }
        this.cellBuffers.release();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    protected synchronized void closeImpl() throws IOException {
        if (this.isClosed()) {
            IOException ioe;
            this.exceptionLastSeen.check(true);
            int minReplication = this.ecPolicy.getNumDataUnits();
            int goodStreamers = 0;
            MultipleIOException.Builder b = new MultipleIOException.Builder();
            for (StripedDataStreamer si : this.streamers) {
                try {
                    si.getLastException().check(true);
                    ++goodStreamers;
                }
                catch (IOException e) {
                    b.add((Throwable)e);
                }
            }
            if (goodStreamers < minReplication && (ioe = b.build()) != null) {
                throw ioe;
            }
            return;
        }
        try {
            try {
                this.flushBuffer();
                if (this.generateParityCellsForLastStripe()) {
                    this.writeParityCells();
                }
                this.enqueueAllCurrentPackets();
                this.flushAllInternals();
                this.checkStreamerFailures();
                for (int i = 0; i < this.numAllBlocks; ++i) {
                    StripedDataStreamer s = this.setCurrentStreamer(i);
                    if (!s.isHealthy()) continue;
                    try {
                        if (s.getBytesCurBlock() > 0L) {
                            this.setCurrentPacketToEmpty();
                        }
                        this.flushInternal();
                        continue;
                    }
                    catch (Exception exception) {
                        // empty catch block
                    }
                }
            }
            finally {
                this.closeThreads(true);
            }
            try (TraceScope ignored = this.dfsClient.getTracer().newScope("completeFile");){
                this.completeFile(this.currentBlockGroup);
            }
            this.logCorruptBlocks();
        }
        catch (ClosedChannelException closedChannelException) {
        }
        finally {
            this.setClosed();
            this.flushAllExecutor.shutdownNow();
            this.encoder.release();
        }
    }

    @VisibleForTesting
    void enqueueAllCurrentPackets() throws IOException {
        int idx = this.streamers.indexOf((Object)this.getCurrentStreamer());
        for (int i = 0; i < this.streamers.size(); ++i) {
            StripedDataStreamer si = this.setCurrentStreamer(i);
            if (!si.isHealthy() || this.currentPacket == null) continue;
            try {
                this.enqueueCurrentPacket();
                continue;
            }
            catch (IOException e) {
                this.handleCurrentStreamerFailure("enqueueAllCurrentPackets, i=" + i, e);
            }
        }
        this.setCurrentStreamer(idx);
    }

    void flushAllInternals() throws IOException {
        int i;
        HashMap<Future<Void>, Integer> flushAllFuturesMap = new HashMap<Future<Void>, Integer>();
        Future<Void> future = null;
        int current = this.getCurrentIndex();
        for (i = 0; i < this.numAllBlocks; ++i) {
            final StripedDataStreamer s = this.setCurrentStreamer(i);
            if (!s.isHealthy()) continue;
            try {
                final long toWaitFor = this.flushInternalWithoutWaitingAck();
                future = this.flushAllExecutorCompletionService.submit(new Callable<Void>(){

                    @Override
                    public Void call() throws Exception {
                        s.waitForAckedSeqno(toWaitFor);
                        return null;
                    }
                });
                flushAllFuturesMap.put(future, i);
                continue;
            }
            catch (Exception e) {
                this.handleCurrentStreamerFailure("flushInternal " + (Object)((Object)s), e);
            }
        }
        this.setCurrentStreamer(current);
        for (i = 0; i < flushAllFuturesMap.size(); ++i) {
            try {
                future = this.flushAllExecutorCompletionService.take();
                future.get();
                continue;
            }
            catch (InterruptedException ie) {
                throw DFSUtilClient.toInterruptedIOException("Interrupted during waiting all streamer flush, ", ie);
            }
            catch (ExecutionException ee) {
                LOG.warn("Caught ExecutionException while waiting all streamer flush, ", (Throwable)ee);
                StripedDataStreamer s = this.streamers.get((Integer)flushAllFuturesMap.get(future));
                this.handleStreamerFailure("flushInternal " + (Object)((Object)s), (Exception)ee.getCause(), s);
            }
        }
    }

    static void sleep(long ms, String op) throws InterruptedIOException {
        try {
            Thread.sleep(ms);
        }
        catch (InterruptedException ie) {
            throw DFSUtilClient.toInterruptedIOException("Sleep interrupted during " + op, ie);
        }
    }

    private void logCorruptBlocks() {
        for (Map.Entry<Integer, Integer> entry : this.corruptBlockCountMap.entrySet()) {
            int bgIndex = entry.getKey();
            int corruptBlockCount = entry.getValue();
            StringBuilder sb = new StringBuilder();
            sb.append("Block group <").append(bgIndex).append("> failed to write ").append(corruptBlockCount).append(" blocks.");
            if (corruptBlockCount == this.numAllBlocks - this.numDataBlocks) {
                sb.append(" It's at high risk of losing data.");
            }
            LOG.warn(sb.toString());
        }
    }

    @Override
    ExtendedBlock getBlock() {
        return this.currentBlockGroup;
    }

    class CellBuffers {
        private final ByteBuffer[] buffers;
        private final byte[][] checksumArrays;

        CellBuffers(int numParityBlocks) {
            int i;
            if (DFSStripedOutputStream.this.cellSize % DFSStripedOutputStream.this.bytesPerChecksum != 0) {
                throw new HadoopIllegalArgumentException("Invalid values: dfs.bytes-per-checksum (=" + DFSStripedOutputStream.this.bytesPerChecksum + ") must divide cell size (=" + DFSStripedOutputStream.this.cellSize + ").");
            }
            this.checksumArrays = new byte[numParityBlocks][];
            int size = DFSStripedOutputStream.this.getChecksumSize() * (DFSStripedOutputStream.this.cellSize / DFSStripedOutputStream.this.bytesPerChecksum);
            for (i = 0; i < this.checksumArrays.length; ++i) {
                this.checksumArrays[i] = new byte[size];
            }
            this.buffers = new ByteBuffer[DFSStripedOutputStream.this.numAllBlocks];
            for (i = 0; i < this.buffers.length; ++i) {
                this.buffers[i] = BUFFER_POOL.getBuffer(DFSStripedOutputStream.this.useDirectBuffer(), DFSStripedOutputStream.this.cellSize);
                this.buffers[i].limit(DFSStripedOutputStream.this.cellSize);
            }
        }

        private ByteBuffer[] getBuffers() {
            return this.buffers;
        }

        byte[] getChecksumArray(int i) {
            return this.checksumArrays[i - DFSStripedOutputStream.this.numDataBlocks];
        }

        private int addTo(int i, byte[] b, int off, int len) {
            ByteBuffer buf = this.buffers[i];
            int pos = buf.position() + len;
            Preconditions.checkState((pos <= DFSStripedOutputStream.this.cellSize ? 1 : 0) != 0);
            buf.put(b, off, len);
            return pos;
        }

        private void clear() {
            for (int i = 0; i < DFSStripedOutputStream.this.numAllBlocks; ++i) {
                this.buffers[i].clear();
                this.buffers[i].limit(DFSStripedOutputStream.this.cellSize);
            }
        }

        private void release() {
            for (int i = 0; i < DFSStripedOutputStream.this.numAllBlocks; ++i) {
                if (this.buffers[i] == null) continue;
                BUFFER_POOL.putBuffer(this.buffers[i]);
                this.buffers[i] = null;
            }
        }

        private void flipDataBuffers() {
            for (int i = 0; i < DFSStripedOutputStream.this.numDataBlocks; ++i) {
                this.buffers[i].flip();
            }
        }
    }

    static class Coordinator {
        private final MultipleBlockingQueue<LocatedBlock> followingBlocks;
        private final MultipleBlockingQueue<ExtendedBlock> endBlocks;
        private final MultipleBlockingQueue<LocatedBlock> newBlocks;
        private final Map<StripedDataStreamer, Boolean> updateStreamerMap;
        private final MultipleBlockingQueue<Boolean> streamerUpdateResult;

        Coordinator(int numAllBlocks) {
            this.followingBlocks = new MultipleBlockingQueue(numAllBlocks, 1);
            this.endBlocks = new MultipleBlockingQueue(numAllBlocks, 1);
            this.newBlocks = new MultipleBlockingQueue(numAllBlocks, 1);
            this.updateStreamerMap = new ConcurrentHashMap<StripedDataStreamer, Boolean>(numAllBlocks);
            this.streamerUpdateResult = new MultipleBlockingQueue(numAllBlocks, 1);
        }

        MultipleBlockingQueue<LocatedBlock> getFollowingBlocks() {
            return this.followingBlocks;
        }

        MultipleBlockingQueue<LocatedBlock> getNewBlocks() {
            return this.newBlocks;
        }

        void offerEndBlock(int i, ExtendedBlock block) {
            this.endBlocks.offer(i, block);
        }

        void offerStreamerUpdateResult(int i, boolean success) {
            this.streamerUpdateResult.offer(i, success);
        }

        boolean takeStreamerUpdateResult(int i) throws InterruptedIOException {
            return this.streamerUpdateResult.take(i);
        }

        void updateStreamer(StripedDataStreamer streamer, boolean success) {
            assert (!this.updateStreamerMap.containsKey((Object)streamer));
            this.updateStreamerMap.put(streamer, success);
        }

        void clearFailureStates() {
            this.newBlocks.clear();
            this.updateStreamerMap.clear();
            this.streamerUpdateResult.clear();
        }
    }

    static class MultipleBlockingQueue<T> {
        private final List<BlockingQueue<T>> queues;

        MultipleBlockingQueue(int numQueue, int queueSize) {
            this.queues = new ArrayList<BlockingQueue<T>>(numQueue);
            for (int i = 0; i < numQueue; ++i) {
                this.queues.add(new LinkedBlockingQueue(queueSize));
            }
        }

        void offer(int i, T object) {
            boolean b = this.queues.get(i).offer(object);
            Preconditions.checkState((boolean)b, (Object)("Failed to offer " + object + " to queue, i=" + i));
        }

        T take(int i) throws InterruptedIOException {
            try {
                return this.queues.get(i).take();
            }
            catch (InterruptedException ie) {
                throw DFSUtilClient.toInterruptedIOException("take interrupted, i=" + i, ie);
            }
        }

        T takeWithTimeout(int i) throws InterruptedIOException {
            try {
                return this.queues.get(i).poll(100L, TimeUnit.MILLISECONDS);
            }
            catch (InterruptedException e) {
                throw DFSUtilClient.toInterruptedIOException("take interrupted, i=" + i, e);
            }
        }

        T poll(int i) {
            return (T)this.queues.get(i).poll();
        }

        T peek(int i) {
            return (T)this.queues.get(i).peek();
        }

        void clear() {
            for (BlockingQueue<T> q : this.queues) {
                q.clear();
            }
        }
    }
}

