/*
 * Copyright (C) 2025  rob at applecommander.org
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package org.applecommander.disassembler.api;

import java.util.Arrays;
import java.util.Objects;

/**
 * Program is a state class that tracks where decoding is, what the address is, and provides
 * a few helper methods to assist with various data types. Note that "get" reads from an absolute
 * position (always from 0) while "peek" reads from a relative position (from offset).
 */
public class Program {
    static final int ADDRESS_SPACE = 64 * 1024; //64k Address Space

    private final int baseAddress;
    private final byte[] code;
    private int offset;

    public Program(byte[] code, int address) {
        Objects.requireNonNull(code);
        this.baseAddress = address;
        this.code = code;
        this.offset = 0;
    }

    /** Length of code. */
    public int length() {
        return code.length;
    }
    /** Test if there are more bytes to process. */
    public boolean hasMore() {
        return offset < code.length;
    }
    /** Get an unsigned byte at the current offset. Does not advance the offset value. */
    public int peekUnsignedByte() {
        return peekUnsignedByte(0);
    }
    /** Get an unsigned byte at the current offset + n. Does not advance the offset value. */
    public int peekUnsignedByte(int n) {
        return offset+n < code.length ? Byte.toUnsignedInt(code[offset+n]) : 0;
    }
    /** Get an unsigned short (2 bytes, little endian) at the current offset + n. Does not advance the offset value. */
    public int peekUnsignedShort(int n) {
        return peekUnsignedByte(n) | peekUnsignedByte(n+1) << 8;
    }
    /** Get a *signed* byte at the current offset + n. Does not advance the offset value. */
    public int peekSignedByte(int n) {
        return offset+n < code.length ? code[offset+n] : 0;
    }
    /** Read n bytes. Advances offset. Used to read all bytes that are part of an instruction. */
    public byte[] read(int n) {
        byte[] x = Arrays.copyOfRange(code, offset, offset+n);
        offset += n;
        return x;
    }
    /** Get an unsigned byte from specified offset. This is not relative like the others. */
    public int getUnsignedByte(int n) {
        return n < code.length ? Byte.toUnsignedInt(code[n]) : 0;
    }
    /** Get the current offset. */
    public int currentOffset() {
        return offset;
    }
    /** Get the base address. This is the initial address passed into the constructor. */
    public int baseAddress() {
        return baseAddress;
    }
    /** Get the current address. Note that the address is wrapped across the address space. Currently, 64K. */
    public int currentAddress() {
        return (baseAddress+offset) % ADDRESS_SPACE; //wrap around to 0 if address exceeds the address space
    }
}
