/*
 * Copyright (C) 2025  rob at applecommander.org
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package org.applecommander.disassembler.api.switching6502;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.function.Function;

import org.applecommander.disassembler.api.Instruction;
import org.applecommander.disassembler.api.InstructionSet;
import org.applecommander.disassembler.api.Program;
import org.applecommander.disassembler.api.mos6502.InstructionSet6502;
import org.applecommander.disassembler.api.sweet16.InstructionSetSWEET16;

public class InstructionSet6502Switching implements InstructionSet {
    public static InstructionSet withSwitching() {
        return new InstructionSet6502Switching(InstructionSet6502.for6502(), InstructionSetSWEET16.forSWEET16());
    }
    
    private final InstructionSet6502 mos6502;
    private final InstructionSetSWEET16 sweet16;
    private Function<Program,Instruction> strategy = this::decode6502;
    private final Queue<Instruction> pending = new LinkedList<>();
    
    private InstructionSet6502Switching(InstructionSet6502 mos6502, InstructionSetSWEET16 sweet16) {
        this.mos6502 = mos6502;
        this.sweet16 = sweet16;
    }

    @Override
    public Defaults defaults() {
        return Defaults.builder()
                .startAddress(0x300)
                .libraryLabels("All")
                .bytesPerInstruction(3)
                .get();
    }

    @Override
    public List<Instruction> decode(Program program) {
        List<Instruction> assembly = new ArrayList<>();
        while (program.hasMore()) {
            if (!pending.isEmpty()) {
                assembly.add(pending.remove());
            }
            assembly.add(strategy.apply(program));
        }
        return assembly;
    }

    @Override
    public List<OpcodeTable> opcodeTables() {
        throw new RuntimeException("Not implemented");
    }

    Instruction decode6502(Program program) {
        Instruction instruction = mos6502.decodeOne(program);
        int operandAddress = instruction.addressRef().flatMap(Instruction.Operand::address).orElse(0);
        if ("JSR".equals(instruction.mnemonic()) && operandAddress == 0xf689) {
            strategy = this::decodeSWEET16;
            pending.add(Instruction.at(program.currentAddress()).mnemonic(".SWEET16").get());
        }
        return instruction;
    }
    
    Instruction decodeSWEET16(Program program) {
        Instruction instruction = sweet16.decodeOne(program);
        if ("RTN".equals(instruction.mnemonic())) {
            strategy = this::decode6502;
            pending.add(Instruction.at(program.currentAddress()).mnemonic(".6502").get());
        }
        return instruction;
    }
}
