/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.tsgen.jackson.module;

import com.fasterxml.jackson.databind.type.SimpleType;
import java.lang.reflect.Field;
import java.util.List;
import org.appops.tsgen.jackson.module.grammar.AnyType;
import org.appops.tsgen.jackson.module.grammar.ArrayType;
import org.appops.tsgen.jackson.module.grammar.BooleanType;
import org.appops.tsgen.jackson.module.grammar.EnumType;
import org.appops.tsgen.jackson.module.grammar.Module;
import org.appops.tsgen.jackson.module.grammar.NumberType;
import org.appops.tsgen.jackson.module.grammar.StaticClassType;
import org.appops.tsgen.jackson.module.grammar.StringType;
import org.appops.tsgen.jackson.module.grammar.base.AbstractType;
import org.appops.tsgen.jackson.module.grammar.base.Value;
import org.appops.tsgen.jackson.module.visitors.TsJsonFormatVisitorWrapper;

/**
 * Exporter class to export and write static fields to typescript.
 */
public class StaticFieldExporter {
  private static final String CLASS_NAME_EXTENSION = "Static";

  private final Module module;
  private final TsJsonFormatVisitorWrapper tsJsonFormatVisitorWrapper;

  /**
   * Object constructor.
   * 
   * @param module Module to be exported.
   * @param conf Exporting configuration.
   */
  public StaticFieldExporter(Module module, Configuration conf) {
    this.module = module;
    if (conf == null) {
      conf = new Configuration();
    }
    tsJsonFormatVisitorWrapper = new TsJsonFormatVisitorWrapper(module, conf);
  }

  /**
   * Method exports java classes passed into typescript equivalents.
   * 
   * @param classesToConvert List of java classes to be converted.
   */
  public void export(List<Class<?>> classesToConvert) throws IllegalArgumentException {
    for (Class<?> clazz : classesToConvert) {
      if (clazz.isEnum()) {
        continue;
      }
      StaticClassType staticClass =
          new StaticClassType(clazz.getSimpleName() + CLASS_NAME_EXTENSION);

      Field[] declaredFields = clazz.getDeclaredFields();
      for (Field field : declaredFields) {
        if (isPublicStaticFinal(field.getModifiers())) {
          Value value;
          try {
            value = constructValue(module, field.getType(), field.get(null));
          } catch (IllegalAccessException e) {
            throw new RuntimeException("Failed to get value of field " + field, e);
          }
          if (value != null) {
            staticClass.getStaticFields().put(field.getName(), value);
          }
        }
      }
      if (staticClass.getStaticFields().size() > 0) {
        module.getNamedTypes().put(staticClass.getName(), staticClass);
      }
    }
  }

  /**
   * Checks method modifiers.
   * 
   * @param modifiers Method modifiers index.
   * @return Boolean result, true if modifiers contains static and final modifier.
   */
  private boolean isPublicStaticFinal(final int modifiers) {
    return java.lang.reflect.Modifier.isPublic(modifiers)
        && java.lang.reflect.Modifier.isStatic(modifiers)
        && java.lang.reflect.Modifier.isFinal(modifiers);
  }

  /**
   * Constructs value object from raw value fetched from java type.
   * 
   * @param module Module to be exported
   * @param type Type under conversion.
   * @param rawValue Raw value which is to be converted into jackson based value equivalent.
   * @return Jackson based type safe value equivalent of raw value.
   */
  private Value constructValue(Module module, Class<?> type, Object rawValue)
      throws IllegalArgumentException, IllegalAccessException {
    if (type == boolean.class) {
      return new Value(BooleanType.getInstance(), rawValue);
    } else if (type == int.class) {
      return new Value(NumberType.getInstance(), rawValue);
    } else if (type == double.class) {
      return new Value(NumberType.getInstance(), rawValue);
    } else if (type == String.class) {
      return new Value(StringType.getInstance(), "'" + (String) rawValue + "'");
    } else if (type.isEnum()) {
      final EnumType enumType =
          tsJsonFormatVisitorWrapper.parseEnumOrGetFromCache(module, SimpleType.construct(type));
      return new Value(enumType, enumType.getName() + "." + rawValue);
    } else if (type.isArray()) {
      final Class<?> componentType = type.getComponentType();
      final Object[] array;
      if (componentType == boolean.class) {
        boolean[] tmpArray = (boolean[]) rawValue;
        array = new Boolean[tmpArray.length];
        for (int i = 0; i < array.length; i++) {
          array[i] = Boolean.valueOf(tmpArray[i]);
        }
      } else if (componentType == int.class) {
        int[] tmpArray = (int[]) rawValue;
        array = new Integer[tmpArray.length];
        for (int i = 0; i < array.length; i++) {
          array[i] = Integer.valueOf(tmpArray[i]);
        }
      } else if (componentType == double.class) {
        double[] tmpArray = (double[]) rawValue;
        array = new Double[tmpArray.length];
        for (int i = 0; i < array.length; i++) {
          array[i] = Double.valueOf(tmpArray[i]);
        }
      } else {
        array = (Object[]) rawValue;
      }
      final StringBuilder arrayValues = new StringBuilder();
      arrayValues.append("[ ");
      for (int i = 0; i < array.length; i++) {
        arrayValues.append(constructValue(module, componentType, array[i]).getValue());
        if (i < array.length - 1) {
          arrayValues.append(", ");
        }
      }
      arrayValues.append(" ]");
      return new Value(new ArrayType(typeScriptTypeFromJavaType(module, componentType)),
          arrayValues.toString());
    }
    return null;
  }

  /**
   * Converts java type into typescript equivalent.
   * 
   * @param module Module to be exported.
   * @param type Jave type which is to be converted.
   * @return Typescript equivalent of java type passed.
   */
  private AbstractType typeScriptTypeFromJavaType(Module module, Class<?> type) {
    if (type == boolean.class) {
      return BooleanType.getInstance();
    } else if (type == int.class) {
      return NumberType.getInstance();
    } else if (type == double.class) {
      return NumberType.getInstance();
    } else if (type == String.class) {
      return StringType.getInstance();
    } else if (type.isEnum()) {
      return tsJsonFormatVisitorWrapper.parseEnumOrGetFromCache(module, SimpleType.construct(type));
    } else if (type.isArray()) {
      return new ArrayType(AnyType.getInstance());
    }
    throw new UnsupportedOperationException();
  }
}
