/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.tsgen.jackson.module.visitors;

import static com.fasterxml.jackson.databind.PropertyName.NO_NAME;
import static java.lang.reflect.Modifier.isPublic;
//
import com.fasterxml.jackson.databind.BeanProperty;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.PropertyMetadata;
import com.fasterxml.jackson.databind.PropertyName;
import com.fasterxml.jackson.databind.introspect.AnnotatedMember;
import com.fasterxml.jackson.databind.introspect.AnnotatedMethod;
import com.fasterxml.jackson.databind.introspect.AnnotatedParameter;
import com.fasterxml.jackson.databind.introspect.AnnotationMap;
import com.fasterxml.jackson.databind.introspect.TypeResolutionContext;
import com.fasterxml.jackson.databind.jsonFormatVisitors.JsonFormatVisitable;
import com.fasterxml.jackson.databind.jsonFormatVisitors.JsonObjectFormatVisitor;
import com.fasterxml.jackson.databind.ser.BeanPropertyWriter;
import com.fasterxml.jackson.databind.type.TypeFactory;
import java.beans.BeanInfo;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import org.appops.tsgen.jackson.module.Configuration;
import org.appops.tsgen.jackson.module.grammar.AnyType;
import org.appops.tsgen.jackson.module.grammar.ClassType;
import org.appops.tsgen.jackson.module.grammar.FunctionType;
import org.appops.tsgen.jackson.module.grammar.VoidType;
import org.appops.tsgen.jackson.module.grammar.base.AbstractType;

public class TsJsonObjectFormatVisitor extends BaseTsJsonFormatVisitor<ClassType>
    implements JsonObjectFormatVisitor {

  private Class clazz;

  /**
   * Object constructor.
   * 
   * @param parentHolder Parent json format visitor.
   * @param className Java type name.
   * @param clazz Java type.
   * @param conf Typescript conversion configuration.
   */
  public TsJsonObjectFormatVisitor(BaseTsJsonFormatVisitor<?> parentHolder, String className,
      Class clazz, Configuration conf) {
    super(parentHolder, conf);
    type = new ClassType(className);
    this.clazz = clazz;
  }

  private void addField(String name, AbstractType fieldType) {
    type.getFields().put(name, fieldType);
  }

  /**
   * Checks if method passed is accessor method or not.
   * 
   * @param method Method to be checked.
   * @param beanInfo Type information.
   * 
   * @return true if method passed is abstract method, false otherwise.
   */
  private boolean isAccessorMethod(Method method, BeanInfo beanInfo) {
    for (PropertyDescriptor property : beanInfo.getPropertyDescriptors()) {
      if (method.equals(property.getReadMethod())) {
        return true;
      }
      if (method.equals(property.getWriteMethod())) {
        return true;
      }
    }
    return false;
  }

  /**
   * Adds public methods to output typescripts by scanning java class provided.
   */
  void addPublicMethods() {

    for (Method method : this.clazz.getDeclaredMethods()) {

      // Only public
      if (!isPublic(method.getModifiers())) {
        continue;
      }

      // Exclude accessors
      try {
        BeanInfo beanInfo = Introspector.getBeanInfo(clazz);
        if (isAccessorMethod(method, beanInfo)) {
          continue;
        }
      } catch (Exception e) {
        throw new RuntimeException(e);
      }

      if (conf.isIgnoredMethod(method)) {
        continue;
      }

      addMethod(method);
    }
  }

  /**
   * Prepares and provides typescript for type passed.
   * 
   * @param member Annotated member.
   * @return Type script variant.
   */
  private AbstractType getTsTypeForClass(AnnotatedMember member) {
    BeanProperty prop = new BeanProperty.Std(new PropertyName(member.getName()), member.getType(),
        NO_NAME, member, PropertyMetadata.STD_OPTIONAL);

    try {
      return getTsTypeForProperty(prop);
    } catch (JsonMappingException e) {
      throw new RuntimeException(e);
    }
  }

  /**
   * Adds method to typescript output.
   * 
   * @param method Method information to be added.
   */
  private void addMethod(Method method) {
    FunctionType function = new FunctionType();
    JavaType type = TypeFactory.defaultInstance().constructType(clazz);
    TypeResolutionContext typeResolutionContext =
        new TypeResolutionContext.Basic(TypeFactory.defaultInstance(), type.getBindings());
    AnnotatedMethod annotMethod =
        new AnnotatedMethod(typeResolutionContext, method, new AnnotationMap(), null);

    Parameter[] parameters = method.getParameters();

    function.setResultType(getTsTypeForClass(annotMethod));
    for (int i = 0; i < annotMethod.getParameterCount(); i++) {
      AnnotatedParameter param = annotMethod.getParameter(i);
      String name = parameters[i].getName();
      function.getParameters().put(name, getTsTypeForClass(param));
    }
    this.type.getMethods().put(method.getName(), function);
  }

  public void property(BeanProperty writer) throws JsonMappingException {
    addField(writer.getName(), getTsTypeForProperty(writer));
  }

  public void property(String name, JsonFormatVisitable handler, JavaType propertyTypeHint)
      throws JsonMappingException {
    addField(name,
        TsJsonFormatVisitorWrapper.getTsTypeForHandler(this, handler, propertyTypeHint, conf));
  }


  public void optionalProperty(BeanProperty writer) throws JsonMappingException {
    addField(writer.getName(), getTsTypeForProperty(writer));
  }

  public void optionalProperty(String name, JsonFormatVisitable handler, JavaType propertyTypeHint)
      throws JsonMappingException {
    addField(name,
        TsJsonFormatVisitorWrapper.getTsTypeForHandler(this, handler, propertyTypeHint, conf));
  }

  /**
   * Fetches typescript type equivalent for property passed.
   * 
   * @param writer Java property.
   * @return type equivalent for java property.
   * @throws JsonMappingException if occurred while fetching typescript type.
   */
  protected AbstractType getTsTypeForProperty(BeanProperty writer) throws JsonMappingException {
    if (writer == null) {
      throw new IllegalArgumentException("Null writer");
    }
    JavaType type = writer.getType();
    if (type.getRawClass().equals(Void.TYPE)) {
      return VoidType.getInstance();
    }

    AbstractType customType = conf.getCustomTypes().get(type.getRawClass().getName());
    if (customType != null) {
      return customType;
    }

    try {
      JsonSerializer<Object> ser = getSer(writer);

      if (ser != null) {
        if (type == null) {
          throw new IllegalStateException("Missing type for property '" + writer.getName() + "'");
        }
        return TsJsonFormatVisitorWrapper.getTsTypeForHandler(this, ser, type, conf);
      } else {
        return AnyType.getInstance();
      }

    } catch (Exception e) {
      throw new RuntimeException(String.format(//
          "Error when serializing %s, you should add a custom mapping for it", type.getRawClass()),
          e);
    }

  }

  /**
   * Fetches type serializer for java bean property.
   * 
   * @param writer Java property.
   * @return Json based type script serializer for property passed.
   * @throws JsonMappingException if occurred while fetching serializer.
   */
  protected JsonSerializer<java.lang.Object> getSer(BeanProperty writer)
      throws JsonMappingException {
    JsonSerializer<Object> ser = null;
    if (writer instanceof BeanPropertyWriter) {
      ser = ((BeanPropertyWriter) writer).getSerializer();
    }
    if (ser == null) {
      ser = getProvider().findValueSerializer(writer.getType(), writer);
    }
    return ser;
  }

}
