/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.tsgen.jackson.module.writer;

import java.io.IOException;
import java.io.Writer;
import org.appops.tsgen.jackson.module.grammar.EnumType;
import org.appops.tsgen.jackson.module.grammar.base.AbstractNamedType;

/**
 * Alternative to writing Java enum type to TypeScript enum type. Usefult, if You have following
 * goals:<br>
 * 1) JavaScript object containing field of "enum" must contain enum name instead of ordinal value.
 * This is good for several reasons: a) easy to understand JSON content b) doesn't mess things up,
 * if enum order changes in backend.<br>
 * 2) You'd still like to write TypeScript as if You were using real enums - type safety, etc.<br>
 * 3) You need to use "reflection" (instanceof) to detect if field is "enum"
 * 
 * @author Ats Uiboupin
 */
public class EnumTypeToEnumPatternWriter implements CustomAbstractTypeWriter {

  public boolean accepts(AbstractNamedType type, WriterPreferences preferences) {
    return type instanceof EnumType;
  }

  /**
   * Writes type definition.
   * 
   * @param type Type to be added into typescript.
   * @param writer Typescript file writer.
   * @param preferences Writing preperences and settings.
   */
  public void writeDef(AbstractNamedType type, Writer writer, WriterPreferences preferences)
      throws IOException {
    EnumType enumType = (EnumType) type;
    String enumTypeName = enumType.getName();
    writer.write(String.format("class %s extends EnumPatternBase {\n", enumTypeName));
    preferences.increaseIndentation();
    for (String value : enumType.getValues()) {
      writer.write(String.format(preferences.getIndentation() + "static %s = new %s('%s');\n",
          value, enumTypeName, value));
    }
    writer.write(preferences.getIndentation() + "constructor(name:string){super(name)}\n");
    preferences.decreaseIndention();
    writer.write(preferences.getIndentation() + "}");
  }

}
