/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.tsgen.jackson.module.writer;

import java.io.IOException;
import java.io.Writer;
import java.util.Collection;
import java.util.Map.Entry;
import org.appops.tsgen.jackson.module.grammar.EnumType;
import org.appops.tsgen.jackson.module.grammar.Module;
import org.appops.tsgen.jackson.module.grammar.base.AbstractNamedType;
import org.appops.tsgen.jackson.module.grammar.base.AbstractType;

/**
 * Generates TypeScript type definitions for given module in external module format.
 */
public class ExternalModuleFormatWriter implements ModuleWriter {

  public WriterPreferences preferences = new WriterPreferences();

  public void write(Module module, Writer writer) throws IOException {
    writeModuleContent(module, writer);
    writer.flush();
  }

  protected void writeModuleContent(Module module, Writer writer) throws IOException {
    Collection<AbstractNamedType> namedTypes = module.getNamedTypes().values();

    writeEnumPatternBaseClassIfNeeded(namedTypes, writer);

    for (AbstractNamedType type : namedTypes) {
      writer.write(preferences.getIndentation() + "export ");
      type.writeDef(writer, preferences);
      writer.write("\n\n");
    }

    for (Entry<String, AbstractType> entry : module.getVars().entrySet()) {
      writer.write(preferences.getIndentation() + "export var " + entry.getKey() + ": ");
      entry.getValue().write(writer);
      writer.write(";\n");
    }
  }

  private void writeEnumPatternBaseClassIfNeeded(Collection<AbstractNamedType> namedTypes,
      Writer writer) throws IOException {
    if (preferences.isUseEnumPattern() && hasEnum(namedTypes)) {
      writeBaseEnum(writer);
      writer.write("\n");
    }
  }

  private boolean hasEnum(Collection<AbstractNamedType> namedTypes) {
    for (AbstractNamedType type : namedTypes) {
      if (type instanceof EnumType) {
        return true;
      }
    }
    return false;
  }

  private void writeBaseEnum(Writer writer) throws IOException {
    writer.write(preferences.getIndentation()
        + "/** base class for implementing enums with Typesafe Enum Pattern "
        + "(to be able to use enum names, instead of ordinal values, in a type-safe manner) */\n");
    writer.write(preferences.getIndentation() + "export class EnumPatternBase {\n");
    preferences.increaseIndentation();
    writer.write(preferences.getIndentation() + "constructor(public name: string){}\n");
    writer.write(preferences.getIndentation() + "toString(){ return this.name; }\n");
    preferences.decreaseIndention();
    writer.write(preferences.getIndentation() + "}\n");
  }

}
