/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.tsgen.jackson.module;

import java.beans.Transient;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.appops.tsgen.jackson.module.conf.typename.SimpleJacksonTsTypeNamingStrategy;
import org.appops.tsgen.jackson.module.conf.typename.TsTypeNamingStrategy;
import org.appops.tsgen.jackson.module.grammar.ArrayType;
import org.appops.tsgen.jackson.module.grammar.base.AbstractType;

/**
 * Configuration required for conversion of java details into typescript equivalents.
 */
public class Configuration {
  private Map<String, AbstractType> customTypes = Collections.emptyMap();
  private List<String> ignoredMethodNames = new ArrayList<String>();
  private TsTypeNamingStrategy namingStrategy = new SimpleJacksonTsTypeNamingStrategy();

  public Map<String, AbstractType> getCustomTypes() {
    return customTypes;
  }

  /**
   * Adds type into configuration.
   * 
   * @param klass Class type to be added.
   * @param tsType Type script type to be added.
   * @return Updated configuration.
   */
  public Configuration addType(Class<?> klass, AbstractType tsType) {
    addType(klass.getName(), tsType);
    addArrayType(klass, tsType);
    return this;
  }

  /**
   * Adds type.
   * 
   * @param className Name of type to be added.
   * @param tsType Type script type to be added.
   */
  public void addType(String className, AbstractType tsType) {
    Map<String, AbstractType> tmp = new HashMap<String, AbstractType>();
    tmp.putAll(customTypes);
    tmp.put(className, tsType);
    customTypes = Collections.unmodifiableMap(tmp);
  }

  public void addArrayType(Class<?> klass, AbstractType tsType) {
    addType("[L" + klass.getName() + ";", new ArrayType(tsType));
  }


  public void addIngoredMethod(String name) {
    ignoredMethodNames.add(name);
  }

  /**
   * Checks if method is ignored or not.
   * 
   * @param method Method to be checked.
   * @return true if method is ignored, false otherwise.
   */
  public boolean isIgnoredMethod(Method method) {
    if (method.getAnnotation(Transient.class) != null) {
      return true;
    }
    return isIgnoredMethod(method.getName());
  }

  private boolean isIgnoredMethod(String name) {
    return ignoredMethodNames.contains(name);
  }

  public TsTypeNamingStrategy getNamingStrategy() {
    return namingStrategy;
  }

  public void setNamingStrategy(TsTypeNamingStrategy namingStrategy) {
    this.namingStrategy = namingStrategy;
  }
}
