/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.tsgen.jackson.module.visitors;

import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.jsonFormatVisitors.JsonAnyFormatVisitor;
import com.fasterxml.jackson.databind.jsonFormatVisitors.JsonArrayFormatVisitor;
import com.fasterxml.jackson.databind.jsonFormatVisitors.JsonBooleanFormatVisitor;
import com.fasterxml.jackson.databind.jsonFormatVisitors.JsonFormatVisitable;
import com.fasterxml.jackson.databind.jsonFormatVisitors.JsonFormatVisitorWrapper;
import com.fasterxml.jackson.databind.jsonFormatVisitors.JsonIntegerFormatVisitor;
import com.fasterxml.jackson.databind.jsonFormatVisitors.JsonMapFormatVisitor;
import com.fasterxml.jackson.databind.jsonFormatVisitors.JsonNullFormatVisitor;
import com.fasterxml.jackson.databind.jsonFormatVisitors.JsonNumberFormatVisitor;
import com.fasterxml.jackson.databind.jsonFormatVisitors.JsonObjectFormatVisitor;
import com.fasterxml.jackson.databind.jsonFormatVisitors.JsonStringFormatVisitor;
import org.appops.tsgen.jackson.module.Configuration;
import org.appops.tsgen.jackson.module.grammar.EnumType;
import org.appops.tsgen.jackson.module.grammar.Module;
import org.appops.tsgen.jackson.module.grammar.base.AbstractNamedType;
import org.appops.tsgen.jackson.module.grammar.base.AbstractType;

public class TsJsonFormatVisitorWrapper extends BaseTsJsonFormatVisitor
    implements JsonFormatVisitorWrapper {

  public TsJsonFormatVisitorWrapper(BaseTsJsonFormatVisitor parentHolder, Configuration conf) {
    super(parentHolder, conf);
  }

  public TsJsonFormatVisitorWrapper(Module module, Configuration conf) {
    super(module, conf);
  }

  private <T extends BaseTsJsonFormatVisitor<?>> T setTypeAndReturn(T actualVisitor) {
    type = actualVisitor.getType();
    return actualVisitor;
  }

  /**
   * Visit recursively the type, or return a cached response.
   * 
   * @param baseVisitor Base visitor instance.
   * @param handler Visitable json node handler.
   * @param typeHint java type meta information.
   * @param conf Jackson configuration.
   * @return Returns typescript equivalent for typeHint passed.
   * @throws JsonMappingException if json conversion for java type fails.
   */
  public static AbstractType getTsTypeForHandler(BaseTsJsonFormatVisitor<?> baseVisitor,
      JsonFormatVisitable handler, JavaType typeHint, Configuration conf)
      throws JsonMappingException {

    AbstractType computedType = baseVisitor.getComputedTypes().get(typeHint);

    if (computedType != null) {
      return computedType;
    }
    TsJsonFormatVisitorWrapper visitor = new TsJsonFormatVisitorWrapper(baseVisitor, conf);
    handler.acceptJsonFormatVisitor(visitor, typeHint);
    baseVisitor.getComputedTypes().put(typeHint, visitor.getType());
    return visitor.getType();
  }

  /**
   * Either Java simple name or @JsonTypeName annotation.
   * 
   * @param type Java type definition.
   * @return type name.
   */
  public String getName(JavaType type) {
    return conf.getNamingStrategy().getName(type);
  }

  private TsJsonObjectFormatVisitor useNamedClassOrParse(JavaType javaType) {

    String name = getName(javaType);

    AbstractNamedType namedType = getModule().getNamedTypes().get(name);

    if (namedType == null) {
      TsJsonObjectFormatVisitor visitor =
          new TsJsonObjectFormatVisitor(this, name, javaType.getRawClass(), conf);
      type = visitor.getType();
      getModule().getNamedTypes().put(visitor.getType().getName(), visitor.getType());
      visitor.addPublicMethods();
      return visitor;
    } else {
      type = namedType;
      return null;
    }
  }

  /**
   * Parser method for java enum type.
   * 
   * @param module Typescript module representation.
   * @param javaType Java type.
   * @return Equivalent enum type for typescript.
   */
  public EnumType parseEnumOrGetFromCache(Module module, JavaType javaType) {
    String name = getName(javaType);
    AbstractType namedType = module.getNamedTypes().get(name);
    if (namedType == null) {
      EnumType enumType = new EnumType(name);
      for (Object val : javaType.getRawClass().getEnumConstants()) {
        enumType.getValues().add(val.toString());
      }
      module.getNamedTypes().put(name, enumType);
      return enumType;
    } else {
      return (EnumType) namedType;
    }
  }

  public JsonObjectFormatVisitor expectObjectFormat(JavaType type) throws JsonMappingException {
    return useNamedClassOrParse(type);
  }

  public JsonArrayFormatVisitor expectArrayFormat(JavaType type) throws JsonMappingException {
    return setTypeAndReturn(new TsJsonArrayFormatVisitor(this, conf));
  }

  /**
   * Sets string format expectations to json based visitor.
   * 
   * @param jtype Java type to be parsed.
   * @return Json based string format visitor.
   */
  public JsonStringFormatVisitor expectStringFormat(JavaType jtype) throws JsonMappingException {
    if (jtype.getRawClass().isEnum()) {
      type = parseEnumOrGetFromCache(getModule(), jtype);
      return null;
    } else {
      return setTypeAndReturn(new TsJsonStringFormatVisitor(this, conf));
    }
  }

  public JsonNumberFormatVisitor expectNumberFormat(JavaType type) throws JsonMappingException {
    return setTypeAndReturn(new TsJsonNumberFormatVisitor(this, conf));
  }

  public JsonIntegerFormatVisitor expectIntegerFormat(JavaType type) throws JsonMappingException {
    return setTypeAndReturn(new TsJsonNumberFormatVisitor(this, conf));
  }

  public JsonBooleanFormatVisitor expectBooleanFormat(JavaType type) throws JsonMappingException {
    return setTypeAndReturn(new TsJsonBooleanFormatVisitor(this, conf));
  }

  public JsonNullFormatVisitor expectNullFormat(JavaType type) throws JsonMappingException {
    return setTypeAndReturn(new TsJsonNullFormatVisitor(this, conf));
  }

  /**
   * Sets string format expectations to json based visitor.
   * 
   * @param type Java type to be parsed.
   * @return Json based any format visitor.
   */
  public JsonAnyFormatVisitor expectAnyFormat(JavaType type) throws JsonMappingException {
    if (Object.class.getCanonicalName().equals(type.getRawClass().getName())) {
      return setTypeAndReturn(new TsJsonAnyFormatVisitor(this, conf));
    }
    // probably just a class without fields/properties
    useNamedClassOrParse(type);
    return null;
  }

  public JsonMapFormatVisitor expectMapFormat(JavaType type) throws JsonMappingException {
    return setTypeAndReturn(new TsJsonMapFormatVisitor(this, conf));
  }

}
