/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.core;

import com.google.inject.Inject;
import com.google.inject.Module;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.appops.core.annotation.ImplModule;
import org.appops.core.annotation.SlimModule;
import org.reflections.Reflections;
import org.reflections.scanners.FieldAnnotationsScanner;
import org.reflections.scanners.MethodAnnotationsScanner;
import org.reflections.scanners.MethodParameterNamesScanner;
import org.reflections.scanners.MethodParameterScanner;
import org.reflections.scanners.SubTypesScanner;
import org.reflections.scanners.TypeAnnotationsScanner;
import org.reflections.scanners.TypeElementsScanner;
import org.reflections.util.FilterBuilder;

/**
 * Performs java reflection based classpath analysis. It provides functionalities such as
 * find-classes-from-annotation, find-class-methods etc.
 *
 * @author deba
 * @version $Id: $Id
 */
public class ClassPathAnalyser {

  private TypeScanner typeScanner;
  private Reflections reflections;
  private final List<String> defaultPackages = Arrays.asList("org.appops", "com.ainosoft");

  /**
   * Object constructor which also initializes java reflections instance.
   */
  public ClassPathAnalyser() {
    this(null);
  }


  /**
   * Object constructor which also initializes java reflections instance.
   *
   * @param loader Custom class loader.
   * @param packages List of packages to be scanned.
   */
  public ClassPathAnalyser(ClassLoader loader, String... packages) {
    Set<Object> scanners = new HashSet<Object>(
        Arrays.asList(new TypeAnnotationsScanner(), new FieldAnnotationsScanner(),
            new MethodParameterScanner(), new MethodParameterNamesScanner(),
            new TypeElementsScanner(), new SubTypesScanner(), new MethodAnnotationsScanner()));
    if (loader != null) {
      scanners.add(loader);
    }
    reflections = createReflections(scanners, Arrays.asList(packages));
  }

  /**
   * Instantiates {@link Reflections} using scanners and package provided.
   * 
   * @param scanners Classpath scanners to be added/enabled for scanning.
   * @param packages List of packages to be scanned.
   * @return Customized {@link Reflections} instance.
   * 
   */
  private Reflections createReflections(Set<Object> scanners, List<String> packages) {
    packages = (packages == null || packages.size() <= 0) ? defaultPackages : packages;
    FilterBuilder filters = new FilterBuilder();
    for (String packagePrefix : packages) {
      filters.include(FilterBuilder.prefix(packagePrefix));
    }
    return new Reflections(scanners, filters);
  }

  /**
   * <p>
   * getAnnotatedTypes.
   * </p>
   *
   * @param annotation a {@link java.lang.Class} object.
   * @return a {@link java.util.Set} object.
   */
  public Set<Class<?>> getAnnotatedTypes(Class<? extends Annotation> annotation) {
    return reflections.getTypesAnnotatedWith(annotation);
  }

  /**
   * Find classes annotated with {@link SlimModule} and create map with service name as key and
   * {@link Set} of module classes as value.
   *
   * @return a {@link java.util.Map} object.
   */
  public Map<String, Set<Class<? extends Module>>> getSlimModuleAnnotatedTypeMap() {
    Set<Class<?>> configClasses = getAnnotatedTypes(SlimModule.class);
    Map<String, Set<Class<? extends Module>>> map =
        new HashMap<String, Set<Class<? extends Module>>>();

    for (Class<?> annotatedClass : configClasses) {
      if (annotatedClass.getAnnotation(SlimModule.class) != null) {
        String ServiceName =
            annotatedClass.getAnnotation(SlimModule.class).serviceName().getCanonicalName();
        if (map.get(ServiceName) == null) {
          HashSet<Class<? extends Module>> list = new HashSet<Class<? extends Module>>();
          list.add((Class<? extends Module>) annotatedClass);
          map.put(ServiceName, list);
        } else {
          map.get(ServiceName).add((Class<? extends Module>) annotatedClass);
        }
      }
    }
    return map;
  }

  /**
   * Find classes annotated with {@link ImplModule} and create map with service name as key and
   * {@link Set} of module classes as value.
   *
   *
   * @return a {@link java.util.Map} object.
   */
  public Map<String, Set<Class<? extends Module>>> getImplModuleAnnotatedTypeMap() {
    Set<Class<?>> configClasses = getAnnotatedTypes(ImplModule.class);
    Map<String, Set<Class<? extends Module>>> map =
        new HashMap<String, Set<Class<? extends Module>>>();

    for (Class<?> annotatedClass : configClasses) {
      if (annotatedClass.getAnnotation(ImplModule.class) != null) {
        String ServiceName =
            annotatedClass.getAnnotation(ImplModule.class).serviceName().getCanonicalName();
        if (map.get(ServiceName) == null) {
          HashSet<Class<? extends Module>> list = new HashSet<Class<? extends Module>>();
          list.add((Class<? extends Module>) annotatedClass);
          map.put(ServiceName, list);
        } else {
          map.get(ServiceName).add((Class<? extends Module>) annotatedClass);
        }
      }
    }
    return map;
  }

  /**
   * Fetches list of methods which has annotation provided.
   *
   * @param clazz Class of which method definiftions to be scanned for.
   * @param methodAnnotation Annotation to be checked for.
   * @return list of annotated methods.
   */
  public List<Method> getAnnotatedMethods(Class<?> clazz,
      Class<? extends Annotation> methodAnnotation) {
    return getTypeScanner().getAnnotatedMethods(clazz, methodAnnotation);
  }

  /**
   * <p>
   * Getter for the field <code>typeScanner</code>.
   * </p>
   *
   * @return a {@link org.appops.core.TypeScanner} object.
   */
  public TypeScanner getTypeScanner() {
    return typeScanner;
  }

  /**
   * <p>
   * Setter for the field <code>typeScanner</code>.
   * </p>
   *
   * @param typeScanner a {@link org.appops.core.TypeScanner} object.
   */
  @Inject
  public void setTypeScanner(TypeScanner typeScanner) {
    this.typeScanner = typeScanner;
  }


  /**
   * <p>
   * subTypesOf.
   * </p>
   *
   * @param superType a {@link java.lang.Class} object.
   * @param <T> a T object.
   * @return a {@link java.util.Collection} object.
   */
  public <T> Collection<Class<? extends T>> subTypesOf(Class<T> superType) {
    return reflections.getSubTypesOf(superType);
  }

}
