/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.core.store;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.LoadingCache;
import java.io.Serializable;
import java.util.concurrent.TimeUnit;
import org.appops.core.exception.CacheException;
import org.appops.core.loader.AppopsCacheLoader;

/**
 * Store implementation which supports caching of data.
 *
 * @author deba
 * @version $Id: $Id
 */
public abstract class CachedStore<K extends Serializable, V extends Serializable>
    implements Store<K, V> {

  private LoadingCache<K, V> cache;
  private AppopsCacheLoader<K, V> cacheLoader;

  /**
   * <p>
   * Constructor for CachedStore.
   * </p>
   *
   * @param <L> a reference
   * @param cacheLoader a L object.
   * @param maxCacheSize a long.
   * @param expiryTimeInSeconds a long.
   */
  protected <L extends AppopsCacheLoader<K, V>> CachedStore(L cacheLoader, long maxCacheSize,
      long expiryTimeInSeconds) {
    this(cacheLoader, maxCacheSize, expiryTimeInSeconds, TimeUnit.SECONDS);
  }

  /**
   * <p>
   * Constructor for CachedStore.
   * </p>
   *
   * @param <L> a reference
   * @param cacheLoader a L object.
   * @param maxCacheSize a long.
   * @param expiryTime a long.
   * @param expiryTimeUnit a {@link java.util.concurrent.TimeUnit} object.
   */
  protected <L extends AppopsCacheLoader<K, V>> CachedStore(L cacheLoader, long maxCacheSize,
      long expiryTime, TimeUnit expiryTimeUnit) {
    setCacheLoader(cacheLoader);
    createCache(maxCacheSize, expiryTime, expiryTimeUnit);
  }


  /**
   * Creates guava cache instance from configuration provided.
   *
   * @param maxCacheSize Maximum number of elements to be stored in cache.
   * @param expiryTime Time after which data should be automatically removed from cache.
   * @param expiryTimeUnit Unit of expiration time e.g. TimeUnit.SECONDS.
   * @return Cache instance.
   */
  private LoadingCache<K, V> createCache(long maxCacheSize, long expiryTime,
      TimeUnit expiryTimeUnit) {
    LoadingCache<K, V> cacheToReturn = CacheBuilder.newBuilder().maximumSize(maxCacheSize)
        .expireAfterAccess(expiryTime, expiryTimeUnit).build(getCacheLoader());
    setCache(cacheToReturn);
    return cacheToReturn;
  }


  /** {@inheritDoc} */
  @Override
  public V get(K key) {
    try {
      return getCache().get(key);
    } catch (Exception e) {
      throw new CacheException(e);
    }
  }

  private void setCache(LoadingCache<K, V> cache) {
    this.cache = cache;
  }



  /**
   * <p>
   * Getter for the field <code>cacheLoader</code>.
   * </p>
   *
   * @return a {@link org.appops.core.loader.AppopsCacheLoader} object.
   */
  public AppopsCacheLoader<K, V> getCacheLoader() {
    return cacheLoader;
  }



  private void setCacheLoader(AppopsCacheLoader<K, V> cacheLoader) {
    this.cacheLoader = cacheLoader;
  }



  /**
   * <p>
   * Getter for the field <code>cache</code>.
   * </p>
   *
   * @return a {@link com.google.common.cache.LoadingCache} object.
   */
  public LoadingCache<K, V> getCache() {
    return cache;
  }



}
